/* test_mlkem.c
 *
 * Copyright (C) 2006-2025 wolfSSL Inc.
 *
 * This file is part of wolfSSL.
 *
 * wolfSSL is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * wolfSSL is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1335, USA
 */

#include <tests/unit.h>

#ifdef NO_INLINE
    #include <wolfssl/wolfcrypt/misc.h>
#else
    #define WOLFSSL_MISC_INCLUDED
    #include <wolfcrypt/src/misc.c>
#endif

#ifdef WOLFSSL_HAVE_MLKEM
    #include <wolfssl/wolfcrypt/mlkem.h>
#ifdef WOLFSSL_WC_MLKEM
    #include <wolfssl/wolfcrypt/wc_mlkem.h>
#endif
#endif
#include <wolfssl/wolfcrypt/types.h>
#include <tests/api/api.h>
#include <tests/api/test_mlkem.h>

int test_wc_mlkem_make_key_kats(void)
{
    EXPECT_DECLS;
#if defined(WOLFSSL_HAVE_MLKEM) && defined(WOLFSSL_WC_MLKEM) && \
    !defined(WOLFSSL_NO_ML_KEM) && !defined(WOLFSSL_MLKEM_NO_MAKE_KEY)
    MlKemKey* key;
#ifndef WOLFSSL_NO_ML_KEM_512
    static const byte seed_512[WC_ML_KEM_MAKEKEY_RAND_SZ] = {
        /* d */
        0x2C, 0xB8, 0x43, 0xA0, 0x2E, 0xF0, 0x2E, 0xE1,
        0x09, 0x30, 0x5F, 0x39, 0x11, 0x9F, 0xAB, 0xF4,
        0x9A, 0xB9, 0x0A, 0x57, 0xFF, 0xEC, 0xB3, 0xA0,
        0xE7, 0x5E, 0x17, 0x94, 0x50, 0xF5, 0x27, 0x61,
        /* z */
        0x84, 0xCC, 0x91, 0x21, 0xAE, 0x56, 0xFB, 0xF3,
        0x9E, 0x67, 0xAD, 0xBD, 0x83, 0xAD, 0x2D, 0x3E,
        0x3B, 0xB8, 0x08, 0x43, 0x64, 0x52, 0x06, 0xBD,
        0xD9, 0xF2, 0xF6, 0x29, 0xE3, 0xCC, 0x49, 0xB7
    };
    static const byte ek_512[WC_ML_KEM_512_PUBLIC_KEY_SIZE] = {
        0xA3, 0x24, 0x39, 0xF8, 0x5A, 0x3C, 0x21, 0xD2,
        0x1A, 0x71, 0xB9, 0xB9, 0x2A, 0x9B, 0x64, 0xEA,
        0x0A, 0xB8, 0x43, 0x12, 0xC7, 0x70, 0x23, 0x69,
        0x4F, 0xD6, 0x4E, 0xAA, 0xB9, 0x07, 0xA4, 0x35,
        0x39, 0xDD, 0xB2, 0x7B, 0xA0, 0xA8, 0x53, 0xCC,
        0x90, 0x69, 0xEA, 0xC8, 0x50, 0x8C, 0x65, 0x3E,
        0x60, 0x0B, 0x2A, 0xC0, 0x18, 0x38, 0x1B, 0x4B,
        0xB4, 0xA8, 0x79, 0xAC, 0xDA, 0xD3, 0x42, 0xF9,
        0x11, 0x79, 0xCA, 0x82, 0x49, 0x52, 0x5C, 0xB1,
        0x96, 0x8B, 0xBE, 0x52, 0xF7, 0x55, 0xB7, 0xF5,
        0xB4, 0x3D, 0x66, 0x63, 0xD7, 0xA3, 0xBF, 0x0F,
        0x33, 0x57, 0xD8, 0xA2, 0x1D, 0x15, 0xB5, 0x2D,
        0xB3, 0x81, 0x8E, 0xCE, 0x5B, 0x40, 0x2A, 0x60,
        0xC9, 0x93, 0xE7, 0xCF, 0x43, 0x64, 0x87, 0xB8,
        0xD2, 0xAE, 0x91, 0xE6, 0xC5, 0xB8, 0x82, 0x75,
        0xE7, 0x58, 0x24, 0xB0, 0x00, 0x7E, 0xF3, 0x12,
        0x3C, 0x0A, 0xB5, 0x1B, 0x5C, 0xC6, 0x1B, 0x9B,
        0x22, 0x38, 0x0D, 0xE6, 0x6C, 0x5B, 0x20, 0xB0,
        0x60, 0xCB, 0xB9, 0x86, 0xF8, 0x12, 0x3D, 0x94,
        0x06, 0x00, 0x49, 0xCD, 0xF8, 0x03, 0x68, 0x73,
        0xA7, 0xBE, 0x10, 0x94, 0x44, 0xA0, 0xA1, 0xCD,
        0x87, 0xA4, 0x8C, 0xAE, 0x54, 0x19, 0x24, 0x84,
        0xAF, 0x84, 0x44, 0x29, 0xC1, 0xC5, 0x8C, 0x29,
        0xAC, 0x62, 0x4C, 0xD5, 0x04, 0xF1, 0xC4, 0x4F,
        0x1E, 0x13, 0x47, 0x82, 0x2B, 0x6F, 0x22, 0x13,
        0x23, 0x85, 0x9A, 0x7F, 0x6F, 0x75, 0x4B, 0xFE,
        0x71, 0x0B, 0xDA, 0x60, 0x27, 0x62, 0x40, 0xA4,
        0xFF, 0x2A, 0x53, 0x50, 0x70, 0x37, 0x86, 0xF5,
        0x67, 0x1F, 0x44, 0x9F, 0x20, 0xC2, 0xA9, 0x5A,
        0xE7, 0xC2, 0x90, 0x3A, 0x42, 0xCB, 0x3B, 0x30,
        0x3F, 0xF4, 0xC4, 0x27, 0xC0, 0x8B, 0x11, 0xB4,
        0xCD, 0x31, 0xC4, 0x18, 0xC6, 0xD1, 0x8D, 0x08,
        0x61, 0x87, 0x3B, 0xFA, 0x03, 0x32, 0xF1, 0x12,
        0x71, 0x55, 0x2E, 0xD7, 0xC0, 0x35, 0xF0, 0xE4,
        0xBC, 0x42, 0x8C, 0x43, 0x72, 0x0B, 0x39, 0xA6,
        0x51, 0x66, 0xBA, 0x9C, 0x2D, 0x3D, 0x77, 0x0E,
        0x13, 0x03, 0x60, 0xCC, 0x23, 0x84, 0xE8, 0x30,
        0x95, 0xB1, 0xA1, 0x59, 0x49, 0x55, 0x33, 0xF1,
        0x16, 0xC7, 0xB5, 0x58, 0xB6, 0x50, 0xDB, 0x04,
        0xD5, 0xA2, 0x6E, 0xAA, 0xA0, 0x8C, 0x3E, 0xE5,
        0x7D, 0xE4, 0x5A, 0x7F, 0x88, 0xC6, 0xA3, 0xCE,
        0xB2, 0x4D, 0xC5, 0x39, 0x7B, 0x88, 0xC3, 0xCE,
        0xF0, 0x03, 0x31, 0x9B, 0xB0, 0x23, 0x3F, 0xD6,
        0x92, 0xFD, 0xA1, 0x52, 0x44, 0x75, 0xB3, 0x51,
        0xF3, 0xC7, 0x82, 0x18, 0x2D, 0xEC, 0xF5, 0x90,
        0xB7, 0x72, 0x3B, 0xE4, 0x00, 0xBE, 0x14, 0x80,
        0x9C, 0x44, 0x32, 0x99, 0x63, 0xFC, 0x46, 0x95,
        0x92, 0x11, 0xD6, 0xA6, 0x23, 0x33, 0x95, 0x37,
        0x84, 0x8C, 0x25, 0x16, 0x69, 0x94, 0x1D, 0x90,
        0xB1, 0x30, 0x25, 0x8A, 0xDF, 0x55, 0xA7, 0x20,
        0xA7, 0x24, 0xE8, 0xB6, 0xA6, 0xCA, 0xE3, 0xC2,
        0x26, 0x4B, 0x16, 0x24, 0xCC, 0xBE, 0x7B, 0x45,
        0x6B, 0x30, 0xC8, 0xC7, 0x39, 0x32, 0x94, 0xCA,
        0x51, 0x80, 0xBC, 0x83, 0x7D, 0xD2, 0xE4, 0x5D,
        0xBD, 0x59, 0xB6, 0xE1, 0x7B, 0x24, 0xFE, 0x93,
        0x05, 0x2E, 0xB7, 0xC4, 0x3B, 0x27, 0xAC, 0x3D,
        0xC2, 0x49, 0xCA, 0x0C, 0xBC, 0xA4, 0xFB, 0x58,
        0x97, 0xC0, 0xB7, 0x44, 0x08, 0x8A, 0x8A, 0x07,
        0x79, 0xD3, 0x22, 0x33, 0x82, 0x6A, 0x01, 0xDD,
        0x64, 0x89, 0x95, 0x2A, 0x48, 0x25, 0xE5, 0x35,
        0x8A, 0x70, 0x0B, 0xE0, 0xE1, 0x79, 0xAC, 0x19,
        0x77, 0x10, 0xD8, 0x3E, 0xCC, 0x85, 0x3E, 0x52,
        0x69, 0x5E, 0x9B, 0xF8, 0x7B, 0xB1, 0xF6, 0xCB,
        0xD0, 0x5B, 0x02, 0xD4, 0xE6, 0x79, 0xE3, 0xB8,
        0x8D, 0xD4, 0x83, 0xB0, 0x74, 0x9B, 0x11, 0xBD,
        0x37, 0xB3, 0x83, 0xDC, 0xCA, 0x71, 0xF9, 0x09,
        0x18, 0x34, 0xA1, 0x69, 0x55, 0x02, 0xC4, 0xB9,
        0x5F, 0xC9, 0x11, 0x8C, 0x1C, 0xFC, 0x34, 0xC8,
        0x4C, 0x22, 0x65, 0xBB, 0xBC, 0x56, 0x3C, 0x28,
        0x26, 0x66, 0xB6, 0x0A, 0xE5, 0xC7, 0xF3, 0x85,
        0x1D, 0x25, 0xEC, 0xBB, 0x50, 0x21, 0xCC, 0x38,
        0xCB, 0x73, 0xEB, 0x6A, 0x34, 0x11, 0xB1, 0xC2,
        0x90, 0x46, 0xCA, 0x66, 0x54, 0x06, 0x67, 0xD1,
        0x36, 0x95, 0x44, 0x60, 0xC6, 0xFC, 0xBC, 0x4B,
        0xC7, 0xC0, 0x49, 0xBB, 0x04, 0x7F, 0xA6, 0x7A,
        0x63, 0xB3, 0xCC, 0x11, 0x11, 0xC1, 0xD8, 0xAC,
        0x27, 0xE8, 0x05, 0x8B, 0xCC, 0xA4, 0xA1, 0x54,
        0x55, 0x85, 0x8A, 0x58, 0x35, 0x8F, 0x7A, 0x61,
        0x02, 0x0B, 0xC9, 0xC4, 0xC1, 0x7F, 0x8B, 0x95,
        0xC2, 0x68, 0xCC, 0xB4, 0x04, 0xB9, 0xAA, 0xB4,
        0xA2, 0x72, 0xA2, 0x1A, 0x70, 0xDA, 0xF6, 0xB6,
        0xF1, 0x51, 0x21, 0xEE, 0x01, 0xC1, 0x56, 0xA3,
        0x54, 0xAA, 0x17, 0x08, 0x7E, 0x07, 0x70, 0x2E,
        0xAB, 0x38, 0xB3, 0x24, 0x1F, 0xDB, 0x55, 0x3F,
        0x65, 0x73, 0x39, 0xD5, 0xE2, 0x9D, 0xC5, 0xD9,
        0x1B, 0x7A, 0x5A, 0x82, 0x8E, 0xE9, 0x59, 0xFE,
        0xBB, 0x90, 0xB0, 0x72, 0x29, 0xF6, 0xE4, 0x9D,
        0x23, 0xC3, 0xA1, 0x90, 0x29, 0x70, 0x42, 0xFB,
        0x43, 0x98, 0x69, 0x55, 0xB6, 0x9C, 0x28, 0xE1,
        0x01, 0x6F, 0x77, 0xA5, 0x8B, 0x43, 0x15, 0x14,
        0xD2, 0x1B, 0x88, 0x88, 0x99, 0xC3, 0x60, 0x82,
        0x76, 0x08, 0x1B, 0x75, 0xF5, 0x68, 0x09, 0x7C,
        0xDC, 0x17, 0x48, 0xF3, 0x23, 0x07, 0x88, 0x58,
        0x15, 0xF3, 0xAE, 0xC9, 0x65, 0x18, 0x19, 0xAA,
        0x68, 0x73, 0xD1, 0xA4, 0xEB, 0x83, 0xB1, 0x95,
        0x38, 0x43, 0xB9, 0x34, 0x22, 0x51, 0x94, 0x83,
        0xFE, 0xF0, 0x05, 0x9D, 0x36, 0xBB, 0x2D, 0xB1,
        0xF3, 0xD4, 0x68, 0xFB, 0x06, 0x8C, 0x86, 0xE8,
        0x97, 0x37, 0x33, 0xC3, 0x98, 0xEA, 0xF0, 0x0E,
        0x17, 0x02, 0xC6, 0x73, 0x4A, 0xD8, 0xEB, 0x3B
    };
    static const byte dk_512[WC_ML_KEM_512_PRIVATE_KEY_SIZE] = {
        0x7F, 0xE4, 0x20, 0x6F, 0x26, 0xBE, 0xDB, 0x64,
        0xC1, 0xED, 0x00, 0x09, 0x61, 0x52, 0x45, 0xDC,
        0x98, 0x48, 0x3F, 0x66, 0x3A, 0xCC, 0x61, 0x7E,
        0x65, 0x89, 0x8D, 0x59, 0x6A, 0x88, 0x36, 0xC4,
        0x9F, 0xBD, 0x3B, 0x4A, 0x84, 0x97, 0x59, 0xAA,
        0x15, 0x46, 0xBD, 0xA8, 0x35, 0xCA, 0xF1, 0x75,
        0x64, 0x2C, 0x28, 0x28, 0x08, 0x92, 0xA7, 0x87,
        0x8C, 0xC3, 0x18, 0xBC, 0xC7, 0x5B, 0x83, 0x4C,
        0xB2, 0x9F, 0xDF, 0x53, 0x60, 0xD7, 0xF9, 0x82,
        0xA5, 0x2C, 0x88, 0xAE, 0x91, 0x4D, 0xBF, 0x02,
        0xB5, 0x8B, 0xEB, 0x8B, 0xA8, 0x87, 0xAE, 0x8F,
        0xAB, 0x5E, 0xB7, 0x87, 0x31, 0xC6, 0x75, 0x78,
        0x05, 0x47, 0x1E, 0xBC, 0xEC, 0x2E, 0x38, 0xDB,
        0x1F, 0x4B, 0x83, 0x10, 0xD2, 0x88, 0x92, 0x0D,
        0x8A, 0x49, 0x27, 0x95, 0xA3, 0x90, 0xA7, 0x4B,
        0xCD, 0x55, 0xCD, 0x85, 0x57, 0xB4, 0xDA, 0xAB,
        0xA8, 0x2C, 0x28, 0xCB, 0x3F, 0x15, 0x2C, 0x52,
        0x31, 0x19, 0x61, 0x93, 0xA6, 0x6A, 0x8C, 0xCF,
        0x34, 0xB8, 0x0E, 0x1F, 0x69, 0x42, 0xC3, 0x2B,
        0xCF, 0xF9, 0x6A, 0x6E, 0x3C, 0xF3, 0x93, 0x9B,
        0x7B, 0x94, 0x24, 0x98, 0xCC, 0x5E, 0x4C, 0xB8,
        0xE8, 0x46, 0x8E, 0x70, 0x27, 0x59, 0x85, 0x2A,
        0xA2, 0x29, 0xC0, 0x25, 0x7F, 0x02, 0x98, 0x20,
        0x97, 0x33, 0x86, 0x07, 0xC0, 0xF0, 0xF4, 0x54,
        0x46, 0xFA, 0xB4, 0x26, 0x79, 0x93, 0xB8, 0xA5,
        0x90, 0x8C, 0xAB, 0x9C, 0x46, 0x78, 0x01, 0x34,
        0x80, 0x4A, 0xE1, 0x88, 0x15, 0xB1, 0x02, 0x05,
        0x27, 0xA2, 0x22, 0xEC, 0x4B, 0x39, 0xA3, 0x19,
        0x4E, 0x66, 0x17, 0x37, 0x79, 0x17, 0x14, 0x12,
        0x26, 0x62, 0xD8, 0xB9, 0x76, 0x9F, 0x6C, 0x67,
        0xDE, 0x62, 0x5C, 0x0D, 0x48, 0x3C, 0x3D, 0x42,
        0x0F, 0xF1, 0xBB, 0x88, 0x9A, 0x72, 0x7E, 0x75,
        0x62, 0x81, 0x51, 0x3A, 0x70, 0x04, 0x76, 0x48,
        0xD2, 0x9C, 0x0C, 0x30, 0xF9, 0xBE, 0x52, 0xEC,
        0x0D, 0xEB, 0x97, 0x7C, 0xF0, 0xF3, 0x4F, 0xC2,
        0x07, 0x84, 0x83, 0x45, 0x69, 0x64, 0x74, 0x34,
        0x10, 0x63, 0x8C, 0x57, 0xB5, 0x53, 0x95, 0x77,
        0xBF, 0x85, 0x66, 0x90, 0x78, 0xC3, 0x56, 0xB3,
        0x46, 0x2E, 0x9F, 0xA5, 0x80, 0x7D, 0x49, 0x59,
        0x1A, 0xFA, 0x41, 0xC1, 0x96, 0x9F, 0x65, 0xE3,
        0x40, 0x5C, 0xB6, 0x4D, 0xDF, 0x16, 0x3F, 0x26,
        0x73, 0x4C, 0xE3, 0x48, 0xB9, 0xCF, 0x45, 0x67,
        0xA3, 0x3A, 0x59, 0x69, 0xEB, 0x32, 0x6C, 0xFB,
        0x5A, 0xDC, 0x69, 0x5D, 0xCA, 0x0C, 0x8B, 0x2A,
        0x7B, 0x1F, 0x4F, 0x40, 0x4C, 0xC7, 0xA0, 0x98,
        0x1E, 0x2C, 0xC2, 0x4C, 0x1C, 0x23, 0xD1, 0x6A,
        0xA9, 0xB4, 0x39, 0x24, 0x15, 0xE2, 0x6C, 0x22,
        0xF4, 0xA9, 0x34, 0xD7, 0x94, 0xC1, 0xFB, 0x4E,
        0x5A, 0x67, 0x05, 0x11, 0x23, 0xCC, 0xD1, 0x53,
        0x76, 0x4D, 0xEC, 0x99, 0xD5, 0x53, 0x52, 0x90,
        0x53, 0xC3, 0xDA, 0x55, 0x0B, 0xCE, 0xA3, 0xAC,
        0x54, 0x13, 0x6A, 0x26, 0xA6, 0x76, 0xD2, 0xBA,
        0x84, 0x21, 0x06, 0x70, 0x68, 0xC6, 0x38, 0x1C,
        0x2A, 0x62, 0xA7, 0x27, 0xC9, 0x33, 0x70, 0x2E,
        0xE5, 0x80, 0x4A, 0x31, 0xCA, 0x86, 0x5A, 0x45,
        0x58, 0x8F, 0xB7, 0x4D, 0xE7, 0xE2, 0x22, 0x3D,
        0x88, 0xC0, 0x60, 0x8A, 0x16, 0xBF, 0xEC, 0x4F,
        0xAD, 0x67, 0x52, 0xDB, 0x56, 0xB4, 0x8B, 0x88,
        0x72, 0xBF, 0x26, 0xBA, 0x2F, 0xFA, 0x0C, 0xED,
        0xE5, 0x34, 0x3B, 0xE8, 0x14, 0x36, 0x89, 0x26,
        0x5E, 0x06, 0x5F, 0x41, 0xA6, 0x92, 0x5B, 0x86,
        0xC8, 0x92, 0xE6, 0x2E, 0xB0, 0x77, 0x27, 0x34,
        0xF5, 0xA3, 0x57, 0xC7, 0x5C, 0xA1, 0xAC, 0x6D,
        0xF7, 0x8A, 0xB1, 0xB8, 0x88, 0x5A, 0xD0, 0x81,
        0x96, 0x15, 0x37, 0x6D, 0x33, 0xEB, 0xB9, 0x8F,
        0x87, 0x33, 0xA6, 0x75, 0x58, 0x03, 0xD9, 0x77,
        0xBF, 0x51, 0xC1, 0x27, 0x40, 0x42, 0x4B, 0x2B,
        0x49, 0xC2, 0x83, 0x82, 0xA6, 0x91, 0x7C, 0xBF,
        0xA0, 0x34, 0xC3, 0xF1, 0x26, 0xA3, 0x8C, 0x21,
        0x6C, 0x03, 0xC3, 0x57, 0x70, 0xAD, 0x48, 0x1B,
        0x90, 0x84, 0xB5, 0x58, 0x8D, 0xA6, 0x5F, 0xF1,
        0x18, 0xA7, 0x4F, 0x93, 0x2C, 0x7E, 0x53, 0x7A,
        0xBE, 0x58, 0x63, 0xFB, 0x29, 0xA1, 0x0C, 0x09,
        0x70, 0x1B, 0x44, 0x1F, 0x83, 0x99, 0xC1, 0xF8,
        0xA6, 0x37, 0x82, 0x5A, 0xCE, 0xA3, 0xE9, 0x31,
        0x80, 0x57, 0x4F, 0xDE, 0xB8, 0x80, 0x76, 0x66,
        0x1A, 0xB4, 0x69, 0x51, 0x71, 0x6A, 0x50, 0x01,
        0x84, 0xA0, 0x40, 0x55, 0x72, 0x66, 0x59, 0x8C,
        0xAF, 0x76, 0x10, 0x5E, 0x1C, 0x18, 0x70, 0xB4,
        0x39, 0x69, 0xC3, 0xBC, 0xC1, 0xA0, 0x49, 0x27,
        0x63, 0x80, 0x17, 0x49, 0x8B, 0xB6, 0x2C, 0xAF,
        0xD3, 0xA6, 0xB0, 0x82, 0xB7, 0xBF, 0x7A, 0x23,
        0x45, 0x0E, 0x19, 0x17, 0x99, 0x61, 0x9B, 0x92,
        0x51, 0x12, 0xD0, 0x72, 0x02, 0x5C, 0xA8, 0x88,
        0x54, 0x8C, 0x79, 0x1A, 0xA4, 0x22, 0x51, 0x50,
        0x4D, 0x5D, 0x1C, 0x1C, 0xDD, 0xB2, 0x13, 0x30,
        0x3B, 0x04, 0x9E, 0x73, 0x46, 0xE8, 0xD8, 0x3A,
        0xD5, 0x87, 0x83, 0x6F, 0x35, 0x28, 0x4E, 0x10,
        0x97, 0x27, 0xE6, 0x6B, 0xBC, 0xC9, 0x52, 0x1F,
        0xE0, 0xB1, 0x91, 0x63, 0x00, 0x47, 0xD1, 0x58,
        0xF7, 0x56, 0x40, 0xFF, 0xEB, 0x54, 0x56, 0x07,
        0x27, 0x40, 0x02, 0x1A, 0xFD, 0x15, 0xA4, 0x54,
        0x69, 0xC5, 0x83, 0x82, 0x9D, 0xAA, 0xC8, 0xA7,
        0xDE, 0xB0, 0x5B, 0x24, 0xF0, 0x56, 0x7E, 0x43,
        0x17, 0xB3, 0xE3, 0xB3, 0x33, 0x89, 0xB5, 0xC5,
        0xF8, 0xB0, 0x4B, 0x09, 0x9F, 0xB4, 0xD1, 0x03,
        0xA3, 0x24, 0x39, 0xF8, 0x5A, 0x3C, 0x21, 0xD2,
        0x1A, 0x71, 0xB9, 0xB9, 0x2A, 0x9B, 0x64, 0xEA,
        0x0A, 0xB8, 0x43, 0x12, 0xC7, 0x70, 0x23, 0x69,
        0x4F, 0xD6, 0x4E, 0xAA, 0xB9, 0x07, 0xA4, 0x35,
        0x39, 0xDD, 0xB2, 0x7B, 0xA0, 0xA8, 0x53, 0xCC,
        0x90, 0x69, 0xEA, 0xC8, 0x50, 0x8C, 0x65, 0x3E,
        0x60, 0x0B, 0x2A, 0xC0, 0x18, 0x38, 0x1B, 0x4B,
        0xB4, 0xA8, 0x79, 0xAC, 0xDA, 0xD3, 0x42, 0xF9,
        0x11, 0x79, 0xCA, 0x82, 0x49, 0x52, 0x5C, 0xB1,
        0x96, 0x8B, 0xBE, 0x52, 0xF7, 0x55, 0xB7, 0xF5,
        0xB4, 0x3D, 0x66, 0x63, 0xD7, 0xA3, 0xBF, 0x0F,
        0x33, 0x57, 0xD8, 0xA2, 0x1D, 0x15, 0xB5, 0x2D,
        0xB3, 0x81, 0x8E, 0xCE, 0x5B, 0x40, 0x2A, 0x60,
        0xC9, 0x93, 0xE7, 0xCF, 0x43, 0x64, 0x87, 0xB8,
        0xD2, 0xAE, 0x91, 0xE6, 0xC5, 0xB8, 0x82, 0x75,
        0xE7, 0x58, 0x24, 0xB0, 0x00, 0x7E, 0xF3, 0x12,
        0x3C, 0x0A, 0xB5, 0x1B, 0x5C, 0xC6, 0x1B, 0x9B,
        0x22, 0x38, 0x0D, 0xE6, 0x6C, 0x5B, 0x20, 0xB0,
        0x60, 0xCB, 0xB9, 0x86, 0xF8, 0x12, 0x3D, 0x94,
        0x06, 0x00, 0x49, 0xCD, 0xF8, 0x03, 0x68, 0x73,
        0xA7, 0xBE, 0x10, 0x94, 0x44, 0xA0, 0xA1, 0xCD,
        0x87, 0xA4, 0x8C, 0xAE, 0x54, 0x19, 0x24, 0x84,
        0xAF, 0x84, 0x44, 0x29, 0xC1, 0xC5, 0x8C, 0x29,
        0xAC, 0x62, 0x4C, 0xD5, 0x04, 0xF1, 0xC4, 0x4F,
        0x1E, 0x13, 0x47, 0x82, 0x2B, 0x6F, 0x22, 0x13,
        0x23, 0x85, 0x9A, 0x7F, 0x6F, 0x75, 0x4B, 0xFE,
        0x71, 0x0B, 0xDA, 0x60, 0x27, 0x62, 0x40, 0xA4,
        0xFF, 0x2A, 0x53, 0x50, 0x70, 0x37, 0x86, 0xF5,
        0x67, 0x1F, 0x44, 0x9F, 0x20, 0xC2, 0xA9, 0x5A,
        0xE7, 0xC2, 0x90, 0x3A, 0x42, 0xCB, 0x3B, 0x30,
        0x3F, 0xF4, 0xC4, 0x27, 0xC0, 0x8B, 0x11, 0xB4,
        0xCD, 0x31, 0xC4, 0x18, 0xC6, 0xD1, 0x8D, 0x08,
        0x61, 0x87, 0x3B, 0xFA, 0x03, 0x32, 0xF1, 0x12,
        0x71, 0x55, 0x2E, 0xD7, 0xC0, 0x35, 0xF0, 0xE4,
        0xBC, 0x42, 0x8C, 0x43, 0x72, 0x0B, 0x39, 0xA6,
        0x51, 0x66, 0xBA, 0x9C, 0x2D, 0x3D, 0x77, 0x0E,
        0x13, 0x03, 0x60, 0xCC, 0x23, 0x84, 0xE8, 0x30,
        0x95, 0xB1, 0xA1, 0x59, 0x49, 0x55, 0x33, 0xF1,
        0x16, 0xC7, 0xB5, 0x58, 0xB6, 0x50, 0xDB, 0x04,
        0xD5, 0xA2, 0x6E, 0xAA, 0xA0, 0x8C, 0x3E, 0xE5,
        0x7D, 0xE4, 0x5A, 0x7F, 0x88, 0xC6, 0xA3, 0xCE,
        0xB2, 0x4D, 0xC5, 0x39, 0x7B, 0x88, 0xC3, 0xCE,
        0xF0, 0x03, 0x31, 0x9B, 0xB0, 0x23, 0x3F, 0xD6,
        0x92, 0xFD, 0xA1, 0x52, 0x44, 0x75, 0xB3, 0x51,
        0xF3, 0xC7, 0x82, 0x18, 0x2D, 0xEC, 0xF5, 0x90,
        0xB7, 0x72, 0x3B, 0xE4, 0x00, 0xBE, 0x14, 0x80,
        0x9C, 0x44, 0x32, 0x99, 0x63, 0xFC, 0x46, 0x95,
        0x92, 0x11, 0xD6, 0xA6, 0x23, 0x33, 0x95, 0x37,
        0x84, 0x8C, 0x25, 0x16, 0x69, 0x94, 0x1D, 0x90,
        0xB1, 0x30, 0x25, 0x8A, 0xDF, 0x55, 0xA7, 0x20,
        0xA7, 0x24, 0xE8, 0xB6, 0xA6, 0xCA, 0xE3, 0xC2,
        0x26, 0x4B, 0x16, 0x24, 0xCC, 0xBE, 0x7B, 0x45,
        0x6B, 0x30, 0xC8, 0xC7, 0x39, 0x32, 0x94, 0xCA,
        0x51, 0x80, 0xBC, 0x83, 0x7D, 0xD2, 0xE4, 0x5D,
        0xBD, 0x59, 0xB6, 0xE1, 0x7B, 0x24, 0xFE, 0x93,
        0x05, 0x2E, 0xB7, 0xC4, 0x3B, 0x27, 0xAC, 0x3D,
        0xC2, 0x49, 0xCA, 0x0C, 0xBC, 0xA4, 0xFB, 0x58,
        0x97, 0xC0, 0xB7, 0x44, 0x08, 0x8A, 0x8A, 0x07,
        0x79, 0xD3, 0x22, 0x33, 0x82, 0x6A, 0x01, 0xDD,
        0x64, 0x89, 0x95, 0x2A, 0x48, 0x25, 0xE5, 0x35,
        0x8A, 0x70, 0x0B, 0xE0, 0xE1, 0x79, 0xAC, 0x19,
        0x77, 0x10, 0xD8, 0x3E, 0xCC, 0x85, 0x3E, 0x52,
        0x69, 0x5E, 0x9B, 0xF8, 0x7B, 0xB1, 0xF6, 0xCB,
        0xD0, 0x5B, 0x02, 0xD4, 0xE6, 0x79, 0xE3, 0xB8,
        0x8D, 0xD4, 0x83, 0xB0, 0x74, 0x9B, 0x11, 0xBD,
        0x37, 0xB3, 0x83, 0xDC, 0xCA, 0x71, 0xF9, 0x09,
        0x18, 0x34, 0xA1, 0x69, 0x55, 0x02, 0xC4, 0xB9,
        0x5F, 0xC9, 0x11, 0x8C, 0x1C, 0xFC, 0x34, 0xC8,
        0x4C, 0x22, 0x65, 0xBB, 0xBC, 0x56, 0x3C, 0x28,
        0x26, 0x66, 0xB6, 0x0A, 0xE5, 0xC7, 0xF3, 0x85,
        0x1D, 0x25, 0xEC, 0xBB, 0x50, 0x21, 0xCC, 0x38,
        0xCB, 0x73, 0xEB, 0x6A, 0x34, 0x11, 0xB1, 0xC2,
        0x90, 0x46, 0xCA, 0x66, 0x54, 0x06, 0x67, 0xD1,
        0x36, 0x95, 0x44, 0x60, 0xC6, 0xFC, 0xBC, 0x4B,
        0xC7, 0xC0, 0x49, 0xBB, 0x04, 0x7F, 0xA6, 0x7A,
        0x63, 0xB3, 0xCC, 0x11, 0x11, 0xC1, 0xD8, 0xAC,
        0x27, 0xE8, 0x05, 0x8B, 0xCC, 0xA4, 0xA1, 0x54,
        0x55, 0x85, 0x8A, 0x58, 0x35, 0x8F, 0x7A, 0x61,
        0x02, 0x0B, 0xC9, 0xC4, 0xC1, 0x7F, 0x8B, 0x95,
        0xC2, 0x68, 0xCC, 0xB4, 0x04, 0xB9, 0xAA, 0xB4,
        0xA2, 0x72, 0xA2, 0x1A, 0x70, 0xDA, 0xF6, 0xB6,
        0xF1, 0x51, 0x21, 0xEE, 0x01, 0xC1, 0x56, 0xA3,
        0x54, 0xAA, 0x17, 0x08, 0x7E, 0x07, 0x70, 0x2E,
        0xAB, 0x38, 0xB3, 0x24, 0x1F, 0xDB, 0x55, 0x3F,
        0x65, 0x73, 0x39, 0xD5, 0xE2, 0x9D, 0xC5, 0xD9,
        0x1B, 0x7A, 0x5A, 0x82, 0x8E, 0xE9, 0x59, 0xFE,
        0xBB, 0x90, 0xB0, 0x72, 0x29, 0xF6, 0xE4, 0x9D,
        0x23, 0xC3, 0xA1, 0x90, 0x29, 0x70, 0x42, 0xFB,
        0x43, 0x98, 0x69, 0x55, 0xB6, 0x9C, 0x28, 0xE1,
        0x01, 0x6F, 0x77, 0xA5, 0x8B, 0x43, 0x15, 0x14,
        0xD2, 0x1B, 0x88, 0x88, 0x99, 0xC3, 0x60, 0x82,
        0x76, 0x08, 0x1B, 0x75, 0xF5, 0x68, 0x09, 0x7C,
        0xDC, 0x17, 0x48, 0xF3, 0x23, 0x07, 0x88, 0x58,
        0x15, 0xF3, 0xAE, 0xC9, 0x65, 0x18, 0x19, 0xAA,
        0x68, 0x73, 0xD1, 0xA4, 0xEB, 0x83, 0xB1, 0x95,
        0x38, 0x43, 0xB9, 0x34, 0x22, 0x51, 0x94, 0x83,
        0xFE, 0xF0, 0x05, 0x9D, 0x36, 0xBB, 0x2D, 0xB1,
        0xF3, 0xD4, 0x68, 0xFB, 0x06, 0x8C, 0x86, 0xE8,
        0x97, 0x37, 0x33, 0xC3, 0x98, 0xEA, 0xF0, 0x0E,
        0x17, 0x02, 0xC6, 0x73, 0x4A, 0xD8, 0xEB, 0x3B,
        0x62, 0x01, 0x30, 0xD6, 0xC2, 0xB8, 0xC9, 0x04,
        0xA3, 0xBB, 0x93, 0x07, 0xBE, 0x51, 0x03, 0xF8,
        0xD8, 0x14, 0x50, 0x5F, 0xB6, 0xA6, 0x0A, 0xF7,
        0x93, 0x7E, 0xA6, 0xCA, 0xA1, 0x17, 0x31, 0x5E,
        0x84, 0xCC, 0x91, 0x21, 0xAE, 0x56, 0xFB, 0xF3,
        0x9E, 0x67, 0xAD, 0xBD, 0x83, 0xAD, 0x2D, 0x3E,
        0x3B, 0xB8, 0x08, 0x43, 0x64, 0x52, 0x06, 0xBD,
        0xD9, 0xF2, 0xF6, 0x29, 0xE3, 0xCC, 0x49, 0xB7
    };
#endif
#ifndef WOLFSSL_NO_ML_KEM_768
    static const byte seed_768[WC_ML_KEM_MAKEKEY_RAND_SZ] = {
        /* d */
        0xE3, 0x4A, 0x70, 0x1C, 0x4C, 0x87, 0x58, 0x2F,
        0x42, 0x26, 0x4E, 0xE4, 0x22, 0xD3, 0xC6, 0x84,
        0xD9, 0x76, 0x11, 0xF2, 0x52, 0x3E, 0xFE, 0x0C,
        0x99, 0x8A, 0xF0, 0x50, 0x56, 0xD6, 0x93, 0xDC,
        /* z */
        0xA8, 0x57, 0x68, 0xF3, 0x48, 0x6B, 0xD3, 0x2A,
        0x01, 0xBF, 0x9A, 0x8F, 0x21, 0xEA, 0x93, 0x8E,
        0x64, 0x8E, 0xAE, 0x4E, 0x54, 0x48, 0xC3, 0x4C,
        0x3E, 0xB8, 0x88, 0x20, 0xB1, 0x59, 0xEE, 0xDD
    };
    static const byte ek_768[WC_ML_KEM_768_PUBLIC_KEY_SIZE] = {
        0x6D, 0x14, 0xA0, 0x71, 0xF7, 0xCC, 0x45, 0x25,
        0x58, 0xD5, 0xE7, 0x1A, 0x7B, 0x08, 0x70, 0x62,
        0xEC, 0xB1, 0x38, 0x68, 0x44, 0x58, 0x82, 0x46,
        0x12, 0x64, 0x02, 0xB1, 0xFA, 0x16, 0x37, 0x73,
        0x3C, 0xD5, 0xF6, 0x0C, 0xC8, 0x4B, 0xCB, 0x64,
        0x6A, 0x78, 0x92, 0x61, 0x4D, 0x7C, 0x51, 0xB1,
        0xC7, 0xF1, 0xA2, 0x79, 0x91, 0x32, 0xF1, 0x34,
        0x27, 0xDC, 0x48, 0x21, 0x58, 0xDA, 0x25, 0x44,
        0x70, 0xA5, 0x9E, 0x00, 0xA4, 0xE4, 0x96, 0x86,
        0xFD, 0xC0, 0x77, 0x55, 0x93, 0x67, 0x27, 0x0C,
        0x21, 0x53, 0xF1, 0x10, 0x07, 0x59, 0x2C, 0x9C,
        0x43, 0x10, 0xCF, 0x8A, 0x12, 0xC6, 0xA8, 0x71,
        0x3B, 0xD6, 0xBB, 0x51, 0xF3, 0x12, 0x4F, 0x98,
        0x9B, 0xA0, 0xD5, 0x40, 0x73, 0xCC, 0x24, 0x2E,
        0x09, 0x68, 0x78, 0x0B, 0x87, 0x5A, 0x86, 0x9E,
        0xFB, 0x85, 0x15, 0x86, 0xB9, 0xA8, 0x68, 0xA3,
        0x84, 0xB9, 0xE6, 0x82, 0x1B, 0x20, 0x1B, 0x93,
        0x2C, 0x45, 0x53, 0x69, 0xA7, 0x39, 0xEC, 0x22,
        0x56, 0x9C, 0x97, 0x7C, 0x21, 0x2B, 0x38, 0x18,
        0x71, 0x81, 0x36, 0x56, 0xAF, 0x5B, 0x56, 0x7E,
        0xF8, 0x93, 0xB5, 0x84, 0x62, 0x4C, 0x86, 0x3A,
        0x25, 0x90, 0x00, 0xF1, 0x7B, 0x25, 0x4B, 0x98,
        0xB1, 0x85, 0x09, 0x7C, 0x50, 0xEB, 0xB6, 0x8B,
        0x24, 0x43, 0x42, 0xE0, 0x5D, 0x4D, 0xE5, 0x20,
        0x12, 0x5B, 0x8E, 0x10, 0x33, 0xB1, 0x43, 0x60,
        0x93, 0xAC, 0xE7, 0xCE, 0x8E, 0x71, 0xB4, 0x58,
        0xD5, 0x25, 0x67, 0x33, 0x63, 0x04, 0x5A, 0x3B,
        0x3E, 0xEA, 0x94, 0x55, 0x42, 0x8A, 0x39, 0x87,
        0x05, 0xA4, 0x23, 0x27, 0xAD, 0xB3, 0x77, 0x4B,
        0x70, 0x57, 0xF4, 0x2B, 0x01, 0x7E, 0xC0, 0x73,
        0x9A, 0x98, 0x3F, 0x19, 0xE8, 0x21, 0x4D, 0x09,
        0x19, 0x5F, 0xA2, 0x4D, 0x2D, 0x57, 0x1D, 0xB7,
        0x3C, 0x19, 0xA6, 0xF8, 0x46, 0x0E, 0x50, 0x83,
        0x0D, 0x41, 0x5F, 0x62, 0x7B, 0x88, 0xE9, 0x4A,
        0x7B, 0x15, 0x37, 0x91, 0xA0, 0xC0, 0xC7, 0xE9,
        0x48, 0x4C, 0x74, 0xD5, 0x3C, 0x71, 0x48, 0x89,
        0xF0, 0xE3, 0x21, 0xB6, 0x66, 0x0A, 0x53, 0x2A,
        0x5B, 0xC0, 0xE5, 0x57, 0xFB, 0xCA, 0x35, 0xE2,
        0x9B, 0xC6, 0x11, 0x20, 0x0E, 0xD3, 0xC6, 0x33,
        0x07, 0x7A, 0x4D, 0x87, 0x3C, 0x5C, 0xC6, 0x70,
        0x06, 0xB7, 0x53, 0xBF, 0x6D, 0x6B, 0x7A, 0xF6,
        0xCA, 0x40, 0x2A, 0xB6, 0x18, 0x23, 0x6C, 0x0A,
        0xFF, 0xBC, 0x80, 0x1F, 0x82, 0x22, 0xFB, 0xC3,
        0x6C, 0xE0, 0x98, 0x4E, 0x2B, 0x18, 0xC9, 0x44,
        0xBB, 0xCB, 0xEF, 0x03, 0xB1, 0xE1, 0x36, 0x1C,
        0x1F, 0x44, 0xB0, 0xD7, 0x34, 0xAF, 0xB1, 0x56,
        0x6C, 0xFF, 0x87, 0x44, 0xDA, 0x8B, 0x99, 0x43,
        0xD6, 0xB4, 0x5A, 0x3C, 0x09, 0x03, 0x07, 0x02,
        0xCA, 0x20, 0x1F, 0xFE, 0x20, 0xCB, 0x7E, 0xC5,
        0xB0, 0xD4, 0x14, 0x9E, 0xE2, 0xC2, 0x8E, 0x8B,
        0x23, 0x37, 0x4F, 0x47, 0x1B, 0x57, 0x15, 0x0D,
        0x0E, 0xC9, 0x33, 0x62, 0x61, 0xA2, 0xD5, 0xCB,
        0x84, 0xA3, 0xAC, 0xAC, 0xC4, 0x28, 0x94, 0x73,
        0xA4, 0xC0, 0xAB, 0xC6, 0x17, 0xC9, 0xAB, 0xC1,
        0x78, 0x73, 0x44, 0x34, 0xC8, 0x2E, 0x16, 0x85,
        0x58, 0x8A, 0x5C, 0x2E, 0xA2, 0x67, 0x8F, 0x6B,
        0x3C, 0x22, 0x28, 0x73, 0x31, 0x30, 0xC4, 0x66,
        0xE5, 0xB8, 0x6E, 0xF4, 0x91, 0x15, 0x3E, 0x48,
        0x66, 0x22, 0x47, 0xB8, 0x75, 0xD2, 0x01, 0x02,
        0x0B, 0x56, 0x6B, 0x81, 0xB6, 0x4D, 0x83, 0x9A,
        0xB4, 0x63, 0x3B, 0xAA, 0x8A, 0xCE, 0x20, 0x2B,
        0xAA, 0xB4, 0x49, 0x62, 0x97, 0xF9, 0x80, 0x7A,
        0xDB, 0xBB, 0x1E, 0x33, 0x2C, 0x6F, 0x80, 0x22,
        0xB2, 0xA1, 0x8C, 0xFD, 0xD4, 0xA8, 0x25, 0x30,
        0xB6, 0xD3, 0xF0, 0x07, 0xC3, 0x35, 0x38, 0x98,
        0xD9, 0x66, 0xCC, 0x2C, 0x21, 0xCB, 0x42, 0x44,
        0xBD, 0x00, 0x44, 0x3F, 0x20, 0x98, 0x70, 0xAC,
        0xC4, 0x2B, 0xC3, 0x30, 0x68, 0xC7, 0x24, 0xEC,
        0x17, 0x22, 0x36, 0x19, 0xC1, 0x09, 0x3C, 0xCA,
        0x6A, 0xEB, 0x29, 0x50, 0x06, 0x64, 0xD1, 0x22,
        0x50, 0x36, 0xB4, 0xB8, 0x10, 0x91, 0x90, 0x69,
        0x69, 0x48, 0x1F, 0x1C, 0x72, 0x3C, 0x14, 0x0B,
        0x9D, 0x6C, 0x16, 0x8F, 0x5B, 0x64, 0xBE, 0xA6,
        0x9C, 0x5F, 0xD6, 0x38, 0x5D, 0xF7, 0x36, 0x4B,
        0x87, 0x23, 0xBC, 0xC8, 0x5E, 0x03, 0x8C, 0x7E,
        0x46, 0x4A, 0x90, 0x0D, 0x68, 0xA2, 0x12, 0x78,
        0x18, 0x99, 0x42, 0x17, 0xAE, 0xC8, 0xBD, 0xB3,
        0x9A, 0x97, 0x0A, 0x99, 0x63, 0xDE, 0x93, 0x68,
        0x8E, 0x2A, 0xC8, 0x2A, 0xBC, 0xC2, 0x2F, 0xB9,
        0x27, 0x7B, 0xA2, 0x20, 0x09, 0xE8, 0x78, 0x38,
        0x1A, 0x38, 0x16, 0x39, 0x01, 0xC7, 0xD4, 0xC8,
        0x50, 0x19, 0x53, 0x8D, 0x35, 0xCA, 0xAE, 0x9C,
        0x41, 0xAF, 0x8C, 0x92, 0x9E, 0xE2, 0x0B, 0xB0,
        0x8C, 0xA6, 0x19, 0xE7, 0x2C, 0x2F, 0x22, 0x62,
        0xC1, 0xC9, 0x93, 0x85, 0x72, 0x55, 0x1A, 0xC0,
        0x2D, 0xC9, 0x26, 0x8F, 0xBC, 0xC3, 0x5D, 0x79,
        0x01, 0x1C, 0x3C, 0x09, 0x0A, 0xD4, 0x0A, 0x4F,
        0x11, 0x1C, 0x9B, 0xE5, 0x5C, 0x42, 0x7E, 0xB7,
        0x96, 0xC1, 0x93, 0x2D, 0x86, 0x73, 0x57, 0x9A,
        0xF1, 0xB4, 0xC6, 0x38, 0xB0, 0x94, 0x44, 0x89,
        0x01, 0x2A, 0x25, 0x59, 0xA3, 0xB0, 0x24, 0x81,
        0xB0, 0x1A, 0xC3, 0x0B, 0xA8, 0x96, 0x0F, 0x80,
        0xC0, 0xC2, 0xB3, 0x94, 0x7D, 0x36, 0xA1, 0x2C,
        0x08, 0x04, 0x98, 0xBE, 0xE4, 0x48, 0x71, 0x6C,
        0x97, 0x34, 0x16, 0xC8, 0x24, 0x28, 0x04, 0xA3,
        0xDA, 0x09, 0x9E, 0xE1, 0x37, 0xB0, 0xBA, 0x90,
        0xFE, 0x4A, 0x5C, 0x6A, 0x89, 0x20, 0x02, 0x76,
        0xA0, 0xCF, 0xB6, 0x43, 0xEC, 0x2C, 0x56, 0xA2,
        0xD7, 0x08, 0xD7, 0xB4, 0x37, 0x3E, 0x44, 0xC1,
        0x50, 0x2A, 0x76, 0x3A, 0x60, 0x05, 0x86, 0xE6,
        0xCD, 0xA6, 0x27, 0x38, 0x97, 0xD4, 0x44, 0x48,
        0x28, 0x7D, 0xC2, 0xE6, 0x02, 0xDC, 0x39, 0x20,
        0x0B, 0xF6, 0x16, 0x62, 0x36, 0x55, 0x9F, 0xD1,
        0x2A, 0x60, 0x89, 0x2A, 0xEB, 0x15, 0x3D, 0xD6,
        0x51, 0xBB, 0x46, 0x99, 0x10, 0xB4, 0xB3, 0x46,
        0x69, 0xF9, 0x1D, 0xA8, 0x65, 0x4D, 0x1E, 0xB7,
        0x2E, 0xB6, 0xE0, 0x28, 0x00, 0xB3, 0xB0, 0xA7,
        0xD0, 0xA4, 0x8C, 0x83, 0x68, 0x54, 0xD3, 0xA8,
        0x3E, 0x65, 0x56, 0x9C, 0xB7, 0x23, 0x0B, 0xB4,
        0x4F, 0x3F, 0x14, 0x3A, 0x6D, 0xEC, 0x5F, 0x2C,
        0x39, 0xAB, 0x90, 0xF2, 0x74, 0xF2, 0x08, 0x8B,
        0xD3, 0xD6, 0xA6, 0xFC, 0xA0, 0x07, 0x02, 0x73,
        0xBE, 0xDC, 0x84, 0x77, 0x7F, 0xB5, 0x2E, 0x3C,
        0x55, 0x8B, 0x0A, 0xE0, 0x61, 0x83, 0xD5, 0xA4,
        0x8D, 0x45, 0x2F, 0x68, 0xE1, 0x52, 0x07, 0xF8,
        0x61, 0x62, 0x7A, 0xCA, 0x14, 0x27, 0x96, 0x30,
        0xF8, 0x2E, 0xC3, 0xA0, 0xCA, 0x07, 0x86, 0x33,
        0xB6, 0x00, 0xAF, 0xA7, 0x97, 0x43, 0xA6, 0x00,
        0x21, 0x5B, 0xE5, 0x63, 0x74, 0x58, 0xCE, 0x2C,
        0xE8, 0xAF, 0xF5, 0xA0, 0x8E, 0xB5, 0x01, 0x7B,
        0x2C, 0x76, 0x65, 0x77, 0x47, 0x9F, 0x8D, 0xC6,
        0xBF, 0x9F, 0x5C, 0xC7, 0x50, 0x89, 0x93, 0x21,
        0x61, 0xB9, 0x6C, 0xEA, 0x40, 0x66, 0x20, 0xAE,
        0xDB, 0x63, 0x04, 0x07, 0xF7, 0x68, 0x7E, 0xBB,
        0xB4, 0x81, 0x4C, 0x79, 0x81, 0x63, 0x7A, 0x48,
        0xA9, 0x0D, 0xE6, 0x80, 0x31, 0xE0, 0x62, 0xA7,
        0xAF, 0x76, 0x12, 0xB4, 0xF5, 0xC7, 0xA6, 0xDA,
        0x86, 0xBD, 0x13, 0x65, 0x29, 0xE6, 0x42, 0x95,
        0xA5, 0x61, 0x3E, 0xA7, 0x3B, 0xD3, 0xD4, 0x44,
        0x8C, 0xB8, 0x1F, 0x24, 0x31, 0x35, 0xC0, 0xA6,
        0x60, 0xBE, 0xB9, 0xC1, 0x7E, 0x65, 0x1D, 0xEF,
        0x46, 0x9A, 0x7D, 0x90, 0xA1, 0x5D, 0x34, 0x81,
        0x09, 0x0B, 0xCB, 0xF2, 0x27, 0x01, 0x23, 0x28,
        0x94, 0x1F, 0xA4, 0x6F, 0x39, 0xC5, 0x00, 0x6A,
        0xD9, 0x3D, 0x45, 0x8A, 0xA6, 0xAD, 0xD6, 0x55,
        0x86, 0x2B, 0x41, 0x8C, 0x30, 0x94, 0xF5, 0x51,
        0x46, 0x0D, 0xF2, 0x15, 0x3A, 0x58, 0x10, 0xA7,
        0xDA, 0x74, 0xF0, 0x61, 0x4C, 0x25, 0x88, 0xBE,
        0x49, 0xDC, 0x6F, 0x5E, 0x88, 0x15, 0x46, 0x42,
        0xBD, 0x1D, 0x37, 0x62, 0x56, 0x33, 0x26, 0x43,
        0x35, 0x07, 0x15, 0x6A, 0x57, 0xC5, 0x76, 0x94,
        0xBD, 0xD2, 0x6E, 0x7A, 0x24, 0x6F, 0xEB, 0x72,
        0x3A, 0xED, 0x67, 0xB0, 0x48, 0x87, 0xC8, 0xE4,
        0x76, 0xB4, 0x8C, 0xAB, 0x59, 0xE5, 0x36, 0x2F,
        0x26, 0xA9, 0xEF, 0x50, 0xC2, 0xBC, 0x80, 0xBA,
        0x14, 0x62, 0x26, 0x21, 0x6F, 0xE6, 0x29, 0x68,
        0xA6, 0x0D, 0x04, 0xE8, 0xC1, 0x70, 0xD7, 0x41,
        0xC7, 0xA2, 0xB0, 0xE1, 0xAB, 0xDA, 0xC9, 0x68
    };
    static const byte dk_768[WC_ML_KEM_768_PRIVATE_KEY_SIZE] = {
        0x98, 0xA1, 0xB2, 0xDA, 0x4A, 0x65, 0xCF, 0xB5,
        0x84, 0x5E, 0xA7, 0x31, 0x1E, 0x6A, 0x06, 0xDB,
        0x73, 0x1F, 0x15, 0x90, 0xC4, 0x1E, 0xE7, 0x4B,
        0xA1, 0x07, 0x82, 0x71, 0x5B, 0x35, 0xA3, 0x10,
        0x2D, 0xF6, 0x37, 0x87, 0x2B, 0xE6, 0x5B, 0xAB,
        0x37, 0xA1, 0xDE, 0x25, 0x11, 0xD7, 0x03, 0xC7,
        0x02, 0x47, 0xB3, 0x5E, 0xF2, 0x74, 0x35, 0x48,
        0x50, 0x24, 0xD9, 0x3F, 0xD9, 0xE7, 0x7C, 0x43,
        0x80, 0x4F, 0x37, 0x17, 0x49, 0xBA, 0x00, 0xB2,
        0x0A, 0x8C, 0x5C, 0x58, 0x8B, 0xC9, 0xAB, 0xE0,
        0x68, 0xAE, 0xAA, 0xA9, 0x38, 0x51, 0x7E, 0xBF,
        0xE5, 0x3B, 0x6B, 0x66, 0x32, 0x82, 0x90, 0x3D,
        0xCD, 0x18, 0x97, 0x36, 0xD7, 0x29, 0x68, 0x16,
        0xC7, 0x33, 0xA1, 0xC7, 0x7C, 0x63, 0x75, 0xE5,
        0x39, 0x7C, 0x0F, 0x18, 0x9B, 0xBF, 0xE4, 0x76,
        0x43, 0xA6, 0x1F, 0x58, 0xF8, 0xA3, 0xC6, 0x91,
        0x1B, 0xE4, 0x61, 0x1A, 0x8C, 0x7B, 0xC0, 0x50,
        0x02, 0x11, 0x63, 0xD0, 0xA4, 0x04, 0xDC, 0x14,
        0x06, 0x57, 0x48, 0xFF, 0x29, 0xBE, 0x60, 0xD2,
        0xB9, 0xFD, 0xCC, 0x8F, 0xFD, 0x98, 0xC5, 0x87,
        0xF3, 0x8C, 0x67, 0x11, 0x57, 0x86, 0x46, 0x4B,
        0xDB, 0x34, 0x2B, 0x17, 0xE8, 0x97, 0xD6, 0x46,
        0x17, 0xCB, 0xFB, 0x11, 0x79, 0x73, 0xA5, 0x45,
        0x89, 0x77, 0xA7, 0xD7, 0x61, 0x7A, 0x1B, 0x4D,
        0x83, 0xBA, 0x03, 0xC6, 0x11, 0x13, 0x8A, 0x46,
        0x73, 0xB1, 0xEB, 0x34, 0xB0, 0x78, 0x03, 0x3F,
        0x97, 0xCF, 0xFE, 0x80, 0xC1, 0x46, 0xA2, 0x69,
        0x43, 0xF8, 0x42, 0xB9, 0x76, 0x32, 0x7B, 0xF1,
        0xCB, 0xC6, 0x01, 0x19, 0x52, 0x5B, 0xB9, 0xA3,
        0xC0, 0x34, 0x93, 0x34, 0x90, 0x00, 0xDD, 0x8F,
        0x51, 0xBA, 0x21, 0xA2, 0xE9, 0x23, 0x61, 0x76,
        0x23, 0x24, 0x60, 0x0E, 0x0C, 0x13, 0xAA, 0xA6,
        0xCB, 0x69, 0xBF, 0xB2, 0x42, 0x76, 0x48, 0x3F,
        0x6B, 0x02, 0x42, 0x12, 0x59, 0xB7, 0x58, 0x52,
        0x63, 0xC1, 0xA0, 0x28, 0xD6, 0x82, 0xC5, 0x08,
        0xBB, 0xC2, 0x80, 0x1A, 0x56, 0xE9, 0x8B, 0x8F,
        0x62, 0x0B, 0x04, 0x83, 0xD7, 0x9B, 0x5A, 0xD8,
        0x58, 0x5A, 0xC0, 0xA4, 0x75, 0xBA, 0xC7, 0x78,
        0x65, 0x19, 0x41, 0x96, 0x33, 0x87, 0x91, 0xB7,
        0x98, 0x5A, 0x05, 0xD1, 0x09, 0x39, 0x5C, 0xCA,
        0x89, 0x32, 0x72, 0x2A, 0x91, 0x95, 0x0D, 0x37,
        0xE1, 0x2B, 0x89, 0x14, 0x20, 0xA5, 0x2B, 0x62,
        0xCB, 0xFA, 0x81, 0x5D, 0xF6, 0x17, 0x4C, 0xE0,
        0x0E, 0x68, 0xBC, 0xA7, 0x5D, 0x48, 0x38, 0xCA,
        0x28, 0x0F, 0x71, 0x3C, 0x7E, 0x69, 0x24, 0xAF,
        0xD9, 0x5B, 0xAA, 0x0D, 0x01, 0xAD, 0xA6, 0x37,
        0xB1, 0x58, 0x34, 0x70, 0x34, 0xC0, 0xAB, 0x1A,
        0x71, 0x83, 0x33, 0x1A, 0x82, 0x0A, 0xCB, 0xCB,
        0x83, 0x19, 0x3A, 0x1A, 0x94, 0xC8, 0xF7, 0xE3,
        0x84, 0xAE, 0xD0, 0xC3, 0x5E, 0xD3, 0xCB, 0x33,
        0x97, 0xBB, 0x63, 0x80, 0x86, 0xE7, 0xA3, 0x5A,
        0x64, 0x08, 0xA3, 0xA4, 0xB9, 0x0C, 0xE9, 0x53,
        0x70, 0x7C, 0x19, 0xBC, 0x46, 0xC3, 0xB2, 0xDA,
        0x3B, 0x2E, 0xE3, 0x23, 0x19, 0xC5, 0x6B, 0x92,
        0x80, 0x32, 0xB5, 0xED, 0x12, 0x56, 0xD0, 0x75,
        0x3D, 0x34, 0x14, 0x23, 0xE9, 0xDB, 0x13, 0x9D,
        0xE7, 0x71, 0x4F, 0xF0, 0x75, 0xCA, 0xF5, 0x8F,
        0xD9, 0xF5, 0x7D, 0x1A, 0x54, 0x01, 0x9B, 0x59,
        0x26, 0x40, 0x68, 0x30, 0xDA, 0xE2, 0x9A, 0x87,
        0x53, 0x02, 0xA8, 0x12, 0x56, 0xF4, 0xD6, 0xCF,
        0x5E, 0x74, 0x03, 0x4E, 0xA6, 0x14, 0xBF, 0x70,
        0xC2, 0x76, 0x4B, 0x20, 0xC9, 0x58, 0x9C, 0xDB,
        0x5C, 0x25, 0x76, 0x1A, 0x04, 0xE5, 0x82, 0x92,
        0x90, 0x7C, 0x57, 0x8A, 0x94, 0xA3, 0x58, 0x36,
        0xBE, 0xE3, 0x11, 0x2D, 0xC2, 0xC3, 0xAE, 0x21,
        0x92, 0xC9, 0xDE, 0xAA, 0x30, 0x4B, 0x29, 0xC7,
        0xFE, 0xA1, 0xBD, 0xF4, 0x7B, 0x3B, 0x6B, 0xCB,
        0xA2, 0xC0, 0xE5, 0x5C, 0x9C, 0xDB, 0x6D, 0xE7,
        0x14, 0x9E, 0x9C, 0xB1, 0x79, 0x17, 0x71, 0x8F,
        0x12, 0xC8, 0x03, 0x2D, 0xE1, 0xAD, 0xE0, 0x64,
        0x8D, 0x40, 0x55, 0x19, 0xC7, 0x07, 0x19, 0xBE,
        0xCC, 0x70, 0x18, 0x45, 0xCF, 0x9F, 0x4B, 0x91,
        0x2F, 0xE7, 0x19, 0x83, 0xCA, 0x34, 0xF9, 0x01,
        0x8C, 0x7C, 0xA7, 0xBB, 0x2F, 0x6C, 0x5D, 0x7F,
        0x8C, 0x5B, 0x29, 0x73, 0x59, 0xEC, 0x75, 0x20,
        0x9C, 0x25, 0x43, 0xFF, 0x11, 0xC4, 0x24, 0x49,
        0x77, 0xC5, 0x96, 0x95, 0x24, 0xEC, 0x45, 0x4D,
        0x44, 0xC3, 0x23, 0xFC, 0xCA, 0x94, 0xAC, 0xAC,
        0x27, 0x3A, 0x0E, 0xC4, 0x9B, 0x4A, 0x8A, 0x58,
        0x5B, 0xCE, 0x7A, 0x5B, 0x30, 0x5C, 0x04, 0xC3,
        0x50, 0x64, 0x22, 0x58, 0x03, 0x57, 0x01, 0x6A,
        0x85, 0x0C, 0x3F, 0x7E, 0xE1, 0x72, 0x05, 0xA7,
        0x7B, 0x29, 0x1C, 0x77, 0x31, 0xC9, 0x83, 0x6C,
        0x02, 0xAE, 0xE5, 0x40, 0x6F, 0x63, 0xC6, 0xA0,
        0x7A, 0x21, 0x43, 0x82, 0xAA, 0x15, 0x33, 0x6C,
        0x05, 0xD1, 0x04, 0x55, 0x88, 0x10, 0x76, 0x45,
        0xEA, 0x7D, 0xE6, 0x87, 0x0F, 0xC0, 0xE5, 0x5E,
        0x15, 0x40, 0x97, 0x43, 0x01, 0xC4, 0x2E, 0xC1,
        0x41, 0x05, 0x51, 0x86, 0x80, 0xF6, 0x88, 0xAB,
        0xE4, 0xCE, 0x45, 0x37, 0x38, 0xFE, 0x47, 0x1B,
        0x87, 0xFC, 0x31, 0xF5, 0xC6, 0x8A, 0x39, 0xE6,
        0x8A, 0xF5, 0x1B, 0x02, 0x40, 0xB9, 0x0E, 0x03,
        0x64, 0xB0, 0x4B, 0xAC, 0x43, 0xD6, 0xFB, 0x68,
        0xAB, 0x65, 0xAE, 0x02, 0x8B, 0x62, 0xBD, 0x68,
        0x3B, 0x7D, 0x28, 0xAD, 0x38, 0x80, 0x6B, 0xEE,
        0x72, 0x5B, 0x5B, 0x24, 0x16, 0xA8, 0xD7, 0x9C,
        0x16, 0xEC, 0x2A, 0x99, 0xEA, 0x4A, 0x8D, 0x92,
        0xA2, 0xF5, 0x05, 0x2E, 0x67, 0xF9, 0x73, 0x52,
        0x28, 0x97, 0x61, 0xC5, 0xC3, 0x9F, 0xC5, 0xC7,
        0x42, 0xE9, 0xC0, 0xA7, 0x40, 0xCA, 0x59, 0xFC,
        0x01, 0x82, 0xF7, 0x09, 0xD0, 0x1B, 0x51, 0x87,
        0xF0, 0x00, 0x63, 0xDA, 0xAB, 0x39, 0x75, 0x96,
        0xEE, 0xA4, 0xA3, 0x1B, 0xDB, 0xCB, 0xD4, 0xC1,
        0xBB, 0x0C, 0x55, 0xBE, 0x7C, 0x68, 0x50, 0xFD,
        0xA9, 0x32, 0x6B, 0x35, 0x3E, 0x28, 0x8C, 0x50,
        0x13, 0x22, 0x6C, 0x3C, 0x39, 0x23, 0xA7, 0x91,
        0x60, 0x9E, 0x80, 0x02, 0xE7, 0x3A, 0x5F, 0x7B,
        0x6B, 0xB4, 0xA8, 0x77, 0xB1, 0xFD, 0xF5, 0x3B,
        0xB2, 0xBA, 0xB3, 0xDD, 0x42, 0x4D, 0x31, 0xBB,
        0xB4, 0x48, 0xE6, 0x09, 0xA6, 0x6B, 0x0E, 0x34,
        0x3C, 0x28, 0x6E, 0x87, 0x60, 0x31, 0x2B, 0x6D,
        0x37, 0xAA, 0x52, 0x01, 0xD2, 0x1F, 0x53, 0x50,
        0x3D, 0x88, 0x38, 0x9A, 0xDC, 0xA2, 0x1C, 0x70,
        0xFB, 0x6C, 0x0F, 0xC9, 0xC6, 0x9D, 0x66, 0x16,
        0xC9, 0xEA, 0x37, 0x80, 0xE3, 0x55, 0x65, 0xC0,
        0xC9, 0x7C, 0x15, 0x17, 0x9C, 0x95, 0x34, 0x3E,
        0xCC, 0x5E, 0x1C, 0x2A, 0x24, 0xDE, 0x46, 0x99,
        0xF6, 0x87, 0x5E, 0xA2, 0xFA, 0x2D, 0xD3, 0xE3,
        0x57, 0xBC, 0x43, 0x91, 0x47, 0x95, 0x20, 0x7E,
        0x02, 0x6B, 0x85, 0x0A, 0x22, 0x37, 0x95, 0x0C,
        0x10, 0x8A, 0x51, 0x2F, 0xC8, 0x8C, 0x22, 0x48,
        0x81, 0x12, 0x60, 0x70, 0x88, 0x18, 0x5F, 0xB0,
        0xE0, 0x9C, 0x2C, 0x41, 0x97, 0xA8, 0x36, 0x87,
        0x26, 0x6B, 0xAB, 0x2E, 0x58, 0x3E, 0x21, 0xC4,
        0x0F, 0x4C, 0xC0, 0x08, 0xFE, 0x65, 0x28, 0x04,
        0xD8, 0x22, 0x3F, 0x15, 0x20, 0xA9, 0x0B, 0x0D,
        0x53, 0x85, 0xC7, 0x55, 0x3C, 0xC7, 0x67, 0xC5,
        0x8D, 0x12, 0x0C, 0xCD, 0x3E, 0xF5, 0xB5, 0xD1,
        0xA6, 0xCD, 0x7B, 0xC0, 0x0D, 0xFF, 0x13, 0x21,
        0xB2, 0xF2, 0xC4, 0x32, 0xB6, 0x4E, 0xFB, 0x8A,
        0x3F, 0x5D, 0x00, 0x64, 0xB3, 0xF3, 0x42, 0x93,
        0x02, 0x6C, 0x85, 0x1C, 0x2D, 0xED, 0x68, 0xB9,
        0xDF, 0xF4, 0xA2, 0x8F, 0x6A, 0x8D, 0x22, 0x55,
        0x35, 0xE0, 0x47, 0x70, 0x84, 0x43, 0x0C, 0xFF,
        0xDA, 0x0A, 0xC0, 0x55, 0x2F, 0x9A, 0x21, 0x27,
        0x85, 0xB7, 0x49, 0x91, 0x3A, 0x06, 0xFA, 0x22,
        0x74, 0xC0, 0xD1, 0x5B, 0xAD, 0x32, 0x54, 0x58,
        0xD3, 0x23, 0xEF, 0x6B, 0xAE, 0x13, 0xC0, 0x01,
        0x0D, 0x52, 0x5C, 0x1D, 0x52, 0x69, 0x97, 0x3A,
        0xC2, 0x9B, 0xDA, 0x7C, 0x98, 0x37, 0x46, 0x91,
        0x8B, 0xA0, 0xE0, 0x02, 0x58, 0x8E, 0x30, 0x37,
        0x5D, 0x78, 0x32, 0x9E, 0x6B, 0x8B, 0xA8, 0xC4,
        0x46, 0x2A, 0x69, 0x2F, 0xB6, 0x08, 0x38, 0x42,
        0xB8, 0xC8, 0xC9, 0x2C, 0x60, 0xF2, 0x52, 0x72,
        0x6D, 0x14, 0xA0, 0x71, 0xF7, 0xCC, 0x45, 0x25,
        0x58, 0xD5, 0xE7, 0x1A, 0x7B, 0x08, 0x70, 0x62,
        0xEC, 0xB1, 0x38, 0x68, 0x44, 0x58, 0x82, 0x46,
        0x12, 0x64, 0x02, 0xB1, 0xFA, 0x16, 0x37, 0x73,
        0x3C, 0xD5, 0xF6, 0x0C, 0xC8, 0x4B, 0xCB, 0x64,
        0x6A, 0x78, 0x92, 0x61, 0x4D, 0x7C, 0x51, 0xB1,
        0xC7, 0xF1, 0xA2, 0x79, 0x91, 0x32, 0xF1, 0x34,
        0x27, 0xDC, 0x48, 0x21, 0x58, 0xDA, 0x25, 0x44,
        0x70, 0xA5, 0x9E, 0x00, 0xA4, 0xE4, 0x96, 0x86,
        0xFD, 0xC0, 0x77, 0x55, 0x93, 0x67, 0x27, 0x0C,
        0x21, 0x53, 0xF1, 0x10, 0x07, 0x59, 0x2C, 0x9C,
        0x43, 0x10, 0xCF, 0x8A, 0x12, 0xC6, 0xA8, 0x71,
        0x3B, 0xD6, 0xBB, 0x51, 0xF3, 0x12, 0x4F, 0x98,
        0x9B, 0xA0, 0xD5, 0x40, 0x73, 0xCC, 0x24, 0x2E,
        0x09, 0x68, 0x78, 0x0B, 0x87, 0x5A, 0x86, 0x9E,
        0xFB, 0x85, 0x15, 0x86, 0xB9, 0xA8, 0x68, 0xA3,
        0x84, 0xB9, 0xE6, 0x82, 0x1B, 0x20, 0x1B, 0x93,
        0x2C, 0x45, 0x53, 0x69, 0xA7, 0x39, 0xEC, 0x22,
        0x56, 0x9C, 0x97, 0x7C, 0x21, 0x2B, 0x38, 0x18,
        0x71, 0x81, 0x36, 0x56, 0xAF, 0x5B, 0x56, 0x7E,
        0xF8, 0x93, 0xB5, 0x84, 0x62, 0x4C, 0x86, 0x3A,
        0x25, 0x90, 0x00, 0xF1, 0x7B, 0x25, 0x4B, 0x98,
        0xB1, 0x85, 0x09, 0x7C, 0x50, 0xEB, 0xB6, 0x8B,
        0x24, 0x43, 0x42, 0xE0, 0x5D, 0x4D, 0xE5, 0x20,
        0x12, 0x5B, 0x8E, 0x10, 0x33, 0xB1, 0x43, 0x60,
        0x93, 0xAC, 0xE7, 0xCE, 0x8E, 0x71, 0xB4, 0x58,
        0xD5, 0x25, 0x67, 0x33, 0x63, 0x04, 0x5A, 0x3B,
        0x3E, 0xEA, 0x94, 0x55, 0x42, 0x8A, 0x39, 0x87,
        0x05, 0xA4, 0x23, 0x27, 0xAD, 0xB3, 0x77, 0x4B,
        0x70, 0x57, 0xF4, 0x2B, 0x01, 0x7E, 0xC0, 0x73,
        0x9A, 0x98, 0x3F, 0x19, 0xE8, 0x21, 0x4D, 0x09,
        0x19, 0x5F, 0xA2, 0x4D, 0x2D, 0x57, 0x1D, 0xB7,
        0x3C, 0x19, 0xA6, 0xF8, 0x46, 0x0E, 0x50, 0x83,
        0x0D, 0x41, 0x5F, 0x62, 0x7B, 0x88, 0xE9, 0x4A,
        0x7B, 0x15, 0x37, 0x91, 0xA0, 0xC0, 0xC7, 0xE9,
        0x48, 0x4C, 0x74, 0xD5, 0x3C, 0x71, 0x48, 0x89,
        0xF0, 0xE3, 0x21, 0xB6, 0x66, 0x0A, 0x53, 0x2A,
        0x5B, 0xC0, 0xE5, 0x57, 0xFB, 0xCA, 0x35, 0xE2,
        0x9B, 0xC6, 0x11, 0x20, 0x0E, 0xD3, 0xC6, 0x33,
        0x07, 0x7A, 0x4D, 0x87, 0x3C, 0x5C, 0xC6, 0x70,
        0x06, 0xB7, 0x53, 0xBF, 0x6D, 0x6B, 0x7A, 0xF6,
        0xCA, 0x40, 0x2A, 0xB6, 0x18, 0x23, 0x6C, 0x0A,
        0xFF, 0xBC, 0x80, 0x1F, 0x82, 0x22, 0xFB, 0xC3,
        0x6C, 0xE0, 0x98, 0x4E, 0x2B, 0x18, 0xC9, 0x44,
        0xBB, 0xCB, 0xEF, 0x03, 0xB1, 0xE1, 0x36, 0x1C,
        0x1F, 0x44, 0xB0, 0xD7, 0x34, 0xAF, 0xB1, 0x56,
        0x6C, 0xFF, 0x87, 0x44, 0xDA, 0x8B, 0x99, 0x43,
        0xD6, 0xB4, 0x5A, 0x3C, 0x09, 0x03, 0x07, 0x02,
        0xCA, 0x20, 0x1F, 0xFE, 0x20, 0xCB, 0x7E, 0xC5,
        0xB0, 0xD4, 0x14, 0x9E, 0xE2, 0xC2, 0x8E, 0x8B,
        0x23, 0x37, 0x4F, 0x47, 0x1B, 0x57, 0x15, 0x0D,
        0x0E, 0xC9, 0x33, 0x62, 0x61, 0xA2, 0xD5, 0xCB,
        0x84, 0xA3, 0xAC, 0xAC, 0xC4, 0x28, 0x94, 0x73,
        0xA4, 0xC0, 0xAB, 0xC6, 0x17, 0xC9, 0xAB, 0xC1,
        0x78, 0x73, 0x44, 0x34, 0xC8, 0x2E, 0x16, 0x85,
        0x58, 0x8A, 0x5C, 0x2E, 0xA2, 0x67, 0x8F, 0x6B,
        0x3C, 0x22, 0x28, 0x73, 0x31, 0x30, 0xC4, 0x66,
        0xE5, 0xB8, 0x6E, 0xF4, 0x91, 0x15, 0x3E, 0x48,
        0x66, 0x22, 0x47, 0xB8, 0x75, 0xD2, 0x01, 0x02,
        0x0B, 0x56, 0x6B, 0x81, 0xB6, 0x4D, 0x83, 0x9A,
        0xB4, 0x63, 0x3B, 0xAA, 0x8A, 0xCE, 0x20, 0x2B,
        0xAA, 0xB4, 0x49, 0x62, 0x97, 0xF9, 0x80, 0x7A,
        0xDB, 0xBB, 0x1E, 0x33, 0x2C, 0x6F, 0x80, 0x22,
        0xB2, 0xA1, 0x8C, 0xFD, 0xD4, 0xA8, 0x25, 0x30,
        0xB6, 0xD3, 0xF0, 0x07, 0xC3, 0x35, 0x38, 0x98,
        0xD9, 0x66, 0xCC, 0x2C, 0x21, 0xCB, 0x42, 0x44,
        0xBD, 0x00, 0x44, 0x3F, 0x20, 0x98, 0x70, 0xAC,
        0xC4, 0x2B, 0xC3, 0x30, 0x68, 0xC7, 0x24, 0xEC,
        0x17, 0x22, 0x36, 0x19, 0xC1, 0x09, 0x3C, 0xCA,
        0x6A, 0xEB, 0x29, 0x50, 0x06, 0x64, 0xD1, 0x22,
        0x50, 0x36, 0xB4, 0xB8, 0x10, 0x91, 0x90, 0x69,
        0x69, 0x48, 0x1F, 0x1C, 0x72, 0x3C, 0x14, 0x0B,
        0x9D, 0x6C, 0x16, 0x8F, 0x5B, 0x64, 0xBE, 0xA6,
        0x9C, 0x5F, 0xD6, 0x38, 0x5D, 0xF7, 0x36, 0x4B,
        0x87, 0x23, 0xBC, 0xC8, 0x5E, 0x03, 0x8C, 0x7E,
        0x46, 0x4A, 0x90, 0x0D, 0x68, 0xA2, 0x12, 0x78,
        0x18, 0x99, 0x42, 0x17, 0xAE, 0xC8, 0xBD, 0xB3,
        0x9A, 0x97, 0x0A, 0x99, 0x63, 0xDE, 0x93, 0x68,
        0x8E, 0x2A, 0xC8, 0x2A, 0xBC, 0xC2, 0x2F, 0xB9,
        0x27, 0x7B, 0xA2, 0x20, 0x09, 0xE8, 0x78, 0x38,
        0x1A, 0x38, 0x16, 0x39, 0x01, 0xC7, 0xD4, 0xC8,
        0x50, 0x19, 0x53, 0x8D, 0x35, 0xCA, 0xAE, 0x9C,
        0x41, 0xAF, 0x8C, 0x92, 0x9E, 0xE2, 0x0B, 0xB0,
        0x8C, 0xA6, 0x19, 0xE7, 0x2C, 0x2F, 0x22, 0x62,
        0xC1, 0xC9, 0x93, 0x85, 0x72, 0x55, 0x1A, 0xC0,
        0x2D, 0xC9, 0x26, 0x8F, 0xBC, 0xC3, 0x5D, 0x79,
        0x01, 0x1C, 0x3C, 0x09, 0x0A, 0xD4, 0x0A, 0x4F,
        0x11, 0x1C, 0x9B, 0xE5, 0x5C, 0x42, 0x7E, 0xB7,
        0x96, 0xC1, 0x93, 0x2D, 0x86, 0x73, 0x57, 0x9A,
        0xF1, 0xB4, 0xC6, 0x38, 0xB0, 0x94, 0x44, 0x89,
        0x01, 0x2A, 0x25, 0x59, 0xA3, 0xB0, 0x24, 0x81,
        0xB0, 0x1A, 0xC3, 0x0B, 0xA8, 0x96, 0x0F, 0x80,
        0xC0, 0xC2, 0xB3, 0x94, 0x7D, 0x36, 0xA1, 0x2C,
        0x08, 0x04, 0x98, 0xBE, 0xE4, 0x48, 0x71, 0x6C,
        0x97, 0x34, 0x16, 0xC8, 0x24, 0x28, 0x04, 0xA3,
        0xDA, 0x09, 0x9E, 0xE1, 0x37, 0xB0, 0xBA, 0x90,
        0xFE, 0x4A, 0x5C, 0x6A, 0x89, 0x20, 0x02, 0x76,
        0xA0, 0xCF, 0xB6, 0x43, 0xEC, 0x2C, 0x56, 0xA2,
        0xD7, 0x08, 0xD7, 0xB4, 0x37, 0x3E, 0x44, 0xC1,
        0x50, 0x2A, 0x76, 0x3A, 0x60, 0x05, 0x86, 0xE6,
        0xCD, 0xA6, 0x27, 0x38, 0x97, 0xD4, 0x44, 0x48,
        0x28, 0x7D, 0xC2, 0xE6, 0x02, 0xDC, 0x39, 0x20,
        0x0B, 0xF6, 0x16, 0x62, 0x36, 0x55, 0x9F, 0xD1,
        0x2A, 0x60, 0x89, 0x2A, 0xEB, 0x15, 0x3D, 0xD6,
        0x51, 0xBB, 0x46, 0x99, 0x10, 0xB4, 0xB3, 0x46,
        0x69, 0xF9, 0x1D, 0xA8, 0x65, 0x4D, 0x1E, 0xB7,
        0x2E, 0xB6, 0xE0, 0x28, 0x00, 0xB3, 0xB0, 0xA7,
        0xD0, 0xA4, 0x8C, 0x83, 0x68, 0x54, 0xD3, 0xA8,
        0x3E, 0x65, 0x56, 0x9C, 0xB7, 0x23, 0x0B, 0xB4,
        0x4F, 0x3F, 0x14, 0x3A, 0x6D, 0xEC, 0x5F, 0x2C,
        0x39, 0xAB, 0x90, 0xF2, 0x74, 0xF2, 0x08, 0x8B,
        0xD3, 0xD6, 0xA6, 0xFC, 0xA0, 0x07, 0x02, 0x73,
        0xBE, 0xDC, 0x84, 0x77, 0x7F, 0xB5, 0x2E, 0x3C,
        0x55, 0x8B, 0x0A, 0xE0, 0x61, 0x83, 0xD5, 0xA4,
        0x8D, 0x45, 0x2F, 0x68, 0xE1, 0x52, 0x07, 0xF8,
        0x61, 0x62, 0x7A, 0xCA, 0x14, 0x27, 0x96, 0x30,
        0xF8, 0x2E, 0xC3, 0xA0, 0xCA, 0x07, 0x86, 0x33,
        0xB6, 0x00, 0xAF, 0xA7, 0x97, 0x43, 0xA6, 0x00,
        0x21, 0x5B, 0xE5, 0x63, 0x74, 0x58, 0xCE, 0x2C,
        0xE8, 0xAF, 0xF5, 0xA0, 0x8E, 0xB5, 0x01, 0x7B,
        0x2C, 0x76, 0x65, 0x77, 0x47, 0x9F, 0x8D, 0xC6,
        0xBF, 0x9F, 0x5C, 0xC7, 0x50, 0x89, 0x93, 0x21,
        0x61, 0xB9, 0x6C, 0xEA, 0x40, 0x66, 0x20, 0xAE,
        0xDB, 0x63, 0x04, 0x07, 0xF7, 0x68, 0x7E, 0xBB,
        0xB4, 0x81, 0x4C, 0x79, 0x81, 0x63, 0x7A, 0x48,
        0xA9, 0x0D, 0xE6, 0x80, 0x31, 0xE0, 0x62, 0xA7,
        0xAF, 0x76, 0x12, 0xB4, 0xF5, 0xC7, 0xA6, 0xDA,
        0x86, 0xBD, 0x13, 0x65, 0x29, 0xE6, 0x42, 0x95,
        0xA5, 0x61, 0x3E, 0xA7, 0x3B, 0xD3, 0xD4, 0x44,
        0x8C, 0xB8, 0x1F, 0x24, 0x31, 0x35, 0xC0, 0xA6,
        0x60, 0xBE, 0xB9, 0xC1, 0x7E, 0x65, 0x1D, 0xEF,
        0x46, 0x9A, 0x7D, 0x90, 0xA1, 0x5D, 0x34, 0x81,
        0x09, 0x0B, 0xCB, 0xF2, 0x27, 0x01, 0x23, 0x28,
        0x94, 0x1F, 0xA4, 0x6F, 0x39, 0xC5, 0x00, 0x6A,
        0xD9, 0x3D, 0x45, 0x8A, 0xA6, 0xAD, 0xD6, 0x55,
        0x86, 0x2B, 0x41, 0x8C, 0x30, 0x94, 0xF5, 0x51,
        0x46, 0x0D, 0xF2, 0x15, 0x3A, 0x58, 0x10, 0xA7,
        0xDA, 0x74, 0xF0, 0x61, 0x4C, 0x25, 0x88, 0xBE,
        0x49, 0xDC, 0x6F, 0x5E, 0x88, 0x15, 0x46, 0x42,
        0xBD, 0x1D, 0x37, 0x62, 0x56, 0x33, 0x26, 0x43,
        0x35, 0x07, 0x15, 0x6A, 0x57, 0xC5, 0x76, 0x94,
        0xBD, 0xD2, 0x6E, 0x7A, 0x24, 0x6F, 0xEB, 0x72,
        0x3A, 0xED, 0x67, 0xB0, 0x48, 0x87, 0xC8, 0xE4,
        0x76, 0xB4, 0x8C, 0xAB, 0x59, 0xE5, 0x36, 0x2F,
        0x26, 0xA9, 0xEF, 0x50, 0xC2, 0xBC, 0x80, 0xBA,
        0x14, 0x62, 0x26, 0x21, 0x6F, 0xE6, 0x29, 0x68,
        0xA6, 0x0D, 0x04, 0xE8, 0xC1, 0x70, 0xD7, 0x41,
        0xC7, 0xA2, 0xB0, 0xE1, 0xAB, 0xDA, 0xC9, 0x68,
        0xE2, 0x90, 0x20, 0x83, 0x9D, 0x05, 0x2F, 0xA3,
        0x72, 0x58, 0x56, 0x27, 0xF8, 0xB5, 0x9E, 0xE3,
        0x12, 0xAE, 0x41, 0x4C, 0x97, 0x9D, 0x82, 0x5F,
        0x06, 0xA6, 0x92, 0x9A, 0x79, 0x62, 0x57, 0x18,
        0xA8, 0x57, 0x68, 0xF3, 0x48, 0x6B, 0xD3, 0x2A,
        0x01, 0xBF, 0x9A, 0x8F, 0x21, 0xEA, 0x93, 0x8E,
        0x64, 0x8E, 0xAE, 0x4E, 0x54, 0x48, 0xC3, 0x4C,
        0x3E, 0xB8, 0x88, 0x20, 0xB1, 0x59, 0xEE, 0xDD
    };
#endif
#ifndef WOLFSSL_NO_ML_KEM_1024
    static const byte seed_1024[WC_ML_KEM_MAKEKEY_RAND_SZ] = {
        /* d */
        0x49, 0xAC, 0x8B, 0x99, 0xBB, 0x1E, 0x6A, 0x8E,
        0xA8, 0x18, 0x26, 0x1F, 0x8B, 0xE6, 0x8B, 0xDE,
        0xAA, 0x52, 0x89, 0x7E, 0x7E, 0xC6, 0xC4, 0x0B,
        0x53, 0x0B, 0xC7, 0x60, 0xAB, 0x77, 0xDC, 0xE3,
        /* z */
        0x99, 0xE3, 0x24, 0x68, 0x84, 0x18, 0x1F, 0x8E,
        0x1D, 0xD4, 0x4E, 0x0C, 0x76, 0x29, 0x09, 0x33,
        0x30, 0x22, 0x1F, 0xD6, 0x7D, 0x9B, 0x7D, 0x6E,
        0x15, 0x10, 0xB2, 0xDB, 0xAD, 0x87, 0x62, 0xF7
    };
    static const byte ek_1024[WC_ML_KEM_1024_PUBLIC_KEY_SIZE] = {
        0xA0, 0x41, 0x84, 0xD4, 0xBC, 0x7B, 0x53, 0x2A,
        0x0F, 0x70, 0xA5, 0x4D, 0x77, 0x57, 0xCD, 0xE6,
        0x17, 0x5A, 0x68, 0x43, 0xB8, 0x61, 0xCB, 0x2B,
        0xC4, 0x83, 0x0C, 0x00, 0x12, 0x55, 0x4C, 0xFC,
        0x5D, 0x2C, 0x8A, 0x20, 0x27, 0xAA, 0x3C, 0xD9,
        0x67, 0x13, 0x0E, 0x9B, 0x96, 0x24, 0x1B, 0x11,
        0xC4, 0x32, 0x0C, 0x76, 0x49, 0xCC, 0x23, 0xA7,
        0x1B, 0xAF, 0xE6, 0x91, 0xAF, 0xC0, 0x8E, 0x68,
        0x0B, 0xCE, 0xF4, 0x29, 0x07, 0x00, 0x07, 0x18,
        0xE4, 0xEA, 0xCE, 0x8D, 0xA2, 0x82, 0x14, 0x19,
        0x7B, 0xE1, 0xC2, 0x69, 0xDA, 0x9C, 0xB5, 0x41,
        0xE1, 0xA3, 0xCE, 0x97, 0xCF, 0xAD, 0xF9, 0xC6,
        0x05, 0x87, 0x80, 0xFE, 0x67, 0x93, 0xDB, 0xFA,
        0x82, 0x18, 0xA2, 0x76, 0x0B, 0x80, 0x2B, 0x8D,
        0xA2, 0xAA, 0x27, 0x1A, 0x38, 0x77, 0x25, 0x23,
        0xA7, 0x67, 0x36, 0xA7, 0xA3, 0x1B, 0x9D, 0x30,
        0x37, 0xAD, 0x21, 0xCE, 0xBB, 0x11, 0xA4, 0x72,
        0xB8, 0x79, 0x2E, 0xB1, 0x75, 0x58, 0xB9, 0x40,
        0xE7, 0x08, 0x83, 0xF2, 0x64, 0x59, 0x2C, 0x68,
        0x9B, 0x24, 0x0B, 0xB4, 0x3D, 0x54, 0x08, 0xBF,
        0x44, 0x64, 0x32, 0xF4, 0x12, 0xF4, 0xB9, 0xA5,
        0xF6, 0x86, 0x5C, 0xC2, 0x52, 0xA4, 0x3C, 0xF4,
        0x0A, 0x32, 0x03, 0x91, 0x55, 0x55, 0x91, 0xD6,
        0x75, 0x61, 0xFD, 0xD0, 0x53, 0x53, 0xAB, 0x6B,
        0x01, 0x9B, 0x3A, 0x08, 0xA7, 0x33, 0x53, 0xD5,
        0x1B, 0x61, 0x13, 0xAB, 0x2F, 0xA5, 0x1D, 0x97,
        0x56, 0x48, 0xEE, 0x25, 0x4A, 0xF8, 0x9A, 0x23,
        0x05, 0x04, 0xA2, 0x36, 0xA4, 0x65, 0x82, 0x57,
        0x74, 0x0B, 0xDC, 0xBB, 0xE1, 0x70, 0x8A, 0xB0,
        0x22, 0xC3, 0xC5, 0x88, 0xA4, 0x10, 0xDB, 0x3B,
        0x9C, 0x30, 0x8A, 0x06, 0x27, 0x5B, 0xDF, 0x5B,
        0x48, 0x59, 0xD3, 0xA2, 0x61, 0x7A, 0x29, 0x5E,
        0x1A, 0x22, 0xF9, 0x01, 0x98, 0xBA, 0xD0, 0x16,
        0x6F, 0x4A, 0x94, 0x34, 0x17, 0xC5, 0xB8, 0x31,
        0x73, 0x6C, 0xB2, 0xC8, 0x58, 0x0A, 0xBF, 0xDE,
        0x57, 0x14, 0xB5, 0x86, 0xAB, 0xEE, 0xC0, 0xA1,
        0x75, 0xA0, 0x8B, 0xC7, 0x10, 0xC7, 0xA2, 0x89,
        0x5D, 0xE9, 0x3A, 0xC4, 0x38, 0x06, 0x1B, 0xF7,
        0x76, 0x5D, 0x0D, 0x21, 0xCD, 0x41, 0x81, 0x67,
        0xCA, 0xF8, 0x9D, 0x1E, 0xFC, 0x34, 0x48, 0xBC,
        0xBB, 0x96, 0xD6, 0x9B, 0x3E, 0x01, 0x0C, 0x82,
        0xD1, 0x5C, 0xAB, 0x6C, 0xAC, 0xC6, 0x79, 0x9D,
        0x36, 0x39, 0x66, 0x9A, 0x5B, 0x21, 0xA6, 0x33,
        0xC8, 0x65, 0xF8, 0x59, 0x3B, 0x5B, 0x7B, 0xC8,
        0x00, 0x26, 0x2B, 0xB8, 0x37, 0xA9, 0x24, 0xA6,
        0xC5, 0x44, 0x0E, 0x4F, 0xC7, 0x3B, 0x41, 0xB2,
        0x30, 0x92, 0xC3, 0x91, 0x2F, 0x4C, 0x6B, 0xEB,
        0xB4, 0xC7, 0xB4, 0xC6, 0x29, 0x08, 0xB0, 0x37,
        0x75, 0x66, 0x6C, 0x22, 0x22, 0x0D, 0xF9, 0xC8,
        0x88, 0x23, 0xE3, 0x44, 0xC7, 0x30, 0x83, 0x32,
        0x34, 0x5C, 0x8B, 0x79, 0x5D, 0x34, 0xE8, 0xC0,
        0x51, 0xF2, 0x1F, 0x5A, 0x21, 0xC2, 0x14, 0xB6,
        0x98, 0x41, 0x35, 0x87, 0x09, 0xB1, 0xC3, 0x05,
        0xB3, 0x2C, 0xC2, 0xC3, 0x80, 0x6A, 0xE9, 0xCC,
        0xD3, 0x81, 0x9F, 0xFF, 0x45, 0x07, 0xFE, 0x52,
        0x0F, 0xBF, 0xC2, 0x71, 0x99, 0xBC, 0x23, 0xBE,
        0x6B, 0x9B, 0x2D, 0x2A, 0xC1, 0x71, 0x75, 0x79,
        0xAC, 0x76, 0x92, 0x79, 0xE2, 0xA7, 0xAA, 0xC6,
        0x8A, 0x37, 0x1A, 0x47, 0xBA, 0x3A, 0x7D, 0xBE,
        0x01, 0x6F, 0x14, 0xE1, 0xA7, 0x27, 0x33, 0x36,
        0x63, 0xC4, 0xA5, 0xCD, 0x1A, 0x0F, 0x88, 0x36,
        0xCF, 0x7B, 0x5C, 0x49, 0xAC, 0x51, 0x48, 0x5C,
        0xA6, 0x03, 0x45, 0xC9, 0x90, 0xE0, 0x68, 0x88,
        0x72, 0x00, 0x03, 0x73, 0x13, 0x22, 0xC5, 0xB8,
        0xCD, 0x5E, 0x69, 0x07, 0xFD, 0xA1, 0x15, 0x7F,
        0x46, 0x8F, 0xD3, 0xFC, 0x20, 0xFA, 0x81, 0x75,
        0xEE, 0xC9, 0x5C, 0x29, 0x1A, 0x26, 0x2B, 0xA8,
        0xC5, 0xBE, 0x99, 0x08, 0x72, 0x41, 0x89, 0x30,
        0x85, 0x23, 0x39, 0xD8, 0x8A, 0x19, 0xB3, 0x7F,
        0xEF, 0xA3, 0xCF, 0xE8, 0x21, 0x75, 0xC2, 0x24,
        0x40, 0x7C, 0xA4, 0x14, 0xBA, 0xEB, 0x37, 0x92,
        0x3B, 0x4D, 0x2D, 0x83, 0x13, 0x4A, 0xE1, 0x54,
        0xE4, 0x90, 0xA9, 0xB4, 0x5A, 0x05, 0x63, 0xB0,
        0x6C, 0x95, 0x3C, 0x33, 0x01, 0x45, 0x0A, 0x21,
        0x76, 0xA0, 0x7C, 0x61, 0x4A, 0x74, 0xE3, 0x47,
        0x8E, 0x48, 0x50, 0x9F, 0x9A, 0x60, 0xAE, 0x94,
        0x5A, 0x8E, 0xBC, 0x78, 0x15, 0x12, 0x1D, 0x90,
        0xA3, 0xB0, 0xE0, 0x70, 0x91, 0xA0, 0x96, 0xCF,
        0x02, 0xC5, 0x7B, 0x25, 0xBC, 0xA5, 0x81, 0x26,
        0xAD, 0x0C, 0x62, 0x9C, 0xE1, 0x66, 0xA7, 0xED,
        0xB4, 0xB3, 0x32, 0x21, 0xA0, 0xD3, 0xF7, 0x2B,
        0x85, 0xD5, 0x62, 0xEC, 0x69, 0x8B, 0x7D, 0x0A,
        0x91, 0x3D, 0x73, 0x80, 0x6F, 0x1C, 0x5C, 0x87,
        0xB3, 0x8E, 0xC0, 0x03, 0xCB, 0x30, 0x3A, 0x3D,
        0xC5, 0x1B, 0x4B, 0x35, 0x35, 0x6A, 0x67, 0x82,
        0x6D, 0x6E, 0xDA, 0xA8, 0xFE, 0xB9, 0x3B, 0x98,
        0x49, 0x3B, 0x2D, 0x1C, 0x11, 0xB6, 0x76, 0xA6,
        0xAD, 0x95, 0x06, 0xA1, 0xAA, 0xAE, 0x13, 0xA8,
        0x24, 0xC7, 0xC0, 0x8D, 0x1C, 0x6C, 0x2C, 0x4D,
        0xBA, 0x96, 0x42, 0xC7, 0x6E, 0xA7, 0xF6, 0xC8,
        0x26, 0x4B, 0x64, 0xA2, 0x3C, 0xCC, 0xA9, 0xA7,
        0x46, 0x35, 0xFC, 0xBF, 0x03, 0xE0, 0x0F, 0x1B,
        0x57, 0x22, 0xB2, 0x14, 0x37, 0x67, 0x90, 0x79,
        0x3B, 0x2C, 0x4F, 0x0A, 0x13, 0xB5, 0xC4, 0x07,
        0x60, 0xB4, 0x21, 0x8E, 0x1D, 0x25, 0x94, 0xDC,
        0xB3, 0x0A, 0x70, 0xD9, 0xC1, 0x78, 0x2A, 0x5D,
        0xD3, 0x05, 0x76, 0xFA, 0x41, 0x44, 0xBF, 0xC8,
        0x41, 0x6E, 0xDA, 0x81, 0x18, 0xFC, 0x64, 0x72,
        0xF5, 0x6A, 0x97, 0x95, 0x86, 0xF3, 0x3B, 0xB0,
        0x70, 0xFB, 0x0F, 0x1B, 0x0B, 0x10, 0xBC, 0x48,
        0x97, 0xEB, 0xE0, 0x1B, 0xCA, 0x38, 0x93, 0xD4,
        0xE1, 0x6A, 0xDB, 0x25, 0x09, 0x3A, 0x74, 0x17,
        0xD0, 0x70, 0x8C, 0x83, 0xA2, 0x63, 0x22, 0xE2,
        0x2E, 0x63, 0x30, 0x09, 0x1E, 0x30, 0x15, 0x2B,
        0xF8, 0x23, 0x59, 0x7C, 0x04, 0xCC, 0xF4, 0xCF,
        0xC7, 0x33, 0x15, 0x78, 0xF4, 0x3A, 0x27, 0x26,
        0xCC, 0xB4, 0x28, 0x28, 0x9A, 0x90, 0xC8, 0x63,
        0x25, 0x9D, 0xD1, 0x80, 0xC5, 0xFF, 0x14, 0x2B,
        0xEF, 0x41, 0xC7, 0x71, 0x70, 0x94, 0xBE, 0x07,
        0x85, 0x6D, 0xA2, 0xB1, 0x40, 0xFA, 0x67, 0x71,
        0x09, 0x67, 0x35, 0x6A, 0xA4, 0x7D, 0xFB, 0xC8,
        0xD2, 0x55, 0xB4, 0x72, 0x2A, 0xB8, 0x6D, 0x43,
        0x9B, 0x7E, 0x0A, 0x60, 0x90, 0x25, 0x1D, 0x2D,
        0x4C, 0x1E, 0xD5, 0xF2, 0x0B, 0xBE, 0x68, 0x07,
        0xBF, 0x65, 0xA9, 0x0B, 0x7C, 0xB2, 0xEC, 0x01,
        0x02, 0xAF, 0x02, 0x80, 0x9D, 0xC9, 0xAC, 0x7D,
        0x0A, 0x3A, 0xBC, 0x69, 0xC1, 0x83, 0x65, 0xBC,
        0xFF, 0x59, 0x18, 0x5F, 0x33, 0x99, 0x68, 0x87,
        0x74, 0x61, 0x85, 0x90, 0x6C, 0x01, 0x91, 0xAE,
        0xD4, 0x40, 0x7E, 0x13, 0x94, 0x46, 0x45, 0x9B,
        0xE2, 0x9C, 0x68, 0x22, 0x71, 0x76, 0x44, 0x35,
        0x3D, 0x24, 0xAB, 0x63, 0x39, 0x15, 0x6A, 0x9C,
        0x42, 0x49, 0x09, 0xF0, 0xA9, 0x02, 0x5B, 0xB7,
        0x47, 0x20, 0x77, 0x9B, 0xE4, 0x3F, 0x16, 0xD8,
        0x1C, 0x8C, 0xC6, 0x66, 0xE9, 0x97, 0x10, 0xD8,
        0xC6, 0x8B, 0xB5, 0xCC, 0x4E, 0x12, 0xF3, 0x14,
        0xE9, 0x25, 0xA5, 0x51, 0xF0, 0x9C, 0xC5, 0x90,
        0x03, 0xA1, 0xF8, 0x81, 0x03, 0xC2, 0x54, 0xBB,
        0x97, 0x8D, 0x75, 0xF3, 0x94, 0xD3, 0x54, 0x0E,
        0x31, 0xE7, 0x71, 0xCD, 0xA3, 0x6E, 0x39, 0xEC,
        0x54, 0xA6, 0x2B, 0x58, 0x32, 0x66, 0x4D, 0x82,
        0x1A, 0x72, 0xF1, 0xE6, 0xAF, 0xBB, 0xA2, 0x7F,
        0x84, 0x29, 0x5B, 0x26, 0x94, 0xC4, 0x98, 0x49,
        0x8E, 0x81, 0x2B, 0xC8, 0xE9, 0x37, 0x8F, 0xE5,
        0x41, 0xCE, 0xC5, 0x89, 0x1B, 0x25, 0x06, 0x29,
        0x01, 0xCB, 0x72, 0x12, 0xE3, 0xCD, 0xC4, 0x61,
        0x79, 0xEC, 0x5B, 0xCE, 0xC1, 0x0B, 0xC0, 0xB9,
        0x31, 0x1D, 0xE0, 0x50, 0x74, 0x29, 0x06, 0x87,
        0xFD, 0x6A, 0x53, 0x92, 0x67, 0x16, 0x54, 0x28,
        0x4C, 0xD9, 0xC8, 0xCC, 0x3E, 0xBA, 0x80, 0xEB,
        0x3B, 0x66, 0x2E, 0xB5, 0x3E, 0xB7, 0x51, 0x16,
        0x70, 0x4A, 0x1F, 0xEB, 0x5C, 0x2D, 0x05, 0x63,
        0x38, 0x53, 0x28, 0x68, 0xDD, 0xF2, 0x4E, 0xB8,
        0x99, 0x2A, 0xB8, 0x56, 0x5D, 0x9E, 0x49, 0x0C,
        0xAD, 0xF1, 0x48, 0x04, 0x36, 0x0D, 0xAA, 0x90,
        0x71, 0x8E, 0xAB, 0x61, 0x6B, 0xAB, 0x07, 0x65,
        0xD3, 0x39, 0x87, 0xB4, 0x7E, 0xFB, 0x65, 0x99,
        0xC5, 0x56, 0x32, 0x35, 0xE6, 0x1E, 0x4B, 0xE6,
        0x70, 0xE9, 0x79, 0x55, 0xAB, 0x29, 0x2D, 0x97,
        0x32, 0xCB, 0x89, 0x30, 0x94, 0x8A, 0xC8, 0x2D,
        0xF2, 0x30, 0xAC, 0x72, 0x29, 0x7A, 0x23, 0x67,
        0x9D, 0x6B, 0x94, 0xC1, 0x7F, 0x13, 0x59, 0x48,
        0x32, 0x54, 0xFE, 0xDC, 0x2F, 0x05, 0x81, 0x9F,
        0x0D, 0x06, 0x9A, 0x44, 0x3B, 0x78, 0xE3, 0xFC,
        0x6C, 0x3E, 0xF4, 0x71, 0x4B, 0x05, 0xA3, 0xFC,
        0xA8, 0x1C, 0xBB, 0xA6, 0x02, 0x42, 0xA7, 0x06,
        0x0C, 0xD8, 0x85, 0xD8, 0xF3, 0x99, 0x81, 0xBB,
        0x18, 0x09, 0x2B, 0x23, 0xDA, 0xA5, 0x9F, 0xD9,
        0x57, 0x83, 0x88, 0x68, 0x8A, 0x09, 0xBB, 0xA0,
        0x79, 0xBC, 0x80, 0x9A, 0x54, 0x84, 0x3A, 0x60,
        0x38, 0x5E, 0x23, 0x10, 0xBB, 0xCB, 0xCC, 0x02,
        0x13, 0xCE, 0x3D, 0xFA, 0xAB, 0x33, 0xB4, 0x7F,
        0x9D, 0x63, 0x05, 0xBC, 0x95, 0xC6, 0x10, 0x78,
        0x13, 0xC5, 0x85, 0xC4, 0xB6, 0x57, 0xBF, 0x30,
        0x54, 0x28, 0x33, 0xB1, 0x49, 0x49, 0xF5, 0x73,
        0xC0, 0x61, 0x2A, 0xD5, 0x24, 0xBA, 0xAE, 0x69,
        0x59, 0x0C, 0x12, 0x77, 0xB8, 0x6C, 0x28, 0x65,
        0x71, 0xBF, 0x66, 0xB3, 0xCF, 0xF4, 0x6A, 0x38,
        0x58, 0xC0, 0x99, 0x06, 0xA7, 0x94, 0xDF, 0x4A,
        0x06, 0xE9, 0xD4, 0xB0, 0xA2, 0xE4, 0x3F, 0x10,
        0xF7, 0x2A, 0x6C, 0x6C, 0x47, 0xE5, 0x64, 0x6E,
        0x2C, 0x79, 0x9B, 0x71, 0xC3, 0x3E, 0xD2, 0xF0,
        0x1E, 0xEB, 0x45, 0x93, 0x8E, 0xB7, 0xA4, 0xE2,
        0xE2, 0x90, 0x8C, 0x53, 0x55, 0x8A, 0x54, 0x0D,
        0x35, 0x03, 0x69, 0xFA, 0x18, 0x9C, 0x61, 0x69,
        0x43, 0xF7, 0x98, 0x1D, 0x76, 0x18, 0xCF, 0x02,
        0xA5, 0xB0, 0xA2, 0xBC, 0xC4, 0x22, 0xE8, 0x57,
        0xD1, 0xA4, 0x78, 0x71, 0x25, 0x3D, 0x08, 0x29,
        0x3C, 0x1C, 0x17, 0x9B, 0xCD, 0xC0, 0x43, 0x70,
        0x69, 0x10, 0x74, 0x18, 0x20, 0x5F, 0xDB, 0x98,
        0x56, 0x62, 0x3B, 0x8C, 0xA6, 0xB6, 0x94, 0xC9,
        0x6C, 0x08, 0x4B, 0x17, 0xF1, 0x3B, 0xB6, 0xDF,
        0x12, 0xB2, 0xCF, 0xBB, 0xC2, 0xB0, 0xE0, 0xC3,
        0x4B, 0x00, 0xD0, 0xFC, 0xD0, 0xAE, 0xCF, 0xB2,
        0x79, 0x24, 0xF6, 0x98, 0x4E, 0x74, 0x7B, 0xE2,
        0xA0, 0x9D, 0x83, 0xA8, 0x66, 0x45, 0x90, 0xA8,
        0x07, 0x73, 0x31, 0x49, 0x1A, 0x4F, 0x7D, 0x72,
        0x08, 0x43, 0xF2, 0x3E, 0x65, 0x2C, 0x6F, 0xA8,
        0x40, 0x30, 0x8D, 0xB4, 0x02, 0x03, 0x37, 0xAA,
        0xD3, 0x79, 0x67, 0x03, 0x4A, 0x9F, 0xB5, 0x23,
        0xB6, 0x7C, 0xA7, 0x03, 0x30, 0xF0, 0x2D, 0x9E,
        0xA2, 0x0C, 0x1E, 0x84, 0xCB, 0x8E, 0x57, 0x57,
        0xC9, 0xE1, 0x89, 0x6B, 0x60, 0x58, 0x14, 0x41,
        0xED, 0x61, 0x8A, 0xA5, 0xB2, 0x6D, 0xA5, 0x6C,
        0x0A, 0x5A, 0x73, 0xC4, 0xDC, 0xFD, 0x75, 0x5E,
        0x61, 0x0B, 0x4F, 0xC8, 0x1F, 0xF8, 0x4E, 0x21
    };
    static const byte dk_1024[WC_ML_KEM_1024_PRIVATE_KEY_SIZE] = {
        0x8C, 0x8B, 0x37, 0x22, 0xA8, 0x2E, 0x55, 0x05,
        0x65, 0x52, 0x16, 0x11, 0xEB, 0xBC, 0x63, 0x07,
        0x99, 0x44, 0xC9, 0xB1, 0xAB, 0xB3, 0xB0, 0x02,
        0x0F, 0xF1, 0x2F, 0x63, 0x18, 0x91, 0xA9, 0xC4,
        0x68, 0xD3, 0xA6, 0x7B, 0xF6, 0x27, 0x12, 0x80,
        0xDA, 0x58, 0xD0, 0x3C, 0xB0, 0x42, 0xB3, 0xA4,
        0x61, 0x44, 0x16, 0x37, 0xF9, 0x29, 0xC2, 0x73,
        0x46, 0x9A, 0xD1, 0x53, 0x11, 0xE9, 0x10, 0xDE,
        0x18, 0xCB, 0x95, 0x37, 0xBA, 0x1B, 0xE4, 0x2E,
        0x98, 0xBB, 0x59, 0xE4, 0x98, 0xA1, 0x3F, 0xD4,
        0x40, 0xD0, 0xE6, 0x9E, 0xE8, 0x32, 0xB4, 0x5C,
        0xD9, 0x5C, 0x38, 0x21, 0x77, 0xD6, 0x70, 0x96,
        0xA1, 0x8C, 0x07, 0xF1, 0x78, 0x16, 0x63, 0x65,
        0x1B, 0xDC, 0xAC, 0x90, 0xDE, 0xDA, 0x3D, 0xDD,
        0x14, 0x34, 0x85, 0x86, 0x41, 0x81, 0xC9, 0x1F,
        0xA2, 0x08, 0x0F, 0x6D, 0xAB, 0x3F, 0x86, 0x20,
        0x4C, 0xEB, 0x64, 0xA7, 0xB4, 0x44, 0x68, 0x95,
        0xC0, 0x39, 0x87, 0xA0, 0x31, 0xCB, 0x4B, 0x6D,
        0x9E, 0x04, 0x62, 0xFD, 0xA8, 0x29, 0x17, 0x2B,
        0x6C, 0x01, 0x2C, 0x63, 0x8B, 0x29, 0xB5, 0xCD,
        0x75, 0xA2, 0xC9, 0x30, 0xA5, 0x59, 0x6A, 0x31,
        0x81, 0xC3, 0x3A, 0x22, 0xD5, 0x74, 0xD3, 0x02,
        0x61, 0x19, 0x6B, 0xC3, 0x50, 0x73, 0x8D, 0x4F,
        0xD9, 0x18, 0x3A, 0x76, 0x33, 0x36, 0x24, 0x3A,
        0xCE, 0xD9, 0x9B, 0x32, 0x21, 0xC7, 0x1D, 0x88,
        0x66, 0x89, 0x5C, 0x4E, 0x52, 0xC1, 0x19, 0xBF,
        0x32, 0x80, 0xDA, 0xF8, 0x0A, 0x95, 0xE1, 0x52,
        0x09, 0xA7, 0x95, 0xC4, 0x43, 0x5F, 0xBB, 0x35,
        0x70, 0xFD, 0xB8, 0xAA, 0x9B, 0xF9, 0xAE, 0xFD,
        0x43, 0xB0, 0x94, 0xB7, 0x81, 0xD5, 0xA8, 0x11,
        0x36, 0xDA, 0xB8, 0x8B, 0x87, 0x99, 0x69, 0x65,
        0x56, 0xFE, 0xC6, 0xAE, 0x14, 0xB0, 0xBB, 0x8B,
        0xE4, 0x69, 0x5E, 0x9A, 0x12, 0x4C, 0x2A, 0xB8,
        0xFF, 0x4A, 0xB1, 0x22, 0x9B, 0x8A, 0xAA, 0x8C,
        0x6F, 0x41, 0xA6, 0x0C, 0x34, 0xC7, 0xB5, 0x61,
        0x82, 0xC5, 0x5C, 0x2C, 0x68, 0x5E, 0x73, 0x7C,
        0x6C, 0xA0, 0x0A, 0x23, 0xFB, 0x8A, 0x68, 0xC1,
        0xCD, 0x61, 0xF3, 0x0D, 0x39, 0x93, 0xA1, 0x65,
        0x3C, 0x16, 0x75, 0xAC, 0x5F, 0x09, 0x01, 0xA7,
        0x16, 0x0A, 0x73, 0x96, 0x64, 0x08, 0xB8, 0x87,
        0x6B, 0x71, 0x53, 0x96, 0xCF, 0xA4, 0x90, 0x3F,
        0xC6, 0x9D, 0x60, 0x49, 0x1F, 0x81, 0x46, 0x80,
        0x8C, 0x97, 0xCD, 0x5C, 0x53, 0x3E, 0x71, 0x01,
        0x79, 0x09, 0xE9, 0x7B, 0x83, 0x5B, 0x86, 0xFF,
        0x84, 0x7B, 0x42, 0xA6, 0x96, 0x37, 0x54, 0x35,
        0xE0, 0x06, 0x06, 0x1C, 0xF7, 0xA4, 0x79, 0x46,
        0x32, 0x72, 0x11, 0x4A, 0x89, 0xEB, 0x3E, 0xAF,
        0x22, 0x46, 0xF0, 0xF8, 0xC1, 0x04, 0xA1, 0x49,
        0x86, 0x82, 0x8E, 0x0A, 0xD2, 0x04, 0x20, 0xC9,
        0xB3, 0x7E, 0xA2, 0x3F, 0x5C, 0x51, 0x49, 0x49,
        0xE7, 0x7A, 0xD9, 0xE9, 0xAD, 0x12, 0x29, 0x0D,
        0xD1, 0x21, 0x5E, 0x11, 0xDA, 0x27, 0x44, 0x57,
        0xAC, 0x86, 0xB1, 0xCE, 0x68, 0x64, 0xB1, 0x22,
        0x67, 0x7F, 0x37, 0x18, 0xAA, 0x31, 0xB0, 0x25,
        0x80, 0xE6, 0x43, 0x17, 0x17, 0x8D, 0x38, 0xF2,
        0x5F, 0x60, 0x9B, 0xC6, 0xC5, 0x5B, 0xC3, 0x74,
        0xA1, 0xBF, 0x78, 0xEA, 0x8E, 0xCC, 0x21, 0x9B,
        0x30, 0xB7, 0x4C, 0xBB, 0x32, 0x72, 0xA5, 0x99,
        0x23, 0x8C, 0x93, 0x98, 0x51, 0x70, 0x04, 0x8F,
        0x17, 0x67, 0x75, 0xFB, 0x19, 0x96, 0x2A, 0xC3,
        0xB1, 0x35, 0xAA, 0x59, 0xDB, 0x10, 0x4F, 0x71,
        0x14, 0xDB, 0xC2, 0xC2, 0xD4, 0x29, 0x49, 0xAD,
        0xEC, 0xA6, 0xA8, 0x5B, 0x32, 0x3E, 0xE2, 0xB2,
        0xB2, 0x3A, 0x77, 0xD9, 0xDB, 0x23, 0x59, 0x79,
        0xA8, 0xE2, 0xD6, 0x7C, 0xF7, 0xD2, 0x13, 0x6B,
        0xBB, 0xA7, 0x1F, 0x26, 0x95, 0x74, 0xB3, 0x88,
        0x88, 0xE1, 0x54, 0x13, 0x40, 0xC1, 0x92, 0x84,
        0x07, 0x4F, 0x9B, 0x7C, 0x8C, 0xF3, 0x7E, 0xB0,
        0x13, 0x84, 0xE6, 0xE3, 0x82, 0x2E, 0xC4, 0x88,
        0x2D, 0xFB, 0xBE, 0xC4, 0xE6, 0x09, 0x8E, 0xF2,
        0xB2, 0xFC, 0x17, 0x7A, 0x1F, 0x0B, 0xCB, 0x65,
        0xA5, 0x7F, 0xDA, 0xA8, 0x93, 0x15, 0x46, 0x1B,
        0xEB, 0x78, 0x85, 0xFB, 0x68, 0xB3, 0xCD, 0x09,
        0x6E, 0xDA, 0x59, 0x6A, 0xC0, 0xE6, 0x1D, 0xD7,
        0xA9, 0xC5, 0x07, 0xBC, 0x63, 0x45, 0xE0, 0x82,
        0x7D, 0xFC, 0xC8, 0xA3, 0xAC, 0x2D, 0xCE, 0x51,
        0xAD, 0x73, 0x1A, 0xA0, 0xEB, 0x93, 0x2A, 0x6D,
        0x09, 0x83, 0x99, 0x23, 0x47, 0xCB, 0xEB, 0x3C,
        0xD0, 0xD9, 0xC9, 0x71, 0x97, 0x97, 0xCC, 0x21,
        0xCF, 0x00, 0x62, 0xB0, 0xAD, 0x94, 0xCA, 0xD7,
        0x34, 0xC6, 0x3E, 0x6B, 0x5D, 0x85, 0x9C, 0xBE,
        0x19, 0xF0, 0x36, 0x82, 0x45, 0x35, 0x1B, 0xF4,
        0x64, 0xD7, 0x50, 0x55, 0x69, 0x79, 0x0D, 0x2B,
        0xB7, 0x24, 0xD8, 0x65, 0x9A, 0x9F, 0xEB, 0x1C,
        0x7C, 0x47, 0x3D, 0xC4, 0xD0, 0x61, 0xE2, 0x98,
        0x63, 0xA2, 0x71, 0x4B, 0xAC, 0x42, 0xAD, 0xCD,
        0x1A, 0x83, 0x72, 0x77, 0x65, 0x56, 0xF7, 0x92,
        0x8A, 0x7A, 0x44, 0xE9, 0x4B, 0x6A, 0x25, 0x32,
        0x2D, 0x03, 0xC0, 0xA1, 0x62, 0x2A, 0x7F, 0xD2,
        0x61, 0x52, 0x2B, 0x73, 0x58, 0xF0, 0x85, 0xBD,
        0xFB, 0x60, 0x75, 0x87, 0x62, 0xCB, 0x90, 0x10,
        0x31, 0x90, 0x1B, 0x5E, 0xEC, 0xF4, 0x92, 0x0C,
        0x81, 0x02, 0x0A, 0x9B, 0x17, 0x81, 0xBC, 0xB9,
        0xDD, 0x19, 0xA9, 0xDF, 0xB6, 0x64, 0x58, 0xE7,
        0x75, 0x7C, 0x52, 0xCE, 0xC7, 0x5B, 0x4B, 0xA7,
        0x40, 0xA2, 0x40, 0x99, 0xCB, 0x56, 0xBB, 0x60,
        0xA7, 0x6B, 0x69, 0x01, 0xAA, 0x3E, 0x01, 0x69,
        0xC9, 0xE8, 0x34, 0x96, 0xD7, 0x3C, 0x4C, 0x99,
        0x43, 0x5A, 0x28, 0xD6, 0x13, 0xE9, 0x7A, 0x11,
        0x77, 0xF5, 0x8B, 0x6C, 0xC5, 0x95, 0xD3, 0xB2,
        0x33, 0x1E, 0x9C, 0xA7, 0xB5, 0x7B, 0x74, 0xDC,
        0x2C, 0x52, 0x77, 0xD2, 0x6F, 0x2F, 0xE1, 0x92,
        0x40, 0xA5, 0x5C, 0x35, 0xD6, 0xCF, 0xCA, 0x26,
        0xC7, 0x3E, 0x9A, 0x2D, 0x7C, 0x98, 0x0D, 0x97,
        0x96, 0x0A, 0xE1, 0xA0, 0x46, 0x98, 0xC1, 0x6B,
        0x39, 0x8A, 0x5F, 0x20, 0xC3, 0x5A, 0x09, 0x14,
        0x14, 0x5C, 0xE1, 0x67, 0x4B, 0x71, 0xAB, 0xC6,
        0x06, 0x6A, 0x90, 0x9A, 0x3E, 0x4B, 0x91, 0x1E,
        0x69, 0xD5, 0xA8, 0x49, 0x43, 0x03, 0x61, 0xF7,
        0x31, 0xB0, 0x72, 0x46, 0xA6, 0x32, 0x9B, 0x52,
        0x36, 0x19, 0x04, 0x22, 0x50, 0x82, 0xD0, 0xAA,
        0xC5, 0xB2, 0x1D, 0x6B, 0x34, 0x86, 0x24, 0x81,
        0xA8, 0x90, 0xC3, 0xC3, 0x60, 0x76, 0x6F, 0x04,
        0x26, 0x36, 0x03, 0xA6, 0xB7, 0x3E, 0x80, 0x2B,
        0x1F, 0x70, 0xB2, 0xEB, 0x00, 0x04, 0x68, 0x36,
        0xB8, 0xF4, 0x93, 0xBF, 0x10, 0xB9, 0x0B, 0x87,
        0x37, 0xC6, 0xC5, 0x48, 0x44, 0x9B, 0x29, 0x4C,
        0x47, 0x25, 0x3B, 0xE2, 0x6C, 0xA7, 0x23, 0x36,
        0xA6, 0x32, 0x06, 0x3A, 0xD3, 0xD0, 0xB4, 0x8C,
        0x8B, 0x0F, 0x4A, 0x34, 0x44, 0x7E, 0xF1, 0x3B,
        0x76, 0x40, 0x20, 0xDE, 0x73, 0x9E, 0xB7, 0x9A,
        0xBA, 0x20, 0xE2, 0xBE, 0x19, 0x51, 0x82, 0x5F,
        0x29, 0x3B, 0xED, 0xD1, 0x08, 0x9F, 0xCB, 0x0A,
        0x91, 0xF5, 0x60, 0xC8, 0xE1, 0x7C, 0xDF, 0x52,
        0x54, 0x1D, 0xC2, 0xB8, 0x1F, 0x97, 0x2A, 0x73,
        0x75, 0xB2, 0x01, 0xF1, 0x0C, 0x08, 0xD9, 0xB5,
        0xBC, 0x8B, 0x95, 0x10, 0x00, 0x54, 0xA3, 0xD0,
        0xAA, 0xFF, 0x89, 0xBD, 0x08, 0xD6, 0xA0, 0xE7,
        0xF2, 0x11, 0x5A, 0x43, 0x52, 0x31, 0x29, 0x04,
        0x60, 0xC9, 0xAD, 0x43, 0x5A, 0x3B, 0x3C, 0xF3,
        0x5E, 0x52, 0x09, 0x1E, 0xDD, 0x18, 0x90, 0x04,
        0x7B, 0xCC, 0x0A, 0xAB, 0xB1, 0xAC, 0xEB, 0xC7,
        0x5F, 0x4A, 0x32, 0xBC, 0x14, 0x51, 0xAC, 0xC4,
        0x96, 0x99, 0x40, 0x78, 0x8E, 0x89, 0x41, 0x21,
        0x88, 0x94, 0x6C, 0x91, 0x43, 0xC5, 0x04, 0x6B,
        0xD1, 0xB4, 0x58, 0xDF, 0x61, 0x7C, 0x5D, 0xF5,
        0x33, 0xB0, 0x52, 0xCD, 0x60, 0x38, 0xB7, 0x75,
        0x40, 0x34, 0xA2, 0x3C, 0x2F, 0x77, 0x20, 0x13,
        0x4C, 0x7B, 0x4E, 0xAC, 0xE0, 0x1F, 0xAC, 0x0A,
        0x28, 0x53, 0xA9, 0x28, 0x58, 0x47, 0xAB, 0xBD,
        0x06, 0xA3, 0x34, 0x3A, 0x77, 0x8A, 0xC6, 0x06,
        0x2E, 0x45, 0x8B, 0xC5, 0xE6, 0x1E, 0xCE, 0x1C,
        0x0D, 0xE0, 0x20, 0x6E, 0x6F, 0xE8, 0xA8, 0x40,
        0x34, 0xA7, 0xC5, 0xF1, 0xB0, 0x05, 0xFB, 0x0A,
        0x58, 0x40, 0x51, 0xD3, 0x22, 0x9B, 0x86, 0xC9,
        0x09, 0xAC, 0x56, 0x47, 0xB3, 0xD7, 0x55, 0x69,
        0xE0, 0x5A, 0x88, 0x27, 0x9D, 0x80, 0xE5, 0xC3,
        0x0F, 0x57, 0x4D, 0xC3, 0x27, 0x51, 0x2C, 0x6B,
        0xBE, 0x81, 0x01, 0x23, 0x9E, 0xC6, 0x28, 0x61,
        0xF4, 0xBE, 0x67, 0xB0, 0x5B, 0x9C, 0xDA, 0x9C,
        0x54, 0x5C, 0x13, 0xE7, 0xEB, 0x53, 0xCF, 0xF2,
        0x60, 0xAD, 0x98, 0x70, 0x19, 0x9C, 0x21, 0xF8,
        0xC6, 0x3D, 0x64, 0xF0, 0x45, 0x8A, 0x71, 0x41,
        0x28, 0x50, 0x23, 0xFE, 0xB8, 0x29, 0x29, 0x08,
        0x72, 0x38, 0x96, 0x44, 0xB0, 0xC3, 0xB7, 0x3A,
        0xC2, 0xC8, 0xE1, 0x21, 0xA2, 0x9B, 0xB1, 0xC4,
        0x3C, 0x19, 0xA2, 0x33, 0xD5, 0x6B, 0xED, 0x82,
        0x74, 0x0E, 0xB0, 0x21, 0xC9, 0x7B, 0x8E, 0xBB,
        0xA4, 0x0F, 0xF3, 0x28, 0xB5, 0x41, 0x76, 0x0F,
        0xCC, 0x37, 0x2B, 0x52, 0xD3, 0xBC, 0x4F, 0xCB,
        0xC0, 0x6F, 0x42, 0x4E, 0xAF, 0x25, 0x38, 0x04,
        0xD4, 0xCB, 0x46, 0xF4, 0x1F, 0xF2, 0x54, 0xC0,
        0xC5, 0xBA, 0x48, 0x3B, 0x44, 0xA8, 0x7C, 0x21,
        0x96, 0x54, 0x55, 0x5E, 0xC7, 0xC1, 0x63, 0xC7,
        0x9B, 0x9C, 0xB7, 0x60, 0xA2, 0xAD, 0x9B, 0xB7,
        0x22, 0xB9, 0x3E, 0x0C, 0x28, 0xBD, 0x4B, 0x16,
        0x85, 0x94, 0x9C, 0x49, 0x6E, 0xAB, 0x1A, 0xFF,
        0x90, 0x91, 0x9E, 0x37, 0x61, 0xB3, 0x46, 0x83,
        0x8A, 0xBB, 0x2F, 0x01, 0xA9, 0x1E, 0x55, 0x43,
        0x75, 0xAF, 0xDA, 0xAA, 0xF3, 0x82, 0x6E, 0x6D,
        0xB7, 0x9F, 0xE7, 0x35, 0x3A, 0x7A, 0x57, 0x8A,
        0x7C, 0x05, 0x98, 0xCE, 0x28, 0xB6, 0xD9, 0x91,
        0x52, 0x14, 0x23, 0x6B, 0xBF, 0xFA, 0x6D, 0x45,
        0xB6, 0x37, 0x6A, 0x07, 0x92, 0x4A, 0x39, 0xA7,
        0xBE, 0x81, 0x82, 0x86, 0x71, 0x5C, 0x8A, 0x3C,
        0x11, 0x0C, 0xD7, 0x6C, 0x02, 0xE0, 0x41, 0x7A,
        0xF1, 0x38, 0xBD, 0xB9, 0x5C, 0x3C, 0xCA, 0x79,
        0x8A, 0xC8, 0x09, 0xED, 0x69, 0xCF, 0xB6, 0x72,
        0xB6, 0xFD, 0xDC, 0x24, 0xD8, 0x9C, 0x06, 0xA6,
        0x55, 0x88, 0x14, 0xAB, 0x0C, 0x21, 0xC6, 0x2B,
        0x2F, 0x84, 0xC0, 0xE3, 0xE0, 0x80, 0x3D, 0xB3,
        0x37, 0xA4, 0xE0, 0xC7, 0x12, 0x7A, 0x6B, 0x4C,
        0x8C, 0x08, 0xB1, 0xD1, 0xA7, 0x6B, 0xF0, 0x7E,
        0xB6, 0xE5, 0xB5, 0xBB, 0x47, 0xA1, 0x6C, 0x74,
        0xBC, 0x54, 0x83, 0x75, 0xFB, 0x29, 0xCD, 0x78,
        0x9A, 0x5C, 0xFF, 0x91, 0xBD, 0xBD, 0x07, 0x18,
        0x59, 0xF4, 0x84, 0x6E, 0x35, 0x5B, 0xB0, 0xD2,
        0x94, 0x84, 0xE2, 0x64, 0xDF, 0xF3, 0x6C, 0x91,
        0x77, 0xA7, 0xAC, 0xA7, 0x89, 0x08, 0x87, 0x96,
        0x95, 0xCA, 0x87, 0xF2, 0x54, 0x36, 0xBC, 0x12,
        0x63, 0x07, 0x24, 0xBB, 0x22, 0xF0, 0xCB, 0x64,
        0x89, 0x7F, 0xE5, 0xC4, 0x11, 0x95, 0x28, 0x0D,
        0xA0, 0x41, 0x84, 0xD4, 0xBC, 0x7B, 0x53, 0x2A,
        0x0F, 0x70, 0xA5, 0x4D, 0x77, 0x57, 0xCD, 0xE6,
        0x17, 0x5A, 0x68, 0x43, 0xB8, 0x61, 0xCB, 0x2B,
        0xC4, 0x83, 0x0C, 0x00, 0x12, 0x55, 0x4C, 0xFC,
        0x5D, 0x2C, 0x8A, 0x20, 0x27, 0xAA, 0x3C, 0xD9,
        0x67, 0x13, 0x0E, 0x9B, 0x96, 0x24, 0x1B, 0x11,
        0xC4, 0x32, 0x0C, 0x76, 0x49, 0xCC, 0x23, 0xA7,
        0x1B, 0xAF, 0xE6, 0x91, 0xAF, 0xC0, 0x8E, 0x68,
        0x0B, 0xCE, 0xF4, 0x29, 0x07, 0x00, 0x07, 0x18,
        0xE4, 0xEA, 0xCE, 0x8D, 0xA2, 0x82, 0x14, 0x19,
        0x7B, 0xE1, 0xC2, 0x69, 0xDA, 0x9C, 0xB5, 0x41,
        0xE1, 0xA3, 0xCE, 0x97, 0xCF, 0xAD, 0xF9, 0xC6,
        0x05, 0x87, 0x80, 0xFE, 0x67, 0x93, 0xDB, 0xFA,
        0x82, 0x18, 0xA2, 0x76, 0x0B, 0x80, 0x2B, 0x8D,
        0xA2, 0xAA, 0x27, 0x1A, 0x38, 0x77, 0x25, 0x23,
        0xA7, 0x67, 0x36, 0xA7, 0xA3, 0x1B, 0x9D, 0x30,
        0x37, 0xAD, 0x21, 0xCE, 0xBB, 0x11, 0xA4, 0x72,
        0xB8, 0x79, 0x2E, 0xB1, 0x75, 0x58, 0xB9, 0x40,
        0xE7, 0x08, 0x83, 0xF2, 0x64, 0x59, 0x2C, 0x68,
        0x9B, 0x24, 0x0B, 0xB4, 0x3D, 0x54, 0x08, 0xBF,
        0x44, 0x64, 0x32, 0xF4, 0x12, 0xF4, 0xB9, 0xA5,
        0xF6, 0x86, 0x5C, 0xC2, 0x52, 0xA4, 0x3C, 0xF4,
        0x0A, 0x32, 0x03, 0x91, 0x55, 0x55, 0x91, 0xD6,
        0x75, 0x61, 0xFD, 0xD0, 0x53, 0x53, 0xAB, 0x6B,
        0x01, 0x9B, 0x3A, 0x08, 0xA7, 0x33, 0x53, 0xD5,
        0x1B, 0x61, 0x13, 0xAB, 0x2F, 0xA5, 0x1D, 0x97,
        0x56, 0x48, 0xEE, 0x25, 0x4A, 0xF8, 0x9A, 0x23,
        0x05, 0x04, 0xA2, 0x36, 0xA4, 0x65, 0x82, 0x57,
        0x74, 0x0B, 0xDC, 0xBB, 0xE1, 0x70, 0x8A, 0xB0,
        0x22, 0xC3, 0xC5, 0x88, 0xA4, 0x10, 0xDB, 0x3B,
        0x9C, 0x30, 0x8A, 0x06, 0x27, 0x5B, 0xDF, 0x5B,
        0x48, 0x59, 0xD3, 0xA2, 0x61, 0x7A, 0x29, 0x5E,
        0x1A, 0x22, 0xF9, 0x01, 0x98, 0xBA, 0xD0, 0x16,
        0x6F, 0x4A, 0x94, 0x34, 0x17, 0xC5, 0xB8, 0x31,
        0x73, 0x6C, 0xB2, 0xC8, 0x58, 0x0A, 0xBF, 0xDE,
        0x57, 0x14, 0xB5, 0x86, 0xAB, 0xEE, 0xC0, 0xA1,
        0x75, 0xA0, 0x8B, 0xC7, 0x10, 0xC7, 0xA2, 0x89,
        0x5D, 0xE9, 0x3A, 0xC4, 0x38, 0x06, 0x1B, 0xF7,
        0x76, 0x5D, 0x0D, 0x21, 0xCD, 0x41, 0x81, 0x67,
        0xCA, 0xF8, 0x9D, 0x1E, 0xFC, 0x34, 0x48, 0xBC,
        0xBB, 0x96, 0xD6, 0x9B, 0x3E, 0x01, 0x0C, 0x82,
        0xD1, 0x5C, 0xAB, 0x6C, 0xAC, 0xC6, 0x79, 0x9D,
        0x36, 0x39, 0x66, 0x9A, 0x5B, 0x21, 0xA6, 0x33,
        0xC8, 0x65, 0xF8, 0x59, 0x3B, 0x5B, 0x7B, 0xC8,
        0x00, 0x26, 0x2B, 0xB8, 0x37, 0xA9, 0x24, 0xA6,
        0xC5, 0x44, 0x0E, 0x4F, 0xC7, 0x3B, 0x41, 0xB2,
        0x30, 0x92, 0xC3, 0x91, 0x2F, 0x4C, 0x6B, 0xEB,
        0xB4, 0xC7, 0xB4, 0xC6, 0x29, 0x08, 0xB0, 0x37,
        0x75, 0x66, 0x6C, 0x22, 0x22, 0x0D, 0xF9, 0xC8,
        0x88, 0x23, 0xE3, 0x44, 0xC7, 0x30, 0x83, 0x32,
        0x34, 0x5C, 0x8B, 0x79, 0x5D, 0x34, 0xE8, 0xC0,
        0x51, 0xF2, 0x1F, 0x5A, 0x21, 0xC2, 0x14, 0xB6,
        0x98, 0x41, 0x35, 0x87, 0x09, 0xB1, 0xC3, 0x05,
        0xB3, 0x2C, 0xC2, 0xC3, 0x80, 0x6A, 0xE9, 0xCC,
        0xD3, 0x81, 0x9F, 0xFF, 0x45, 0x07, 0xFE, 0x52,
        0x0F, 0xBF, 0xC2, 0x71, 0x99, 0xBC, 0x23, 0xBE,
        0x6B, 0x9B, 0x2D, 0x2A, 0xC1, 0x71, 0x75, 0x79,
        0xAC, 0x76, 0x92, 0x79, 0xE2, 0xA7, 0xAA, 0xC6,
        0x8A, 0x37, 0x1A, 0x47, 0xBA, 0x3A, 0x7D, 0xBE,
        0x01, 0x6F, 0x14, 0xE1, 0xA7, 0x27, 0x33, 0x36,
        0x63, 0xC4, 0xA5, 0xCD, 0x1A, 0x0F, 0x88, 0x36,
        0xCF, 0x7B, 0x5C, 0x49, 0xAC, 0x51, 0x48, 0x5C,
        0xA6, 0x03, 0x45, 0xC9, 0x90, 0xE0, 0x68, 0x88,
        0x72, 0x00, 0x03, 0x73, 0x13, 0x22, 0xC5, 0xB8,
        0xCD, 0x5E, 0x69, 0x07, 0xFD, 0xA1, 0x15, 0x7F,
        0x46, 0x8F, 0xD3, 0xFC, 0x20, 0xFA, 0x81, 0x75,
        0xEE, 0xC9, 0x5C, 0x29, 0x1A, 0x26, 0x2B, 0xA8,
        0xC5, 0xBE, 0x99, 0x08, 0x72, 0x41, 0x89, 0x30,
        0x85, 0x23, 0x39, 0xD8, 0x8A, 0x19, 0xB3, 0x7F,
        0xEF, 0xA3, 0xCF, 0xE8, 0x21, 0x75, 0xC2, 0x24,
        0x40, 0x7C, 0xA4, 0x14, 0xBA, 0xEB, 0x37, 0x92,
        0x3B, 0x4D, 0x2D, 0x83, 0x13, 0x4A, 0xE1, 0x54,
        0xE4, 0x90, 0xA9, 0xB4, 0x5A, 0x05, 0x63, 0xB0,
        0x6C, 0x95, 0x3C, 0x33, 0x01, 0x45, 0x0A, 0x21,
        0x76, 0xA0, 0x7C, 0x61, 0x4A, 0x74, 0xE3, 0x47,
        0x8E, 0x48, 0x50, 0x9F, 0x9A, 0x60, 0xAE, 0x94,
        0x5A, 0x8E, 0xBC, 0x78, 0x15, 0x12, 0x1D, 0x90,
        0xA3, 0xB0, 0xE0, 0x70, 0x91, 0xA0, 0x96, 0xCF,
        0x02, 0xC5, 0x7B, 0x25, 0xBC, 0xA5, 0x81, 0x26,
        0xAD, 0x0C, 0x62, 0x9C, 0xE1, 0x66, 0xA7, 0xED,
        0xB4, 0xB3, 0x32, 0x21, 0xA0, 0xD3, 0xF7, 0x2B,
        0x85, 0xD5, 0x62, 0xEC, 0x69, 0x8B, 0x7D, 0x0A,
        0x91, 0x3D, 0x73, 0x80, 0x6F, 0x1C, 0x5C, 0x87,
        0xB3, 0x8E, 0xC0, 0x03, 0xCB, 0x30, 0x3A, 0x3D,
        0xC5, 0x1B, 0x4B, 0x35, 0x35, 0x6A, 0x67, 0x82,
        0x6D, 0x6E, 0xDA, 0xA8, 0xFE, 0xB9, 0x3B, 0x98,
        0x49, 0x3B, 0x2D, 0x1C, 0x11, 0xB6, 0x76, 0xA6,
        0xAD, 0x95, 0x06, 0xA1, 0xAA, 0xAE, 0x13, 0xA8,
        0x24, 0xC7, 0xC0, 0x8D, 0x1C, 0x6C, 0x2C, 0x4D,
        0xBA, 0x96, 0x42, 0xC7, 0x6E, 0xA7, 0xF6, 0xC8,
        0x26, 0x4B, 0x64, 0xA2, 0x3C, 0xCC, 0xA9, 0xA7,
        0x46, 0x35, 0xFC, 0xBF, 0x03, 0xE0, 0x0F, 0x1B,
        0x57, 0x22, 0xB2, 0x14, 0x37, 0x67, 0x90, 0x79,
        0x3B, 0x2C, 0x4F, 0x0A, 0x13, 0xB5, 0xC4, 0x07,
        0x60, 0xB4, 0x21, 0x8E, 0x1D, 0x25, 0x94, 0xDC,
        0xB3, 0x0A, 0x70, 0xD9, 0xC1, 0x78, 0x2A, 0x5D,
        0xD3, 0x05, 0x76, 0xFA, 0x41, 0x44, 0xBF, 0xC8,
        0x41, 0x6E, 0xDA, 0x81, 0x18, 0xFC, 0x64, 0x72,
        0xF5, 0x6A, 0x97, 0x95, 0x86, 0xF3, 0x3B, 0xB0,
        0x70, 0xFB, 0x0F, 0x1B, 0x0B, 0x10, 0xBC, 0x48,
        0x97, 0xEB, 0xE0, 0x1B, 0xCA, 0x38, 0x93, 0xD4,
        0xE1, 0x6A, 0xDB, 0x25, 0x09, 0x3A, 0x74, 0x17,
        0xD0, 0x70, 0x8C, 0x83, 0xA2, 0x63, 0x22, 0xE2,
        0x2E, 0x63, 0x30, 0x09, 0x1E, 0x30, 0x15, 0x2B,
        0xF8, 0x23, 0x59, 0x7C, 0x04, 0xCC, 0xF4, 0xCF,
        0xC7, 0x33, 0x15, 0x78, 0xF4, 0x3A, 0x27, 0x26,
        0xCC, 0xB4, 0x28, 0x28, 0x9A, 0x90, 0xC8, 0x63,
        0x25, 0x9D, 0xD1, 0x80, 0xC5, 0xFF, 0x14, 0x2B,
        0xEF, 0x41, 0xC7, 0x71, 0x70, 0x94, 0xBE, 0x07,
        0x85, 0x6D, 0xA2, 0xB1, 0x40, 0xFA, 0x67, 0x71,
        0x09, 0x67, 0x35, 0x6A, 0xA4, 0x7D, 0xFB, 0xC8,
        0xD2, 0x55, 0xB4, 0x72, 0x2A, 0xB8, 0x6D, 0x43,
        0x9B, 0x7E, 0x0A, 0x60, 0x90, 0x25, 0x1D, 0x2D,
        0x4C, 0x1E, 0xD5, 0xF2, 0x0B, 0xBE, 0x68, 0x07,
        0xBF, 0x65, 0xA9, 0x0B, 0x7C, 0xB2, 0xEC, 0x01,
        0x02, 0xAF, 0x02, 0x80, 0x9D, 0xC9, 0xAC, 0x7D,
        0x0A, 0x3A, 0xBC, 0x69, 0xC1, 0x83, 0x65, 0xBC,
        0xFF, 0x59, 0x18, 0x5F, 0x33, 0x99, 0x68, 0x87,
        0x74, 0x61, 0x85, 0x90, 0x6C, 0x01, 0x91, 0xAE,
        0xD4, 0x40, 0x7E, 0x13, 0x94, 0x46, 0x45, 0x9B,
        0xE2, 0x9C, 0x68, 0x22, 0x71, 0x76, 0x44, 0x35,
        0x3D, 0x24, 0xAB, 0x63, 0x39, 0x15, 0x6A, 0x9C,
        0x42, 0x49, 0x09, 0xF0, 0xA9, 0x02, 0x5B, 0xB7,
        0x47, 0x20, 0x77, 0x9B, 0xE4, 0x3F, 0x16, 0xD8,
        0x1C, 0x8C, 0xC6, 0x66, 0xE9, 0x97, 0x10, 0xD8,
        0xC6, 0x8B, 0xB5, 0xCC, 0x4E, 0x12, 0xF3, 0x14,
        0xE9, 0x25, 0xA5, 0x51, 0xF0, 0x9C, 0xC5, 0x90,
        0x03, 0xA1, 0xF8, 0x81, 0x03, 0xC2, 0x54, 0xBB,
        0x97, 0x8D, 0x75, 0xF3, 0x94, 0xD3, 0x54, 0x0E,
        0x31, 0xE7, 0x71, 0xCD, 0xA3, 0x6E, 0x39, 0xEC,
        0x54, 0xA6, 0x2B, 0x58, 0x32, 0x66, 0x4D, 0x82,
        0x1A, 0x72, 0xF1, 0xE6, 0xAF, 0xBB, 0xA2, 0x7F,
        0x84, 0x29, 0x5B, 0x26, 0x94, 0xC4, 0x98, 0x49,
        0x8E, 0x81, 0x2B, 0xC8, 0xE9, 0x37, 0x8F, 0xE5,
        0x41, 0xCE, 0xC5, 0x89, 0x1B, 0x25, 0x06, 0x29,
        0x01, 0xCB, 0x72, 0x12, 0xE3, 0xCD, 0xC4, 0x61,
        0x79, 0xEC, 0x5B, 0xCE, 0xC1, 0x0B, 0xC0, 0xB9,
        0x31, 0x1D, 0xE0, 0x50, 0x74, 0x29, 0x06, 0x87,
        0xFD, 0x6A, 0x53, 0x92, 0x67, 0x16, 0x54, 0x28,
        0x4C, 0xD9, 0xC8, 0xCC, 0x3E, 0xBA, 0x80, 0xEB,
        0x3B, 0x66, 0x2E, 0xB5, 0x3E, 0xB7, 0x51, 0x16,
        0x70, 0x4A, 0x1F, 0xEB, 0x5C, 0x2D, 0x05, 0x63,
        0x38, 0x53, 0x28, 0x68, 0xDD, 0xF2, 0x4E, 0xB8,
        0x99, 0x2A, 0xB8, 0x56, 0x5D, 0x9E, 0x49, 0x0C,
        0xAD, 0xF1, 0x48, 0x04, 0x36, 0x0D, 0xAA, 0x90,
        0x71, 0x8E, 0xAB, 0x61, 0x6B, 0xAB, 0x07, 0x65,
        0xD3, 0x39, 0x87, 0xB4, 0x7E, 0xFB, 0x65, 0x99,
        0xC5, 0x56, 0x32, 0x35, 0xE6, 0x1E, 0x4B, 0xE6,
        0x70, 0xE9, 0x79, 0x55, 0xAB, 0x29, 0x2D, 0x97,
        0x32, 0xCB, 0x89, 0x30, 0x94, 0x8A, 0xC8, 0x2D,
        0xF2, 0x30, 0xAC, 0x72, 0x29, 0x7A, 0x23, 0x67,
        0x9D, 0x6B, 0x94, 0xC1, 0x7F, 0x13, 0x59, 0x48,
        0x32, 0x54, 0xFE, 0xDC, 0x2F, 0x05, 0x81, 0x9F,
        0x0D, 0x06, 0x9A, 0x44, 0x3B, 0x78, 0xE3, 0xFC,
        0x6C, 0x3E, 0xF4, 0x71, 0x4B, 0x05, 0xA3, 0xFC,
        0xA8, 0x1C, 0xBB, 0xA6, 0x02, 0x42, 0xA7, 0x06,
        0x0C, 0xD8, 0x85, 0xD8, 0xF3, 0x99, 0x81, 0xBB,
        0x18, 0x09, 0x2B, 0x23, 0xDA, 0xA5, 0x9F, 0xD9,
        0x57, 0x83, 0x88, 0x68, 0x8A, 0x09, 0xBB, 0xA0,
        0x79, 0xBC, 0x80, 0x9A, 0x54, 0x84, 0x3A, 0x60,
        0x38, 0x5E, 0x23, 0x10, 0xBB, 0xCB, 0xCC, 0x02,
        0x13, 0xCE, 0x3D, 0xFA, 0xAB, 0x33, 0xB4, 0x7F,
        0x9D, 0x63, 0x05, 0xBC, 0x95, 0xC6, 0x10, 0x78,
        0x13, 0xC5, 0x85, 0xC4, 0xB6, 0x57, 0xBF, 0x30,
        0x54, 0x28, 0x33, 0xB1, 0x49, 0x49, 0xF5, 0x73,
        0xC0, 0x61, 0x2A, 0xD5, 0x24, 0xBA, 0xAE, 0x69,
        0x59, 0x0C, 0x12, 0x77, 0xB8, 0x6C, 0x28, 0x65,
        0x71, 0xBF, 0x66, 0xB3, 0xCF, 0xF4, 0x6A, 0x38,
        0x58, 0xC0, 0x99, 0x06, 0xA7, 0x94, 0xDF, 0x4A,
        0x06, 0xE9, 0xD4, 0xB0, 0xA2, 0xE4, 0x3F, 0x10,
        0xF7, 0x2A, 0x6C, 0x6C, 0x47, 0xE5, 0x64, 0x6E,
        0x2C, 0x79, 0x9B, 0x71, 0xC3, 0x3E, 0xD2, 0xF0,
        0x1E, 0xEB, 0x45, 0x93, 0x8E, 0xB7, 0xA4, 0xE2,
        0xE2, 0x90, 0x8C, 0x53, 0x55, 0x8A, 0x54, 0x0D,
        0x35, 0x03, 0x69, 0xFA, 0x18, 0x9C, 0x61, 0x69,
        0x43, 0xF7, 0x98, 0x1D, 0x76, 0x18, 0xCF, 0x02,
        0xA5, 0xB0, 0xA2, 0xBC, 0xC4, 0x22, 0xE8, 0x57,
        0xD1, 0xA4, 0x78, 0x71, 0x25, 0x3D, 0x08, 0x29,
        0x3C, 0x1C, 0x17, 0x9B, 0xCD, 0xC0, 0x43, 0x70,
        0x69, 0x10, 0x74, 0x18, 0x20, 0x5F, 0xDB, 0x98,
        0x56, 0x62, 0x3B, 0x8C, 0xA6, 0xB6, 0x94, 0xC9,
        0x6C, 0x08, 0x4B, 0x17, 0xF1, 0x3B, 0xB6, 0xDF,
        0x12, 0xB2, 0xCF, 0xBB, 0xC2, 0xB0, 0xE0, 0xC3,
        0x4B, 0x00, 0xD0, 0xFC, 0xD0, 0xAE, 0xCF, 0xB2,
        0x79, 0x24, 0xF6, 0x98, 0x4E, 0x74, 0x7B, 0xE2,
        0xA0, 0x9D, 0x83, 0xA8, 0x66, 0x45, 0x90, 0xA8,
        0x07, 0x73, 0x31, 0x49, 0x1A, 0x4F, 0x7D, 0x72,
        0x08, 0x43, 0xF2, 0x3E, 0x65, 0x2C, 0x6F, 0xA8,
        0x40, 0x30, 0x8D, 0xB4, 0x02, 0x03, 0x37, 0xAA,
        0xD3, 0x79, 0x67, 0x03, 0x4A, 0x9F, 0xB5, 0x23,
        0xB6, 0x7C, 0xA7, 0x03, 0x30, 0xF0, 0x2D, 0x9E,
        0xA2, 0x0C, 0x1E, 0x84, 0xCB, 0x8E, 0x57, 0x57,
        0xC9, 0xE1, 0x89, 0x6B, 0x60, 0x58, 0x14, 0x41,
        0xED, 0x61, 0x8A, 0xA5, 0xB2, 0x6D, 0xA5, 0x6C,
        0x0A, 0x5A, 0x73, 0xC4, 0xDC, 0xFD, 0x75, 0x5E,
        0x61, 0x0B, 0x4F, 0xC8, 0x1F, 0xF8, 0x4E, 0x21,
        0xD2, 0xE5, 0x74, 0xDF, 0xD8, 0xCD, 0x0A, 0xE8,
        0x93, 0xAA, 0x7E, 0x12, 0x5B, 0x44, 0xB9, 0x24,
        0xF4, 0x52, 0x23, 0xEC, 0x09, 0xF2, 0xAD, 0x11,
        0x41, 0xEA, 0x93, 0xA6, 0x80, 0x50, 0xDB, 0xF6,
        0x99, 0xE3, 0x24, 0x68, 0x84, 0x18, 0x1F, 0x8E,
        0x1D, 0xD4, 0x4E, 0x0C, 0x76, 0x29, 0x09, 0x33,
        0x30, 0x22, 0x1F, 0xD6, 0x7D, 0x9B, 0x7D, 0x6E,
        0x15, 0x10, 0xB2, 0xDB, 0xAD, 0x87, 0x62, 0xF7
    };
#endif
    static byte pubKey[WC_ML_KEM_MAX_PUBLIC_KEY_SIZE];
    static byte privKey[WC_ML_KEM_MAX_PRIVATE_KEY_SIZE];

    key = (MlKemKey*)XMALLOC(sizeof(MlKemKey), NULL, DYNAMIC_TYPE_TMP_BUFFER);
    ExpectNotNull(key);
    if (key != NULL) {
        XMEMSET(key, 0, sizeof(MlKemKey));
    }

#ifndef WOLFSSL_NO_ML_KEM_512
    ExpectIntEQ(wc_MlKemKey_Init(key, WC_ML_KEM_512, NULL, INVALID_DEVID), 0);
    ExpectIntEQ(wc_MlKemKey_MakeKeyWithRandom(key, seed_512, sizeof(seed_512)),
        0);
    ExpectIntEQ(wc_MlKemKey_EncodePublicKey(key, pubKey,
        WC_ML_KEM_512_PUBLIC_KEY_SIZE), 0);
    ExpectIntEQ(wc_MlKemKey_EncodePrivateKey(key, privKey,
        WC_ML_KEM_512_PRIVATE_KEY_SIZE), 0);
    ExpectIntEQ(XMEMCMP(pubKey, ek_512, WC_ML_KEM_512_PUBLIC_KEY_SIZE), 0);
    ExpectIntEQ(XMEMCMP(privKey, dk_512, WC_ML_KEM_512_PRIVATE_KEY_SIZE), 0);
    wc_MlKemKey_Free(key);
#endif
#ifndef WOLFSSL_NO_ML_KEM_768
    ExpectIntEQ(wc_MlKemKey_Init(key, WC_ML_KEM_768, NULL, INVALID_DEVID), 0);
    ExpectIntEQ(wc_MlKemKey_MakeKeyWithRandom(key, seed_768, sizeof(seed_768)),
        0);
    ExpectIntEQ(wc_MlKemKey_EncodePublicKey(key, pubKey,
        WC_ML_KEM_768_PUBLIC_KEY_SIZE), 0);
    ExpectIntEQ(wc_MlKemKey_EncodePrivateKey(key, privKey,
        WC_ML_KEM_768_PRIVATE_KEY_SIZE), 0);
    ExpectIntEQ(XMEMCMP(pubKey, ek_768, WC_ML_KEM_768_PUBLIC_KEY_SIZE), 0);
    ExpectIntEQ(XMEMCMP(privKey, dk_768, WC_ML_KEM_768_PRIVATE_KEY_SIZE), 0);
    wc_MlKemKey_Free(key);
#endif
#ifndef WOLFSSL_NO_ML_KEM_1024
    ExpectIntEQ(wc_MlKemKey_Init(key, WC_ML_KEM_1024, NULL, INVALID_DEVID), 0);
    ExpectIntEQ(wc_MlKemKey_MakeKeyWithRandom(key, seed_1024,
        sizeof(seed_1024)), 0);
    ExpectIntEQ(wc_MlKemKey_EncodePublicKey(key, pubKey,
        WC_ML_KEM_1024_PUBLIC_KEY_SIZE), 0);
    ExpectIntEQ(wc_MlKemKey_EncodePrivateKey(key, privKey,
        WC_ML_KEM_1024_PRIVATE_KEY_SIZE), 0);
    ExpectIntEQ(XMEMCMP(pubKey, ek_1024, WC_ML_KEM_1024_PUBLIC_KEY_SIZE), 0);
    ExpectIntEQ(XMEMCMP(privKey, dk_1024, WC_ML_KEM_1024_PRIVATE_KEY_SIZE), 0);
    wc_MlKemKey_Free(key);
#endif

    XFREE(key, NULL, DYNAMIC_TYPE_TMP_BUFFER);
#endif
    return EXPECT_RESULT();
}

int test_wc_mlkem_encapsulate_kats(void)
{
    EXPECT_DECLS;
#if defined(WOLFSSL_HAVE_MLKEM) && defined(WOLFSSL_WC_MLKEM) && \
    !defined(WOLFSSL_NO_ML_KEM) && !defined(WOLFSSL_MLKEM_NO_ENCAPSULATE)
    MlKemKey* key;
#ifndef WOLFSSL_NO_ML_KEM_512
    static const byte ek_512[WC_ML_KEM_512_PUBLIC_KEY_SIZE] = {
        0xDD, 0x19, 0x24, 0x93, 0x5A, 0xA8, 0xE6, 0x17,
        0xAF, 0x18, 0xB5, 0xA0, 0x65, 0xAC, 0x45, 0x72,
        0x77, 0x67, 0xEE, 0x89, 0x7C, 0xF4, 0xF9, 0x44,
        0x2B, 0x2A, 0xCE, 0x30, 0xC0, 0x23, 0x7B, 0x30,
        0x7D, 0x3E, 0x76, 0xBF, 0x8E, 0xEB, 0x78, 0xAD,
        0xDC, 0x4A, 0xAC, 0xD1, 0x64, 0x63, 0xD8, 0x60,
        0x2F, 0xD5, 0x48, 0x7B, 0x63, 0xC8, 0x8B, 0xB6,
        0x60, 0x27, 0xF3, 0x7D, 0x0D, 0x61, 0x4D, 0x6F,
        0x9C, 0x24, 0x60, 0x3C, 0x42, 0x94, 0x76, 0x64,
        0xAC, 0x43, 0x98, 0xC6, 0xC5, 0x23, 0x83, 0x46,
        0x9B, 0x4F, 0x97, 0x77, 0xE5, 0xEC, 0x72, 0x06,
        0x21, 0x0F, 0x3E, 0x5A, 0x79, 0x6B, 0xF4, 0x5C,
        0x53, 0x26, 0x8E, 0x25, 0xF3, 0x9A, 0xC2, 0x61,
        0xAF, 0x3B, 0xFA, 0x2E, 0xE7, 0x55, 0xBE, 0xB8,
        0xB6, 0x7A, 0xB3, 0xAC, 0x8D, 0xF6, 0xC6, 0x29,
        0xC1, 0x17, 0x6E, 0x9E, 0x3B, 0x96, 0x5E, 0x93,
        0x69, 0xF9, 0xB3, 0xB9, 0x2A, 0xD7, 0xC2, 0x09,
        0x55, 0x64, 0x1D, 0x99, 0x52, 0x6F, 0xE7, 0xB9,
        0xFE, 0x8C, 0x85, 0x08, 0x20, 0x27, 0x5C, 0xD9,
        0x64, 0x84, 0x92, 0x50, 0x09, 0x07, 0x33, 0xCE,
        0x12, 0x4E, 0xCF, 0x31, 0x66, 0x24, 0x37, 0x4B,
        0xD1, 0x8B, 0x7C, 0x35, 0x8C, 0x06, 0xE9, 0xC1,
        0x36, 0xEE, 0x12, 0x59, 0xA9, 0x24, 0x5A, 0xBC,
        0x55, 0xB9, 0x64, 0xD6, 0x89, 0xF5, 0xA0, 0x82,
        0x92, 0xD2, 0x82, 0x65, 0x65, 0x8E, 0xBB, 0x40,
        0xCB, 0xFE, 0x48, 0x8A, 0x22, 0x28, 0x27, 0x55,
        0x90, 0xAB, 0x9F, 0x32, 0xA3, 0x41, 0x09, 0x70,
        0x9C, 0x1C, 0x29, 0x1D, 0x4A, 0x23, 0x33, 0x72,
        0x74, 0xC7, 0xA5, 0xA5, 0x99, 0x1C, 0x7A, 0x87,
        0xB8, 0x1C, 0x97, 0x4A, 0xB1, 0x8C, 0xE7, 0x78,
        0x59, 0xE4, 0x99, 0x5E, 0x7C, 0x14, 0xF0, 0x37,
        0x17, 0x48, 0xB7, 0x71, 0x2F, 0xB5, 0x2C, 0x59,
        0x66, 0xCD, 0x63, 0x06, 0x3C, 0x4F, 0x3B, 0x81,
        0xB4, 0x7C, 0x45, 0xDD, 0xE8, 0x3F, 0xB3, 0xA2,
        0x72, 0x40, 0x29, 0xB1, 0x0B, 0x32, 0x30, 0x21,
        0x4C, 0x04, 0xFA, 0x05, 0x77, 0xFC, 0x29, 0xAC,
        0x90, 0x86, 0xAE, 0x18, 0xC5, 0x3B, 0x3E, 0xD4,
        0x4E, 0x50, 0x74, 0x12, 0xFC, 0xA0, 0x4B, 0x4F,
        0x53, 0x8A, 0x51, 0x58, 0x8E, 0xC1, 0xF1, 0x02,
        0x9D, 0x15, 0x2D, 0x9A, 0xE7, 0x73, 0x5F, 0x76,
        0xA0, 0x77, 0xAA, 0x94, 0x84, 0x38, 0x0A, 0xED,
        0x91, 0x89, 0xE5, 0x91, 0x24, 0x87, 0xFC, 0xC5,
        0xB7, 0xC7, 0x01, 0x2D, 0x92, 0x23, 0xDD, 0x96,
        0x7E, 0xEC, 0xDA, 0xC3, 0x00, 0x8A, 0x89, 0x31,
        0xB6, 0x48, 0x24, 0x35, 0x37, 0xF5, 0x48, 0xC1,
        0x71, 0x69, 0x8C, 0x5B, 0x38, 0x1D, 0x84, 0x6A,
        0x72, 0xE5, 0xC9, 0x2D, 0x42, 0x26, 0xC5, 0xA8,
        0x90, 0x98, 0x84, 0xF1, 0xC4, 0xA3, 0x40, 0x4C,
        0x17, 0x20, 0xA5, 0x27, 0x94, 0x14, 0xD7, 0xF2,
        0x7B, 0x2B, 0x98, 0x26, 0x52, 0xB6, 0x74, 0x02,
        0x19, 0xC5, 0x6D, 0x21, 0x77, 0x80, 0xD7, 0xA5,
        0xE5, 0xBA, 0x59, 0x83, 0x63, 0x49, 0xF7, 0x26,
        0x88, 0x1D, 0xEA, 0x18, 0xEF, 0x75, 0xC0, 0x77,
        0x2A, 0x8B, 0x92, 0x27, 0x66, 0x95, 0x37, 0x18,
        0xCA, 0xCC, 0x14, 0xCC, 0xBA, 0xCB, 0x5F, 0xC4,
        0x12, 0xA2, 0xD0, 0xBE, 0x52, 0x18, 0x17, 0x64,
        0x5A, 0xB2, 0xBF, 0x6A, 0x47, 0x85, 0xE9, 0x2B,
        0xC9, 0x4C, 0xAF, 0x47, 0x7A, 0x96, 0x78, 0x76,
        0x79, 0x6C, 0x0A, 0x51, 0x90, 0x31, 0x5A, 0xC0,
        0x88, 0x56, 0x71, 0xA4, 0xC7, 0x49, 0x56, 0x4C,
        0x3B, 0x2C, 0x7A, 0xED, 0x90, 0x64, 0xEB, 0xA2,
        0x99, 0xEF, 0x21, 0x4B, 0xA2, 0xF4, 0x04, 0x93,
        0x66, 0x7C, 0x8B, 0xD0, 0x32, 0xAE, 0xC5, 0x62,
        0x17, 0x11, 0xB4, 0x1A, 0x38, 0x52, 0xC5, 0xC2,
        0xBA, 0xB4, 0xA3, 0x49, 0xCE, 0x4B, 0x7F, 0x08,
        0x5A, 0x81, 0x2B, 0xBB, 0xC8, 0x20, 0xB8, 0x1B,
        0xEF, 0xE6, 0x3A, 0x05, 0xB8, 0xBC, 0xDF, 0xE9,
        0xC2, 0xA7, 0x0A, 0x8B, 0x1A, 0xCA, 0x9B, 0xF9,
        0x81, 0x64, 0x81, 0x90, 0x7F, 0xF4, 0x43, 0x24,
        0x61, 0x11, 0x12, 0x87, 0x30, 0x3F, 0x0B, 0xD8,
        0x17, 0xC0, 0x57, 0x26, 0xBF, 0xA1, 0x8A, 0x2E,
        0x24, 0xC7, 0x72, 0x49, 0x21, 0x02, 0x80, 0x32,
        0xF6, 0x22, 0xBD, 0x96, 0x0A, 0x31, 0x7D, 0x83,
        0xB3, 0x56, 0xB5, 0x7F, 0x4A, 0x80, 0x04, 0x49,
        0x9C, 0xBC, 0x73, 0xC9, 0x7D, 0x1E, 0xB7, 0x74,
        0x59, 0x72, 0x63, 0x1C, 0x05, 0x61, 0xC1, 0xA3,
        0xAB, 0x6E, 0xF9, 0x1B, 0xD3, 0x63, 0x28, 0x0A,
        0x10, 0x54, 0x5D, 0xA6, 0x93, 0xE6, 0xD5, 0x8A,
        0xED, 0x68, 0x45, 0xE7, 0xCC, 0x5F, 0x0D, 0x08,
        0xCA, 0x79, 0x05, 0x05, 0x2C, 0x77, 0x36, 0x6D,
        0x19, 0x72, 0xCC, 0xFC, 0xC1, 0xA2, 0x76, 0x10,
        0xCB, 0x54, 0x36, 0x65, 0xAA, 0x79, 0x8E, 0x20,
        0x94, 0x01, 0x28, 0xB9, 0x56, 0x7A, 0x7E, 0xDB,
        0x7A, 0x90, 0x04, 0x07, 0xC7, 0x0D, 0x35, 0x94,
        0x38, 0x43, 0x5E, 0x13, 0x96, 0x16, 0x08, 0xD5,
        0x52, 0xA9, 0x4C, 0x5C, 0xDA, 0x78, 0x59, 0x22,
        0x05, 0x09, 0xB4, 0x83, 0xC5, 0xC5, 0x2A, 0x21,
        0x0E, 0x9C, 0x81, 0x2B, 0xC0, 0xC2, 0x32, 0x8C,
        0xA0, 0x0E, 0x78, 0x9A, 0x56, 0xB2, 0x60, 0x6B,
        0x90, 0x29, 0x2E, 0x35, 0x43, 0xDA, 0xCA, 0xA2,
        0x43, 0x18, 0x41, 0xD6, 0x1A, 0x22, 0xCA, 0x90,
        0xC1, 0xCC, 0xF0, 0xB5, 0xB4, 0xE0, 0xA6, 0xF6,
        0x40, 0x53, 0x6D, 0x1A, 0x26, 0xAB, 0x5B, 0x8D,
        0x21, 0x51, 0x32, 0x79, 0x28, 0xCE, 0x02, 0x90,
        0x4C, 0xF1, 0xD1, 0x5E, 0x32, 0x78, 0x8A, 0x95,
        0xF6, 0x2D, 0x3C, 0x27, 0x0B, 0x6F, 0xA1, 0x50,
        0x8F, 0x97, 0xB9, 0x15, 0x5A, 0x27, 0x26, 0xD8,
        0x0A, 0x1A, 0xFA, 0x3C, 0x53, 0x87, 0xA2, 0x76,
        0xA4, 0xD0, 0x31, 0xA0, 0x8A, 0xBF, 0x4F, 0x2E,
        0x74, 0xF1, 0xA0, 0xBB, 0x8A, 0x0F, 0xD3, 0xCB
    };
    static const byte seed_512[WC_ML_KEM_ENC_RAND_SZ] = {
        0x6F, 0xF0, 0x2E, 0x1D, 0xC7, 0xFD, 0x91, 0x1B,
        0xEE, 0xE0, 0xC6, 0x92, 0xC8, 0xBD, 0x10, 0x0C,
        0x3E, 0x5C, 0x48, 0x96, 0x4D, 0x31, 0xDF, 0x92,
        0x99, 0x42, 0x18, 0xE8, 0x06, 0x64, 0xA6, 0xCA
    };
    static const byte c_512[WC_ML_KEM_512_CIPHER_TEXT_SIZE] = {
        0x19, 0xC5, 0x92, 0x50, 0x59, 0x07, 0xC2, 0x4C,
        0x5F, 0xA2, 0xEB, 0xFA, 0x93, 0x2D, 0x2C, 0xBB,
        0x48, 0xF3, 0xE4, 0x34, 0x0A, 0x28, 0xF7, 0xEB,
        0xA5, 0xD0, 0x68, 0xFC, 0xAC, 0xAB, 0xED, 0xF7,
        0x77, 0x84, 0xE2, 0xB2, 0x4D, 0x79, 0x61, 0x77,
        0x5F, 0x0B, 0xF1, 0xA9, 0x97, 0xAE, 0x8B, 0xA9,
        0xFC, 0x43, 0x11, 0xBE, 0x63, 0x71, 0x67, 0x79,
        0xC2, 0xB7, 0x88, 0xF8, 0x12, 0xCB, 0xB7, 0x8C,
        0x74, 0xE7, 0x51, 0x7E, 0x22, 0xE9, 0x10, 0xEF,
        0xF5, 0xF3, 0x8D, 0x44, 0x46, 0x9C, 0x50, 0xDE,
        0x16, 0x75, 0xAE, 0x19, 0x8F, 0xD6, 0xA2, 0x89,
        0xAE, 0x7E, 0x6C, 0x30, 0xA9, 0xD4, 0x35, 0x1B,
        0x3D, 0x1F, 0x4C, 0x36, 0xEF, 0xF9, 0xC6, 0x8D,
        0xA9, 0x1C, 0x40, 0xB8, 0x2D, 0xC9, 0xB2, 0x79,
        0x9A, 0x33, 0xA2, 0x6B, 0x60, 0xA4, 0xE7, 0x0D,
        0x71, 0x01, 0x86, 0x27, 0x79, 0x46, 0x9F, 0x3A,
        0x9D, 0xAE, 0xC8, 0xE3, 0xE8, 0xF8, 0xC6, 0xA1,
        0x6B, 0xF0, 0x92, 0xFB, 0xA5, 0x86, 0x61, 0x86,
        0xB8, 0xD2, 0x08, 0xFD, 0xEB, 0x27, 0x4A, 0xC1,
        0xF8, 0x29, 0x65, 0x9D, 0xC2, 0xBE, 0x4A, 0xC4,
        0xF3, 0x06, 0xCB, 0x55, 0x84, 0xBA, 0xD1, 0x93,
        0x6A, 0x92, 0xC9, 0xB7, 0x68, 0x19, 0x23, 0x42,
        0x81, 0xBB, 0x39, 0x58, 0x41, 0xC2, 0x57, 0x56,
        0x08, 0x6E, 0xA5, 0x64, 0xCA, 0x3E, 0x22, 0x7E,
        0x3D, 0x9F, 0x10, 0x52, 0xC0, 0x76, 0x6D, 0x2E,
        0xB7, 0x9A, 0x47, 0xC1, 0x50, 0x72, 0x1E, 0x0D,
        0xEA, 0x7C, 0x00, 0x69, 0xD5, 0x51, 0xB2, 0x64,
        0x80, 0x1B, 0x77, 0x27, 0xEC, 0xAF, 0x82, 0xEE,
        0xCB, 0x99, 0xA8, 0x76, 0xFD, 0xA0, 0x90, 0xBF,
        0x6C, 0x3F, 0xC6, 0xB1, 0x09, 0xF1, 0x70, 0x14,
        0x85, 0xF0, 0x3C, 0xE6, 0x62, 0x74, 0xB8, 0x43,
        0x5B, 0x0A, 0x01, 0x4C, 0xFB, 0x3E, 0x79, 0xCC,
        0xED, 0x67, 0x05, 0x7B, 0x5A, 0xE2, 0xAD, 0x7F,
        0x52, 0x79, 0xEB, 0x71, 0x49, 0x42, 0xE4, 0xC1,
        0xCC, 0xFF, 0x7E, 0x85, 0xC0, 0xDB, 0x43, 0xE5,
        0xD4, 0x12, 0x89, 0x20, 0x73, 0x63, 0xB4, 0x44,
        0xBB, 0x51, 0xBB, 0x8A, 0xB0, 0x37, 0x1E, 0x70,
        0xCB, 0xD5, 0x5F, 0x0F, 0x3D, 0xAD, 0x40, 0x3E,
        0x10, 0x51, 0x76, 0xE3, 0xE8, 0xA2, 0x25, 0xD8,
        0x4A, 0xC8, 0xBE, 0xE3, 0x8C, 0x82, 0x1E, 0xE0,
        0xF5, 0x47, 0x43, 0x11, 0x45, 0xDC, 0xB3, 0x13,
        0x92, 0x86, 0xAB, 0xB1, 0x17, 0x94, 0xA4, 0x3A,
        0x3C, 0x1B, 0x52, 0x29, 0xE4, 0xBC, 0xFE, 0x95,
        0x9C, 0x78, 0xAD, 0xAE, 0xE2, 0xD5, 0xF2, 0x49,
        0x7B, 0x5D, 0x24, 0xBC, 0x21, 0xFA, 0x03, 0xA9,
        0xA5, 0x8C, 0x24, 0x55, 0x37, 0x3E, 0xC8, 0x95,
        0x83, 0xE7, 0xE5, 0x88, 0xD7, 0xFE, 0x67, 0x99,
        0x1E, 0xE9, 0x37, 0x83, 0xED, 0x4A, 0x6F, 0x9E,
        0xEA, 0xE0, 0x4E, 0x64, 0xE2, 0xE1, 0xE0, 0xE6,
        0x99, 0xF6, 0xDC, 0x9C, 0x5D, 0x39, 0xEF, 0x92,
        0x78, 0xC9, 0x85, 0xE7, 0xFD, 0xF2, 0xA7, 0x64,
        0xFF, 0xD1, 0xA0, 0xB9, 0x57, 0x92, 0xAD, 0x68,
        0x1E, 0x93, 0x0D, 0x76, 0xDF, 0x4E, 0xFE, 0x5D,
        0x65, 0xDB, 0xBD, 0x0F, 0x14, 0x38, 0x48, 0x1E,
        0xD8, 0x33, 0xAD, 0x49, 0x46, 0xAD, 0x1C, 0x69,
        0xAD, 0x21, 0xDD, 0x7C, 0x86, 0x18, 0x57, 0x74,
        0x42, 0x6F, 0x3F, 0xCF, 0x53, 0xB5, 0x2A, 0xD4,
        0xB4, 0x0D, 0x22, 0x8C, 0xE1, 0x24, 0x07, 0x2F,
        0x59, 0x2C, 0x7D, 0xAA, 0x05, 0x7F, 0x17, 0xD7,
        0x90, 0xA5, 0xBD, 0x5B, 0x93, 0x83, 0x4D, 0x58,
        0xC0, 0x8C, 0x88, 0xDC, 0x8F, 0x0E, 0xF4, 0x88,
        0x15, 0x64, 0x25, 0xB7, 0x44, 0x65, 0x4E, 0xAC,
        0xA9, 0xD6, 0x48, 0x58, 0xA4, 0xD6, 0xCE, 0xB4,
        0x78, 0x79, 0x51, 0x94, 0xBF, 0xAD, 0xB1, 0x8D,
        0xC0, 0xEA, 0x05, 0x4F, 0x97, 0x71, 0x21, 0x5A,
        0xD3, 0xCB, 0x1F, 0xD0, 0x31, 0xD7, 0xBE, 0x45,
        0x98, 0x62, 0x19, 0x26, 0x47, 0x8D, 0x37, 0x5A,
        0x18, 0x45, 0xAA, 0x91, 0xD7, 0xC7, 0x33, 0xF8,
        0xF0, 0xE1, 0x88, 0xC8, 0x38, 0x96, 0xED, 0xF8,
        0x3B, 0x86, 0x46, 0xC9, 0x9E, 0x29, 0xC0, 0xDA,
        0x22, 0x90, 0xE7, 0x1C, 0x3D, 0x2E, 0x97, 0x07,
        0x20, 0xC9, 0x7B, 0x5B, 0x7F, 0x95, 0x04, 0x86,
        0x03, 0x3C, 0x6A, 0x25, 0x71, 0xDD, 0xF2, 0xBC,
        0xCD, 0xAB, 0xB2, 0xDF, 0xA5, 0xFC, 0xE4, 0xC3,
        0xA1, 0x88, 0x46, 0x06, 0x04, 0x1D, 0x18, 0x1C,
        0x72, 0x87, 0x94, 0xAE, 0x0E, 0x80, 0x6E, 0xCB,
        0x49, 0xAF, 0x16, 0x75, 0x6A, 0x4C, 0xE7, 0x3C,
        0x87, 0xBD, 0x42, 0x34, 0xE6, 0x0F, 0x05, 0x53,
        0x5F, 0xA5, 0x92, 0x9F, 0xD5, 0xA3, 0x44, 0x73,
        0x26, 0x64, 0x01, 0xF6, 0x3B, 0xBD, 0x6B, 0x90,
        0xE0, 0x03, 0x47, 0x2A, 0xC0, 0xCE, 0x88, 0xF1,
        0xB6, 0x66, 0x59, 0x72, 0x79, 0xD0, 0x56, 0xA6,
        0x32, 0xC8, 0xD6, 0xB7, 0x90, 0xFD, 0x41, 0x17,
        0x67, 0x84, 0x8A, 0x69, 0xE3, 0x7A, 0x8A, 0x83,
        0x9B, 0xC7, 0x66, 0xA0, 0x2C, 0xA2, 0xF6, 0x95,
        0xEC, 0x63, 0xF0, 0x56, 0xA4, 0xE2, 0xA1, 0x14,
        0xCA, 0xCF, 0x9F, 0xD9, 0x0D, 0x73, 0x0C, 0x97,
        0x0D, 0xB3, 0x87, 0xF6, 0xDE, 0x73, 0x39, 0x5F,
        0x70, 0x1A, 0x1D, 0x95, 0x3B, 0x2A, 0x89, 0xDD,
        0x7E, 0xDA, 0xD4, 0x39, 0xFC, 0x20, 0x5A, 0x54,
        0xA4, 0x81, 0xE8, 0x89, 0xB0, 0x98, 0xD5, 0x25,
        0x56, 0x70, 0xF0, 0x26, 0xB4, 0xA2, 0xBF, 0x02,
        0xD2, 0xBD, 0xDE, 0x87, 0xC7, 0x66, 0xB2, 0x5F,
        0xC5, 0xE0, 0xFD, 0x45, 0x37, 0x57, 0xE7, 0x56,
        0xD1, 0x8C, 0x8C, 0xD9, 0x12, 0xF9, 0xA7, 0x7F,
        0x8E, 0x6B, 0xF0, 0x20, 0x53, 0x74, 0xB4, 0x62
    };
    static const byte k_512[WC_ML_KEM_SS_SZ] = {
        0x0B, 0xF3, 0x23, 0x33, 0x8D, 0x6F, 0x0A, 0x21,
        0xD5, 0x51, 0x4B, 0x67, 0x3C, 0xD1, 0x0B, 0x71,
        0x4C, 0xE6, 0xE3, 0x6F, 0x35, 0xBC, 0xD1, 0xBF,
        0x54, 0x41, 0x96, 0x36, 0x8E, 0xE5, 0x1A, 0x13
    };
#endif
#ifndef WOLFSSL_NO_ML_KEM_768
    static const byte ek_768[WC_ML_KEM_768_PUBLIC_KEY_SIZE] = {
        0x89, 0xD2, 0xCB, 0x65, 0xF9, 0x4D, 0xCB, 0xFC,
        0x89, 0x0E, 0xFC, 0x7D, 0x0E, 0x5A, 0x7A, 0x38,
        0x34, 0x4D, 0x16, 0x41, 0xA3, 0xD0, 0xB0, 0x24,
        0xD5, 0x07, 0x97, 0xA5, 0xF2, 0x3C, 0x3A, 0x18,
        0xB3, 0x10, 0x1A, 0x12, 0x69, 0x06, 0x9F, 0x43,
        0xA8, 0x42, 0xBA, 0xCC, 0x09, 0x8A, 0x88, 0x21,
        0x27, 0x1C, 0x67, 0x3D, 0xB1, 0xBE, 0xB3, 0x30,
        0x34, 0xE4, 0xD7, 0x77, 0x4D, 0x16, 0x63, 0x5C,
        0x7C, 0x2C, 0x3C, 0x27, 0x63, 0x45, 0x35, 0x38,
        0xBC, 0x16, 0x32, 0xE1, 0x85, 0x15, 0x91, 0xA5,
        0x16, 0x42, 0x97, 0x4E, 0x59, 0x28, 0xAB, 0xB8,
        0xE5, 0x5F, 0xE5, 0x56, 0x12, 0xF9, 0xB1, 0x41,
        0xAF, 0xF0, 0x15, 0x54, 0x53, 0x94, 0xB2, 0x09,
        0x2E, 0x59, 0x09, 0x70, 0xEC, 0x29, 0xA7, 0xB7,
        0xE7, 0xAA, 0x1F, 0xB4, 0x49, 0x3B, 0xF7, 0xCB,
        0x73, 0x19, 0x06, 0xC2, 0xA5, 0xCB, 0x49, 0xE6,
        0x61, 0x48, 0x59, 0x06, 0x4E, 0x19, 0xB8, 0xFA,
        0x26, 0xAF, 0x51, 0xC4, 0x4B, 0x5E, 0x75, 0x35,
        0xBF, 0xDA, 0xC0, 0x72, 0xB6, 0x46, 0xD3, 0xEA,
        0x49, 0x0D, 0x27, 0x7F, 0x0D, 0x97, 0xCE, 0xD4,
        0x73, 0x95, 0xFE, 0xD9, 0x1E, 0x8F, 0x2B, 0xCE,
        0x0E, 0x3C, 0xA1, 0x22, 0xC2, 0x02, 0x5F, 0x74,
        0x06, 0x7A, 0xB9, 0x28, 0xA8, 0x22, 0xB3, 0x56,
        0x53, 0xA7, 0x4F, 0x06, 0x75, 0x76, 0x29, 0xAF,
        0xB1, 0xA1, 0xCA, 0xF2, 0x37, 0x10, 0x0E, 0xA9,
        0x35, 0xE7, 0x93, 0xC8, 0xF5, 0x8A, 0x71, 0xB3,
        0xD6, 0xAE, 0x2C, 0x86, 0x58, 0xB1, 0x01, 0x50,
        0xD4, 0xA3, 0x8F, 0x57, 0x2A, 0x0D, 0x49, 0xD2,
        0x8A, 0xE8, 0x94, 0x51, 0xD3, 0x38, 0x32, 0x6F,
        0xDB, 0x3B, 0x43, 0x50, 0x03, 0x6C, 0x10, 0x81,
        0x11, 0x77, 0x40, 0xED, 0xB8, 0x6B, 0x12, 0x08,
        0x1C, 0x5C, 0x12, 0x23, 0xDB, 0xB5, 0x66, 0x0D,
        0x5B, 0x3C, 0xB3, 0x78, 0x7D, 0x48, 0x18, 0x49,
        0x30, 0x4C, 0x68, 0xBE, 0x87, 0x54, 0x66, 0xF1,
        0x4E, 0xE5, 0x49, 0x5C, 0x2B, 0xD7, 0x95, 0xAE,
        0x41, 0x2D, 0x09, 0x00, 0x2D, 0x65, 0xB8, 0x71,
        0x9B, 0x90, 0xCB, 0xA3, 0x60, 0x3A, 0xC4, 0x95,
        0x8E, 0xA0, 0x3C, 0xC1, 0x38, 0xC8, 0x6F, 0x78,
        0x51, 0x59, 0x31, 0x25, 0x33, 0x47, 0x01, 0xB6,
        0x77, 0xF8, 0x2F, 0x49, 0x52, 0xA4, 0xC9, 0x3B,
        0x5B, 0x4C, 0x13, 0x4B, 0xB4, 0x2A, 0x85, 0x7F,
        0xD1, 0x5C, 0x65, 0x08, 0x64, 0xA6, 0xAA, 0x94,
        0xEB, 0x69, 0x1C, 0x0B, 0x69, 0x1B, 0xE4, 0x68,
        0x4C, 0x1F, 0x5B, 0x74, 0x90, 0x46, 0x7F, 0xC0,
        0x1B, 0x1D, 0x1F, 0xDA, 0x4D, 0xDA, 0x35, 0xC4,
        0xEC, 0xC2, 0x31, 0xBC, 0x73, 0xA6, 0xFE, 0xF4,
        0x2C, 0x99, 0xD3, 0x4E, 0xB8, 0x2A, 0x4D, 0x01,
        0x49, 0x87, 0xB3, 0xE3, 0x86, 0x91, 0x0C, 0x62,
        0x67, 0x9A, 0x11, 0x8F, 0x3C, 0x5B, 0xD9, 0xF4,
        0x67, 0xE4, 0x16, 0x20, 0x42, 0x42, 0x43, 0x57,
        0xDB, 0x92, 0xEF, 0x48, 0x4A, 0x4A, 0x17, 0x98,
        0xC1, 0x25, 0x7E, 0x87, 0x0A, 0x30, 0xCB, 0x20,
        0xAA, 0xA0, 0x33, 0x5D, 0x83, 0x31, 0x4F, 0xE0,
        0xAA, 0x7E, 0x63, 0xA8, 0x62, 0x64, 0x80, 0x41,
        0xA7, 0x2A, 0x63, 0x21, 0x52, 0x32, 0x20, 0xB1,
        0xAC, 0xE9, 0xBB, 0x70, 0x1B, 0x21, 0xAC, 0x12,
        0x53, 0xCB, 0x81, 0x2C, 0x15, 0x57, 0x5A, 0x90,
        0x85, 0xEA, 0xBE, 0xAD, 0xE7, 0x3A, 0x4A, 0xE7,
        0x6E, 0x6A, 0x7B, 0x15, 0x8A, 0x20, 0x58, 0x6D,
        0x78, 0xA5, 0xAC, 0x62, 0x0A, 0x5C, 0x9A, 0xBC,
        0xC9, 0xC0, 0x43, 0x35, 0x0A, 0x73, 0x65, 0x6B,
        0x0A, 0xBE, 0x82, 0x2D, 0xA5, 0xE0, 0xBA, 0x76,
        0x04, 0x5F, 0xAD, 0x75, 0x40, 0x1D, 0x7A, 0x3B,
        0x70, 0x37, 0x91, 0xB7, 0xE9, 0x92, 0x61, 0x71,
        0x0F, 0x86, 0xB7, 0x24, 0x21, 0xD2, 0x40, 0xA3,
        0x47, 0x63, 0x83, 0x77, 0x20, 0x5A, 0x15, 0x2C,
        0x79, 0x41, 0x30, 0xA4, 0xE0, 0x47, 0x74, 0x2B,
        0x88, 0x83, 0x03, 0xBD, 0xDC, 0x30, 0x91, 0x16,
        0x76, 0x4D, 0xE7, 0x42, 0x4C, 0xEB, 0xEA, 0x6D,
        0xB6, 0x53, 0x48, 0xAC, 0x53, 0x7E, 0x01, 0xA9,
        0xCC, 0x56, 0xEA, 0x66, 0x7D, 0x5A, 0xA8, 0x7A,
        0xC9, 0xAA, 0xA4, 0x31, 0x7D, 0x26, 0x2C, 0x10,
        0x14, 0x30, 0x50, 0xB8, 0xD0, 0x7A, 0x72, 0x8C,
        0xA6, 0x33, 0xC1, 0x3E, 0x46, 0x8A, 0xBC, 0xEA,
        0xD3, 0x72, 0xC7, 0x7B, 0x8E, 0xCF, 0x3B, 0x98,
        0x6B, 0x98, 0xC1, 0xE5, 0x58, 0x60, 0xB2, 0xB4,
        0x21, 0x67, 0x66, 0xAD, 0x87, 0x4C, 0x35, 0xED,
        0x72, 0x05, 0x06, 0x87, 0x39, 0x23, 0x02, 0x20,
        0xB5, 0xA2, 0x31, 0x7D, 0x10, 0x2C, 0x59, 0x83,
        0x56, 0xF1, 0x68, 0xAC, 0xBE, 0x80, 0x60, 0x8D,
        0xE4, 0xC9, 0xA7, 0x10, 0xB8, 0xDD, 0x07, 0x07,
        0x8C, 0xD7, 0xC6, 0x71, 0x05, 0x8A, 0xF1, 0xB0,
        0xB8, 0x30, 0x4A, 0x31, 0x4F, 0x7B, 0x29, 0xBE,
        0x78, 0xA9, 0x33, 0xC7, 0xB9, 0x29, 0x44, 0x24,
        0x95, 0x4A, 0x1B, 0xF8, 0xBC, 0x74, 0x5D, 0xE8,
        0x61, 0x98, 0x65, 0x9E, 0x0E, 0x12, 0x25, 0xA9,
        0x10, 0x72, 0x60, 0x74, 0x96, 0x9C, 0x39, 0xA9,
        0x7C, 0x19, 0x24, 0x06, 0x01, 0xA4, 0x6E, 0x01,
        0x3D, 0xCD, 0xCB, 0x67, 0x7A, 0x8C, 0xBD, 0x2C,
        0x95, 0xA4, 0x06, 0x29, 0xC2, 0x56, 0xF2, 0x4A,
        0x32, 0x89, 0x51, 0xDF, 0x57, 0x50, 0x2A, 0xB3,
        0x07, 0x72, 0xCC, 0x7E, 0x5B, 0x85, 0x00, 0x27,
        0xC8, 0x55, 0x17, 0x81, 0xCE, 0x49, 0x85, 0xBD,
        0xAC, 0xF6, 0xB8, 0x65, 0xC1, 0x04, 0xE8, 0xA4,
        0xBC, 0x65, 0xC4, 0x16, 0x94, 0xD4, 0x56, 0xB7,
        0x16, 0x9E, 0x45, 0xAB, 0x3D, 0x7A, 0xCA, 0xBE,
        0xAF, 0xE2, 0x3A, 0xD6, 0xA7, 0xB9, 0x4D, 0x19,
        0x79, 0xA2, 0xF4, 0xC1, 0xCA, 0xE7, 0xCD, 0x77,
        0xD6, 0x81, 0xD2, 0x90, 0xB5, 0xD8, 0xE4, 0x51,
        0xBF, 0xDC, 0xCC, 0xF5, 0x31, 0x0B, 0x9D, 0x12,
        0xA8, 0x8E, 0xC2, 0x9B, 0x10, 0x25, 0x5D, 0x5E,
        0x17, 0xA1, 0x92, 0x67, 0x0A, 0xA9, 0x73, 0x1C,
        0x5C, 0xA6, 0x7E, 0xC7, 0x84, 0xC5, 0x02, 0x78,
        0x1B, 0xE8, 0x52, 0x7D, 0x6F, 0xC0, 0x03, 0xC6,
        0x70, 0x1B, 0x36, 0x32, 0x28, 0x4B, 0x40, 0x30,
        0x7A, 0x52, 0x7C, 0x76, 0x20, 0x37, 0x7F, 0xEB,
        0x0B, 0x73, 0xF7, 0x22, 0xC9, 0xE3, 0xCD, 0x4D,
        0xEC, 0x64, 0x87, 0x6B, 0x93, 0xAB, 0x5B, 0x7C,
        0xFC, 0x4A, 0x65, 0x7F, 0x85, 0x2B, 0x65, 0x92,
        0x82, 0x86, 0x43, 0x84, 0xF4, 0x42, 0xB2, 0x2E,
        0x8A, 0x21, 0x10, 0x93, 0x87, 0xB8, 0xB4, 0x75,
        0x85, 0xFC, 0x68, 0x0D, 0x0B, 0xA4, 0x5C, 0x7A,
        0x8B, 0x1D, 0x72, 0x74, 0xBD, 0xA5, 0x78, 0x45,
        0xD1, 0x00, 0xD0, 0xF4, 0x2A, 0x3B, 0x74, 0x62,
        0x87, 0x73, 0x35, 0x1F, 0xD7, 0xAC, 0x30, 0x5B,
        0x24, 0x97, 0x63, 0x9B, 0xE9, 0x0B, 0x3F, 0x4F,
        0x71, 0xA6, 0xAA, 0x35, 0x61, 0xEE, 0xCC, 0x6A,
        0x69, 0x1B, 0xB5, 0xCB, 0x39, 0x14, 0xD8, 0x63,
        0x4C, 0xA1, 0xE1, 0xAF, 0x54, 0x3C, 0x04, 0x9A,
        0x8C, 0x6E, 0x86, 0x8C, 0x51, 0xF0, 0x42, 0x3B,
        0xD2, 0xD5, 0xAE, 0x09, 0xB7, 0x9E, 0x57, 0xC2,
        0x7F, 0x3F, 0xE3, 0xAE, 0x2B, 0x26, 0xA4, 0x41,
        0xBA, 0xBF, 0xC6, 0x71, 0x8C, 0xE8, 0xC0, 0x5B,
        0x4F, 0xE7, 0x93, 0xB9, 0x10, 0xB8, 0xFB, 0xCB,
        0xBE, 0x7F, 0x10, 0x13, 0x24, 0x2B, 0x40, 0xE0,
        0x51, 0x4D, 0x0B, 0xDC, 0x5C, 0x88, 0xBA, 0xC5,
        0x94, 0xC7, 0x94, 0xCE, 0x51, 0x22, 0xFB, 0xF3,
        0x48, 0x96, 0x81, 0x91, 0x47, 0xB9, 0x28, 0x38,
        0x15, 0x87, 0x96, 0x3B, 0x0B, 0x90, 0x03, 0x4A,
        0xA0, 0x7A, 0x10, 0xBE, 0x17, 0x6E, 0x01, 0xC8,
        0x0A, 0xD6, 0xA4, 0xB7, 0x1B, 0x10, 0xAF, 0x42,
        0x41, 0x40, 0x0A, 0x2A, 0x4C, 0xBB, 0xC0, 0x59,
        0x61, 0xA1, 0x5E, 0xC1, 0x47, 0x4E, 0xD5, 0x1A,
        0x3C, 0xC6, 0xD3, 0x58, 0x00, 0x67, 0x9A, 0x46,
        0x28, 0x09, 0xCA, 0xA3, 0xAB, 0x4F, 0x70, 0x94,
        0xCD, 0x66, 0x10, 0xB4, 0xA7, 0x00, 0xCB, 0xA9,
        0x39, 0xE7, 0xEA, 0xC9, 0x3E, 0x38, 0xC9, 0x97,
        0x55, 0x90, 0x87, 0x27, 0x61, 0x9E, 0xD7, 0x6A,
        0x34, 0xE5, 0x3C, 0x4F, 0xA2, 0x5B, 0xFC, 0x97,
        0x00, 0x82, 0x06, 0x69, 0x7D, 0xD1, 0x45, 0xE5,
        0xB9, 0x18, 0x8E, 0x5B, 0x01, 0x4E, 0x94, 0x16,
        0x81, 0xE1, 0x5F, 0xE3, 0xE1, 0x32, 0xB8, 0xA3,
        0x90, 0x34, 0x74, 0x14, 0x8B, 0xA2, 0x8B, 0x98,
        0x71, 0x11, 0xC9, 0xBC, 0xB3, 0x98, 0x9B, 0xBB,
        0xC6, 0x71, 0xC5, 0x81, 0xB4, 0x4A, 0x49, 0x28,
        0x45, 0xF2, 0x88, 0xE6, 0x21, 0x96, 0xE4, 0x71,
        0xFE, 0xD3, 0xC3, 0x9C, 0x1B, 0xBD, 0xDB, 0x08,
        0x37, 0xD0, 0xD4, 0x70, 0x6B, 0x09, 0x22, 0xC4
    };
    static const byte seed_768[WC_ML_KEM_ENC_RAND_SZ] = {
        0x2C, 0xE7, 0x4A, 0xD2, 0x91, 0x13, 0x35, 0x18,
        0xFE, 0x60, 0xC7, 0xDF, 0x5D, 0x25, 0x1B, 0x9D,
        0x82, 0xAD, 0xD4, 0x84, 0x62, 0xFF, 0x50, 0x5C,
        0x6E, 0x54, 0x7E, 0x94, 0x9E, 0x6B, 0x6B, 0xF7
    };
    static const byte c_768[WC_ML_KEM_768_CIPHER_TEXT_SIZE] = {
        0x56, 0xB4, 0x2D, 0x59, 0x3A, 0xAB, 0x8E, 0x87,
        0x73, 0xBD, 0x92, 0xD7, 0x6E, 0xAB, 0xDD, 0xF3,
        0xB1, 0x54, 0x6F, 0x83, 0x26, 0xF5, 0x7A, 0x7B,
        0x77, 0x37, 0x64, 0xB6, 0xC0, 0xDD, 0x30, 0x47,
        0x0F, 0x68, 0xDF, 0xF8, 0x2E, 0x0D, 0xCA, 0x92,
        0x50, 0x92, 0x74, 0xEC, 0xFE, 0x83, 0xA9, 0x54,
        0x73, 0x5F, 0xDE, 0x6E, 0x14, 0x67, 0x6D, 0xAA,
        0xA3, 0x68, 0x0C, 0x30, 0xD5, 0x24, 0xF4, 0xEF,
        0xA7, 0x9E, 0xD6, 0xA1, 0xF9, 0xED, 0x7E, 0x1C,
        0x00, 0x56, 0x0E, 0x86, 0x83, 0x53, 0x8C, 0x31,
        0x05, 0xAB, 0x93, 0x1B, 0xE0, 0xD2, 0xB2, 0x49,
        0xB3, 0x8C, 0xB9, 0xB1, 0x3A, 0xF5, 0xCE, 0xAF,
        0x78, 0x87, 0xA5, 0x9D, 0xBA, 0x16, 0x68, 0x8A,
        0x7F, 0x28, 0xDE, 0x0B, 0x14, 0xD1, 0x9F, 0x39,
        0x1E, 0xB4, 0x18, 0x32, 0xA5, 0x64, 0x79, 0x41,
        0x6C, 0xCF, 0x94, 0xE9, 0x97, 0x39, 0x0E, 0xD7,
        0x87, 0x8E, 0xEA, 0xFF, 0x49, 0x32, 0x8A, 0x70,
        0xE0, 0xAB, 0x5F, 0xCE, 0x6C, 0x63, 0xC0, 0x9B,
        0x35, 0xF4, 0xE4, 0x59, 0x94, 0xDE, 0x61, 0x5B,
        0x88, 0xBB, 0x72, 0x2F, 0x70, 0xE8, 0x7D, 0x2B,
        0xBD, 0x72, 0xAE, 0x71, 0xE1, 0xEE, 0x90, 0x08,
        0xE4, 0x59, 0xD8, 0xE7, 0x43, 0x03, 0x9A, 0x8D,
        0xDE, 0xB8, 0x74, 0xFC, 0xE5, 0x30, 0x1A, 0x2F,
        0x8C, 0x0E, 0xE8, 0xC2, 0xFE, 0xE7, 0xA4, 0xEE,
        0x68, 0xB5, 0xED, 0x6A, 0x6D, 0x9A, 0xB7, 0x4F,
        0x98, 0xBB, 0x3B, 0xA0, 0xFE, 0x89, 0xE8, 0x2B,
        0xD5, 0xA5, 0x25, 0xC5, 0xE8, 0x79, 0x0F, 0x81,
        0x8C, 0xCC, 0x60, 0x58, 0x77, 0xD4, 0x6C, 0x8B,
        0xDB, 0x5C, 0x33, 0x7B, 0x02, 0x5B, 0xB8, 0x40,
        0xFF, 0x47, 0x18, 0x96, 0xE4, 0x3B, 0xFA, 0x99,
        0xD7, 0x3D, 0xBE, 0x31, 0x80, 0x5C, 0x27, 0xA4,
        0x3E, 0x57, 0xF0, 0x61, 0x8B, 0x3A, 0xE5, 0x22,
        0xA4, 0x64, 0x4E, 0x0D, 0x4E, 0x4C, 0x1C, 0x54,
        0x84, 0x89, 0x43, 0x1B, 0xE5, 0x58, 0xF3, 0xBF,
        0xC5, 0x0E, 0x16, 0x61, 0x7E, 0x11, 0x0D, 0xD7,
        0xAF, 0x9A, 0x6F, 0xD8, 0x3E, 0x3F, 0xBB, 0x68,
        0xC3, 0x04, 0xD1, 0x5F, 0x6C, 0xB7, 0x00, 0xD6,
        0x1D, 0x7A, 0xA9, 0x15, 0xA6, 0x75, 0x1E, 0xA3,
        0xBA, 0x80, 0x22, 0x3E, 0x65, 0x41, 0x32, 0xA2,
        0x09, 0x99, 0xA4, 0x3B, 0xF4, 0x08, 0x59, 0x27,
        0x30, 0xB9, 0xA9, 0x49, 0x96, 0x36, 0xC0, 0x9F,
        0xA7, 0x29, 0xF9, 0xCB, 0x1F, 0x9D, 0x34, 0x42,
        0xF4, 0x73, 0x57, 0xA2, 0xB9, 0xCF, 0x15, 0xD3,
        0x10, 0x3B, 0x9B, 0xF3, 0x96, 0xC2, 0x30, 0x88,
        0xF1, 0x18, 0xED, 0xE3, 0x46, 0xB5, 0xC0, 0x38,
        0x91, 0xCF, 0xA5, 0xD5, 0x17, 0xCE, 0xF8, 0x47,
        0x13, 0x22, 0xE7, 0xE3, 0x10, 0x87, 0xC4, 0xB0,
        0x36, 0xAB, 0xAD, 0x78, 0x4B, 0xFF, 0x72, 0xA9,
        0xB1, 0x1F, 0xA1, 0x98, 0xFA, 0xCB, 0xCB, 0x91,
        0xF0, 0x67, 0xFE, 0xAF, 0x76, 0xFC, 0xFE, 0x53,
        0x27, 0xC1, 0x07, 0x0B, 0x3D, 0xA6, 0x98, 0x84,
        0x00, 0x75, 0x67, 0x60, 0xD2, 0xD1, 0xF0, 0x60,
        0x29, 0x8F, 0x16, 0x83, 0xD5, 0x1E, 0x36, 0x16,
        0xE9, 0x8C, 0x51, 0xC9, 0xC0, 0x3A, 0xA4, 0x2F,
        0x2E, 0x63, 0x36, 0x51, 0xA4, 0x7A, 0xD3, 0xCC,
        0x2A, 0xB4, 0xA8, 0x52, 0xAE, 0x0C, 0x4B, 0x04,
        0xB4, 0xE1, 0xC3, 0xDD, 0x94, 0x44, 0x45, 0xA2,
        0xB1, 0x2B, 0x4F, 0x42, 0xA6, 0x43, 0x51, 0x05,
        0xC0, 0x41, 0x22, 0xFC, 0x35, 0x87, 0xAF, 0xE4,
        0x09, 0xA0, 0x0B, 0x30, 0x8D, 0x63, 0xC5, 0xDD,
        0x81, 0x63, 0x65, 0x45, 0x04, 0xEE, 0xDB, 0xB7,
        0xB5, 0x32, 0x95, 0x77, 0xC3, 0x5F, 0xBE, 0xB3,
        0xF4, 0x63, 0x87, 0x2C, 0xAC, 0x28, 0x14, 0x2B,
        0x3C, 0x12, 0xA7, 0x40, 0xEC, 0x6E, 0xA7, 0xCE,
        0x9A, 0xD7, 0x8C, 0x6F, 0xC8, 0xFE, 0x1B, 0x4D,
        0xF5, 0xFC, 0x55, 0xC1, 0x66, 0x7F, 0x31, 0xF2,
        0x31, 0x2D, 0xA0, 0x77, 0x99, 0xDC, 0x87, 0x0A,
        0x47, 0x86, 0x08, 0x54, 0x9F, 0xED, 0xAF, 0xE0,
        0x21, 0xF1, 0xCF, 0x29, 0x84, 0x18, 0x03, 0x64,
        0xE9, 0x0A, 0xD9, 0x8D, 0x84, 0x56, 0x52, 0xAA,
        0x3C, 0xDD, 0x7A, 0x8E, 0xB0, 0x9F, 0x5E, 0x51,
        0x42, 0x3F, 0xAB, 0x42, 0xA7, 0xB7, 0xBB, 0x4D,
        0x51, 0x48, 0x64, 0xBE, 0x8D, 0x71, 0x29, 0x7E,
        0x9C, 0x3B, 0x17, 0xA9, 0x93, 0xF0, 0xAE, 0x62,
        0xE8, 0xEF, 0x52, 0x63, 0x7B, 0xD1, 0xB8, 0x85,
        0xBD, 0x9B, 0x6A, 0xB7, 0x27, 0x85, 0x4D, 0x70,
        0x3D, 0x8D, 0xC4, 0x78, 0xF9, 0x6C, 0xB8, 0x1F,
        0xCE, 0x4C, 0x60, 0x38, 0x3A, 0xC0, 0x1F, 0xCF,
        0x0F, 0x97, 0x1D, 0x4C, 0x8F, 0x35, 0x2B, 0x7A,
        0x82, 0xE2, 0x18, 0x65, 0x2F, 0x2C, 0x10, 0x6C,
        0xA9, 0x2A, 0xE6, 0x86, 0xBA, 0xCF, 0xCE, 0xF5,
        0xD3, 0x27, 0x34, 0x7A, 0x97, 0xA9, 0xB3, 0x75,
        0xD6, 0x73, 0x41, 0x55, 0x2B, 0xC2, 0xC5, 0x38,
        0x77, 0x8E, 0x0F, 0x98, 0x01, 0x82, 0x3C, 0xCD,
        0xFC, 0xD1, 0xEA, 0xAD, 0xED, 0x55, 0xB1, 0x8C,
        0x97, 0x57, 0xE3, 0xF2, 0x12, 0xB2, 0x88, 0x9D,
        0x38, 0x57, 0xDB, 0x51, 0xF9, 0x81, 0xD1, 0x61,
        0x85, 0xFD, 0x0F, 0x90, 0x08, 0x53, 0xA7, 0x50,
        0x05, 0xE3, 0x02, 0x0A, 0x8B, 0x95, 0xB7, 0xD8,
        0xF2, 0xF2, 0x63, 0x1C, 0x70, 0xD7, 0x8A, 0x95,
        0x7C, 0x7A, 0x62, 0xE1, 0xB3, 0x71, 0x90, 0x70,
        0xAC, 0xD1, 0xFD, 0x48, 0x0C, 0x25, 0xB8, 0x38,
        0x47, 0xDA, 0x02, 0x7B, 0x6E, 0xBB, 0xC2, 0xEE,
        0xC2, 0xDF, 0x22, 0xC8, 0x7F, 0x9B, 0x46, 0xD5,
        0xD7, 0xBA, 0xF1, 0x56, 0xB5, 0x3C, 0xEE, 0x92,
        0x95, 0x72, 0xB9, 0x2C, 0x47, 0x84, 0xC4, 0xE8,
        0x29, 0xF3, 0x44, 0x6A, 0x1F, 0xFE, 0x47, 0xF9,
        0x9D, 0xEC, 0xD0, 0x43, 0x60, 0x29, 0xDD, 0xEB,
        0xD3, 0xED, 0x8E, 0x87, 0xE5, 0xE7, 0x3D, 0x12,
        0x3D, 0xBE, 0x8A, 0x4D, 0xDA, 0xCF, 0x2A, 0xBD,
        0xE8, 0x7F, 0x33, 0xAE, 0x2B, 0x62, 0x1C, 0x0E,
        0xC5, 0xD5, 0xCA, 0xD1, 0x25, 0x9D, 0xEE, 0xC2,
        0xAE, 0xFF, 0x60, 0x88, 0xF0, 0x4F, 0x27, 0xA2,
        0x03, 0x38, 0xB5, 0x76, 0x25, 0x43, 0xE5, 0x10,
        0x08, 0x99, 0xA4, 0xCB, 0xFB, 0x7B, 0x3C, 0xA4,
        0x56, 0xB3, 0xA1, 0x9B, 0x83, 0xA4, 0xC4, 0x32,
        0x23, 0x0C, 0x23, 0xE1, 0xC7, 0xF1, 0x07, 0xC4,
        0xCB, 0x11, 0x21, 0x52, 0xF1, 0xC0, 0xF3, 0x0D,
        0xA0, 0xBB, 0x33, 0xF4, 0xF1, 0x1F, 0x47, 0xEE,
        0xA4, 0x38, 0x72, 0xBA, 0xFA, 0x84, 0xAE, 0x22,
        0x25, 0x6D, 0x70, 0x8E, 0x06, 0x04, 0xDA, 0xDE,
        0x4B, 0x2A, 0x4D, 0xDE, 0x8C, 0xCC, 0xF1, 0x19,
        0x30, 0xE1, 0x35, 0x53, 0x93, 0x4A, 0xE3, 0xEC,
        0xE5, 0x2F, 0x3D, 0x7C, 0xCC, 0x00, 0x28, 0x73,
        0x77, 0x87, 0x9F, 0xE6, 0xB8, 0xEC, 0xE7, 0xEF,
        0x79, 0x42, 0x35, 0x07, 0xC9, 0xDA, 0x33, 0x95,
        0x59, 0xC2, 0x0D, 0xE1, 0xC5, 0x19, 0x55, 0x99,
        0x9B, 0xAE, 0x47, 0x40, 0x1D, 0xC3, 0xCD, 0xFA,
        0xA1, 0xB2, 0x56, 0xD0, 0x9C, 0x7D, 0xB9, 0xFC,
        0x86, 0x98, 0xBF, 0xCE, 0xFA, 0x73, 0x02, 0xD5,
        0x6F, 0xBC, 0xDE, 0x1F, 0xBA, 0xAA, 0x1C, 0x65,
        0x34, 0x54, 0xE6, 0xFD, 0x3D, 0x84, 0xE4, 0xF7,
        0x9A, 0x93, 0x1C, 0x68, 0x1C, 0xBB, 0x6C, 0xB4,
        0x62, 0xB1, 0x0D, 0xAE, 0x11, 0x2B, 0xDF, 0xB7,
        0xF6, 0x5C, 0x7F, 0xDF, 0x6E, 0x5F, 0xC5, 0x94,
        0xEC, 0x3A, 0x47, 0x4A, 0x94, 0xBD, 0x97, 0xE6,
        0xEC, 0x81, 0xF7, 0x1C, 0x23, 0x0B, 0xF7, 0x0C,
        0xA0, 0xF1, 0x3C, 0xE3, 0xDF, 0xFB, 0xD9, 0xFF,
        0x98, 0x04, 0xEF, 0xD8, 0xF3, 0x7A, 0x4D, 0x36,
        0x29, 0xB4, 0x3A, 0x8F, 0x55, 0x54, 0x4E, 0xBC,
        0x5A, 0xC0, 0xAB, 0xD9, 0xA3, 0x3D, 0x79, 0x69,
        0x90, 0x68, 0x34, 0x6A, 0x0F, 0x1A, 0x3A, 0x96,
        0xE1, 0x15, 0xA5, 0xD8, 0x0B, 0xE1, 0x65, 0xB5,
        0x62, 0xD0, 0x82, 0x98, 0x4D, 0x5A, 0xAC, 0xC3,
        0xA2, 0x30, 0x19, 0x81, 0xA6, 0x41, 0x8F, 0x8B,
        0xA7, 0xD7, 0xB0, 0xD7, 0xCA, 0x58, 0x75, 0xC6
    };
    static const byte k_768[WC_ML_KEM_SS_SZ] = {
        0x26, 0x96, 0xD2, 0x8E, 0x9C, 0x61, 0xC2, 0xA0,
        0x1C, 0xE9, 0xB1, 0x60, 0x8D, 0xCB, 0x9D, 0x29,
        0x27, 0x85, 0xA0, 0xCD, 0x58, 0xEF, 0xB7, 0xFE,
        0x13, 0xB1, 0xDE, 0x95, 0xF0, 0xDB, 0x55, 0xB3
    };
#endif
#ifndef WOLFSSL_NO_ML_KEM_1024
    static const byte ek_1024[WC_ML_KEM_1024_PUBLIC_KEY_SIZE] = {
        0x30, 0x7A, 0x4C, 0xEA, 0x41, 0x48, 0x21, 0x9B,
        0x95, 0x8E, 0xA0, 0xB7, 0x88, 0x66, 0x59, 0x23,
        0x5A, 0x4D, 0x19, 0x80, 0xB1, 0x92, 0x61, 0x08,
        0x47, 0xD8, 0x6E, 0xF3, 0x27, 0x39, 0xF9, 0x4C,
        0x3B, 0x44, 0x6C, 0x4D, 0x81, 0xD8, 0x9B, 0x8B,
        0x42, 0x2A, 0x9D, 0x07, 0x9C, 0x88, 0xB1, 0x1A,
        0xCA, 0xF3, 0x21, 0xB0, 0x14, 0x29, 0x4E, 0x18,
        0xB2, 0x96, 0xE5, 0x2F, 0x3F, 0x74, 0x4C, 0xF9,
        0x63, 0x4A, 0x4F, 0xB0, 0x1D, 0xB0, 0xD9, 0x9E,
        0xF2, 0x0A, 0x63, 0x3A, 0x55, 0x2E, 0x76, 0xA0,
        0x58, 0x5C, 0x61, 0x09, 0xF0, 0x18, 0x76, 0x8B,
        0x76, 0x3A, 0xF3, 0x67, 0x8B, 0x47, 0x80, 0x08,
        0x9C, 0x13, 0x42, 0xB9, 0x69, 0x07, 0xA2, 0x9A,
        0x1C, 0x11, 0x52, 0x1C, 0x74, 0x4C, 0x27, 0x97,
        0xD0, 0xBF, 0x2B, 0x9C, 0xCD, 0xCA, 0x61, 0x46,
        0x72, 0xB4, 0x50, 0x76, 0x77, 0x3F, 0x45, 0x8A,
        0x31, 0xEF, 0x86, 0x9B, 0xE1, 0xEB, 0x2E, 0xFE,
        0xB5, 0x0D, 0x0E, 0x37, 0x49, 0x5D, 0xC5, 0xCA,
        0x55, 0xE0, 0x75, 0x28, 0x93, 0x4F, 0x62, 0x93,
        0xC4, 0x16, 0x80, 0x27, 0xD0, 0xE5, 0x3D, 0x07,
        0xFA, 0xCC, 0x66, 0x30, 0xCB, 0x08, 0x19, 0x7E,
        0x53, 0xFB, 0x19, 0x3A, 0x17, 0x11, 0x35, 0xDC,
        0x8A, 0xD9, 0x97, 0x94, 0x02, 0xA7, 0x1B, 0x69,
        0x26, 0xBC, 0xDC, 0xDC, 0x47, 0xB9, 0x34, 0x01,
        0x91, 0x0A, 0x5F, 0xCC, 0x1A, 0x81, 0x3B, 0x68,
        0x2B, 0x09, 0xBA, 0x7A, 0x72, 0xD2, 0x48, 0x6D,
        0x6C, 0x79, 0x95, 0x16, 0x46, 0x5C, 0x14, 0x72,
        0x9B, 0x26, 0x94, 0x9B, 0x0B, 0x7C, 0xBC, 0x7C,
        0x64, 0x0F, 0x26, 0x7F, 0xED, 0x80, 0xB1, 0x62,
        0xC5, 0x1F, 0xD8, 0xE0, 0x92, 0x27, 0xC1, 0x01,
        0xD5, 0x05, 0xA8, 0xFA, 0xE8, 0xA2, 0xD7, 0x05,
        0x4E, 0x28, 0xA7, 0x8B, 0xA8, 0x75, 0x0D, 0xEC,
        0xF9, 0x05, 0x7C, 0x83, 0x97, 0x9F, 0x7A, 0xBB,
        0x08, 0x49, 0x45, 0x64, 0x80, 0x06, 0xC5, 0xB2,
        0x88, 0x04, 0xF3, 0x4E, 0x73, 0xB2, 0x38, 0x11,
        0x1A, 0x65, 0xA1, 0xF5, 0x00, 0xB1, 0xCC, 0x60,
        0x6A, 0x84, 0x8F, 0x28, 0x59, 0x07, 0x0B, 0xEB,
        0xA7, 0x57, 0x31, 0x79, 0xF3, 0x61, 0x49, 0xCF,
        0x58, 0x01, 0xBF, 0x89, 0xA1, 0xC3, 0x8C, 0xC2,
        0x78, 0x41, 0x55, 0x28, 0xD0, 0x3B, 0xDB, 0x94,
        0x3F, 0x96, 0x28, 0x0C, 0x8C, 0xC5, 0x20, 0x42,
        0xD9, 0xB9, 0x1F, 0xAA, 0x9D, 0x6E, 0xA7, 0xBC,
        0xBB, 0x7A, 0xB1, 0x89, 0x7A, 0x32, 0x66, 0x96,
        0x6F, 0x78, 0x39, 0x34, 0x26, 0xC7, 0x6D, 0x8A,
        0x49, 0x57, 0x8B, 0x98, 0xB1, 0x59, 0xEB, 0xB4,
        0x6E, 0xE0, 0xA8, 0x83, 0xA2, 0x70, 0xD8, 0x05,
        0x7C, 0xD0, 0x23, 0x1C, 0x86, 0x90, 0x6A, 0x91,
        0xDB, 0xBA, 0xDE, 0x6B, 0x24, 0x69, 0x58, 0x1E,
        0x2B, 0xCA, 0x2F, 0xEA, 0x83, 0x89, 0xF7, 0xC7,
        0x4B, 0xCD, 0x70, 0x96, 0x1E, 0xA5, 0xB9, 0x34,
        0xFB, 0xCF, 0x9A, 0x65, 0x90, 0xBF, 0x86, 0xB8,
        0xDB, 0x54, 0x88, 0x54, 0xD9, 0xA3, 0xFB, 0x30,
        0x11, 0x04, 0x33, 0xBD, 0x7A, 0x1B, 0x65, 0x9C,
        0xA8, 0x56, 0x80, 0x85, 0x63, 0x92, 0x37, 0xB3,
        0xBD, 0xC3, 0x7B, 0x7F, 0xA7, 0x16, 0xD4, 0x82,
        0xA2, 0x5B, 0x54, 0x10, 0x6B, 0x3A, 0x8F, 0x54,
        0xD3, 0xAA, 0x99, 0xB5, 0x12, 0x3D, 0xA9, 0x60,
        0x66, 0x90, 0x45, 0x92, 0xF3, 0xA5, 0x4E, 0xE2,
        0x3A, 0x79, 0x81, 0xAB, 0x60, 0x8A, 0x2F, 0x44,
        0x13, 0xCC, 0x65, 0x89, 0x46, 0xC6, 0xD7, 0x78,
        0x0E, 0xA7, 0x65, 0x64, 0x4B, 0x3C, 0xC0, 0x6C,
        0x70, 0x03, 0x4A, 0xB4, 0xEB, 0x35, 0x19, 0x12,
        0xE7, 0x71, 0x5B, 0x56, 0x75, 0x5D, 0x09, 0x02,
        0x15, 0x71, 0xBF, 0x34, 0x0A, 0xB9, 0x25, 0x98,
        0xA2, 0x4E, 0x81, 0x18, 0x93, 0x19, 0x5B, 0x96,
        0xA1, 0x62, 0x9F, 0x80, 0x41, 0xF5, 0x86, 0x58,
        0x43, 0x15, 0x61, 0xFC, 0x0A, 0xB1, 0x52, 0x92,
        0xB9, 0x13, 0xEC, 0x47, 0x3F, 0x04, 0x47, 0x9B,
        0xC1, 0x45, 0xCD, 0x4C, 0x56, 0x3A, 0x28, 0x62,
        0x35, 0x64, 0x6C, 0xD3, 0x05, 0xA9, 0xBE, 0x10,
        0x14, 0xE2, 0xC7, 0xB1, 0x30, 0xC3, 0x3E, 0xB7,
        0x7C, 0xC4, 0xA0, 0xD9, 0x78, 0x6B, 0xD6, 0xBC,
        0x2A, 0x95, 0x4B, 0xF3, 0x00, 0x57, 0x78, 0xF8,
        0x91, 0x7C, 0xE1, 0x37, 0x89, 0xBB, 0xB9, 0x62,
        0x80, 0x78, 0x58, 0xB6, 0x77, 0x31, 0x57, 0x2B,
        0x6D, 0x3C, 0x9B, 0x4B, 0x52, 0x06, 0xFA, 0xC9,
        0xA7, 0xC8, 0x96, 0x16, 0x98, 0xD8, 0x83, 0x24,
        0xA9, 0x15, 0x18, 0x68, 0x99, 0xB2, 0x99, 0x23,
        0xF0, 0x84, 0x42, 0xA3, 0xD3, 0x86, 0xBD, 0x41,
        0x6B, 0xCC, 0x9A, 0x10, 0x01, 0x64, 0xC9, 0x30,
        0xEC, 0x35, 0xEA, 0xFB, 0x6A, 0xB3, 0x58, 0x51,
        0xB6, 0xC8, 0xCE, 0x63, 0x77, 0x36, 0x6A, 0x17,
        0x5F, 0x3D, 0x75, 0x29, 0x8C, 0x51, 0x8D, 0x44,
        0x89, 0x89, 0x33, 0xF5, 0x3D, 0xEE, 0x61, 0x71,
        0x45, 0x09, 0x33, 0x79, 0xC4, 0x65, 0x9F, 0x68,
        0x58, 0x3B, 0x2B, 0x28, 0x12, 0x26, 0x66, 0xBE,
        0xC5, 0x78, 0x38, 0x99, 0x1F, 0xF1, 0x6C, 0x36,
        0x8D, 0xD2, 0x2C, 0x36, 0xE7, 0x80, 0xC9, 0x1A,
        0x35, 0x82, 0xE2, 0x5E, 0x19, 0x79, 0x4C, 0x6B,
        0xF2, 0xAB, 0x42, 0x45, 0x8A, 0x8D, 0xD7, 0x70,
        0x5D, 0xE2, 0xC2, 0xAA, 0x20, 0xC0, 0x54, 0xE8,
        0x4B, 0x3E, 0xF3, 0x50, 0x32, 0x79, 0x86, 0x26,
        0xC2, 0x48, 0x26, 0x32, 0x53, 0xA7, 0x1A, 0x11,
        0x94, 0x35, 0x71, 0x34, 0x0A, 0x97, 0x8C, 0xD0,
        0xA6, 0x02, 0xE4, 0x7D, 0xEE, 0x54, 0x0A, 0x88,
        0x14, 0xBA, 0x06, 0xF3, 0x14, 0x14, 0x79, 0x7C,
        0xDF, 0x60, 0x49, 0x58, 0x23, 0x61, 0xBB, 0xAB,
        0xA3, 0x87, 0xA8, 0x3D, 0x89, 0x91, 0x3F, 0xE4,
        0xC0, 0xC1, 0x12, 0xB9, 0x56, 0x21, 0xA4, 0xBD,
        0xA8, 0x12, 0x3A, 0x14, 0xD1, 0xA8, 0x42, 0xFB,
        0x57, 0xB8, 0x3A, 0x4F, 0xBA, 0xF3, 0x3A, 0x8E,
        0x55, 0x22, 0x38, 0xA5, 0x96, 0xAA, 0xE7, 0xA1,
        0x50, 0xD7, 0x5D, 0xA6, 0x48, 0xBC, 0x44, 0x64,
        0x49, 0x77, 0xBA, 0x1F, 0x87, 0xA4, 0xC6, 0x8A,
        0x8C, 0x4B, 0xD2, 0x45, 0xB7, 0xD0, 0x07, 0x21,
        0xF7, 0xD6, 0x4E, 0x82, 0x2B, 0x08, 0x5B, 0x90,
        0x13, 0x12, 0xEC, 0x37, 0xA8, 0x16, 0x98, 0x02,
        0x16, 0x0C, 0xCE, 0x11, 0x60, 0xF0, 0x10, 0xBE,
        0x8C, 0xBC, 0xAC, 0xE8, 0xE7, 0xB0, 0x05, 0xD7,
        0x83, 0x92, 0x34, 0xA7, 0x07, 0x86, 0x83, 0x09,
        0xD0, 0x37, 0x84, 0xB4, 0x27, 0x3B, 0x1C, 0x8A,
        0x16, 0x01, 0x33, 0xED, 0x29, 0x81, 0x84, 0x70,
        0x46, 0x25, 0xF2, 0x9C, 0xFA, 0x08, 0x6D, 0x13,
        0x26, 0x3E, 0xE5, 0x89, 0x91, 0x23, 0xC5, 0x96,
        0xBA, 0x78, 0x8E, 0x5C, 0x54, 0xA8, 0xE9, 0xBA,
        0x82, 0x9B, 0x8A, 0x9D, 0x90, 0x4B, 0xC4, 0xBC,
        0x0B, 0xBE, 0xA7, 0x6B, 0xC5, 0x3F, 0xF8, 0x11,
        0x21, 0x45, 0x98, 0x47, 0x2C, 0x9C, 0x20, 0x2B,
        0x73, 0xEF, 0xF0, 0x35, 0xDC, 0x09, 0x70, 0x3A,
        0xF7, 0xBF, 0x1B, 0xAB, 0xAA, 0xC7, 0x31, 0x93,
        0xCB, 0x46, 0x11, 0x7A, 0x7C, 0x94, 0x92, 0xA4,
        0x3F, 0xC9, 0x57, 0x89, 0xA9, 0x24, 0xC5, 0x91,
        0x27, 0x87, 0xB2, 0xE2, 0x09, 0x0E, 0xBB, 0xCF,
        0xD3, 0x79, 0x62, 0x21, 0xF0, 0x6D, 0xEB, 0xF9,
        0xCF, 0x70, 0xE0, 0x56, 0xB8, 0xB9, 0x16, 0x1D,
        0x63, 0x47, 0xF4, 0x73, 0x35, 0xF3, 0xE1, 0x77,
        0x6D, 0xA4, 0xBB, 0x87, 0xC1, 0x5C, 0xC8, 0x26,
        0x14, 0x6F, 0xF0, 0x24, 0x9A, 0x41, 0x3B, 0x45,
        0xAA, 0x93, 0xA8, 0x05, 0x19, 0x6E, 0xA4, 0x53,
        0x11, 0x4B, 0x52, 0x4E, 0x31, 0x0A, 0xED, 0xAA,
        0x46, 0xE3, 0xB9, 0x96, 0x42, 0x36, 0x87, 0x82,
        0x56, 0x6D, 0x04, 0x9A, 0x72, 0x6D, 0x6C, 0xCA,
        0x91, 0x09, 0x93, 0xAE, 0xD6, 0x21, 0xD0, 0x14,
        0x9E, 0xA5, 0x88, 0xA9, 0xAB, 0xD9, 0x09, 0xDB,
        0xB6, 0x9A, 0xA2, 0x28, 0x29, 0xD9, 0xB8, 0x3A,
        0xDA, 0x22, 0x09, 0xA6, 0xC2, 0x65, 0x9F, 0x21,
        0x69, 0xD6, 0x68, 0xB9, 0x31, 0x48, 0x42, 0xC6,
        0xE2, 0x2A, 0x74, 0x95, 0x8B, 0x4C, 0x25, 0xBB,
        0xDC, 0xD2, 0x93, 0xD9, 0x9C, 0xB6, 0x09, 0xD8,
        0x66, 0x74, 0x9A, 0x48, 0x5D, 0xFB, 0x56, 0x02,
        0x48, 0x83, 0xCF, 0x54, 0x65, 0xDB, 0xA0, 0x36,
        0x32, 0x06, 0x58, 0x7F, 0x45, 0x59, 0x7F, 0x89,
        0x00, 0x2F, 0xB8, 0x60, 0x72, 0x32, 0x13, 0x8E,
        0x03, 0xB2, 0xA8, 0x94, 0x52, 0x5F, 0x26, 0x53,
        0x70, 0x05, 0x4B, 0x48, 0x86, 0x36, 0x14, 0x47,
        0x2B, 0x95, 0xD0, 0xA2, 0x30, 0x34, 0x42, 0xE3,
        0x78, 0xB0, 0xDD, 0x1C, 0x75, 0xAC, 0xBA, 0xB9,
        0x71, 0xA9, 0xA8, 0xD1, 0x28, 0x1C, 0x79, 0x61,
        0x3A, 0xCE, 0xC6, 0x93, 0x3C, 0x37, 0x7B, 0x3C,
        0x57, 0x8C, 0x2A, 0x61, 0xA1, 0xEC, 0x18, 0x1B,
        0x10, 0x12, 0x97, 0xA3, 0x7C, 0xC5, 0x19, 0x7B,
        0x29, 0x42, 0xF6, 0xA0, 0xE4, 0x70, 0x4C, 0x0E,
        0xC6, 0x35, 0x40, 0x48, 0x1B, 0x9F, 0x15, 0x9D,
        0xC2, 0x55, 0xB5, 0x9B, 0xB5, 0x5D, 0xF4, 0x96,
        0xAE, 0x54, 0x21, 0x7B, 0x76, 0x89, 0xBD, 0x51,
        0xDB, 0xA0, 0x38, 0x3A, 0x3D, 0x72, 0xD8, 0x52,
        0xFF, 0xCA, 0x76, 0xDF, 0x05, 0xB6, 0x6E, 0xEC,
        0xCB, 0xD4, 0x7B, 0xC5, 0x30, 0x40, 0x81, 0x76,
        0x28, 0xC7, 0x1E, 0x36, 0x1D, 0x6A, 0xF8, 0x89,
        0x08, 0x49, 0x16, 0xB4, 0x08, 0xA4, 0x66, 0xC9,
        0x6E, 0x70, 0x86, 0xC4, 0xA6, 0x0A, 0x10, 0xFC,
        0xF7, 0x53, 0x7B, 0xB9, 0x4A, 0xFB, 0xCC, 0x7D,
        0x43, 0x75, 0x90, 0x91, 0x9C, 0x28, 0x65, 0x0C,
        0x4F, 0x23, 0x68, 0x25, 0x92, 0x26, 0xA9, 0xBF,
        0xDA, 0x3A, 0x3A, 0x0B, 0xA1, 0xB5, 0x08, 0x7D,
        0x9D, 0x76, 0x44, 0x2F, 0xD7, 0x86, 0xC6, 0xF8,
        0x1C, 0x68, 0xC0, 0x36, 0x0D, 0x71, 0x94, 0xD7,
        0x07, 0x2C, 0x45, 0x33, 0xAE, 0xA8, 0x6C, 0x2D,
        0x1F, 0x8C, 0x0A, 0x27, 0x69, 0x60, 0x66, 0xF6,
        0xCF, 0xD1, 0x10, 0x03, 0xF7, 0x97, 0x27, 0x0B,
        0x32, 0x38, 0x97, 0x13, 0xCF, 0xFA, 0x09, 0x3D,
        0x99, 0x1B, 0x63, 0x84, 0x4C, 0x38, 0x5E, 0x72,
        0x27, 0x7F, 0x16, 0x6F, 0x5A, 0x39, 0x34, 0xD6,
        0xBB, 0x89, 0xA4, 0x78, 0x8D, 0xE2, 0x83, 0x21,
        0xDE, 0xFC, 0x74, 0x57, 0xAB, 0x48, 0x4B, 0xD3,
        0x09, 0x86, 0xDC, 0x1D, 0xAB, 0x30, 0x08, 0xCD,
        0x7B, 0x22, 0xF6, 0x97, 0x02, 0xFA, 0xBB, 0x9A,
        0x10, 0x45, 0x40, 0x7D, 0xA4, 0x79, 0x1C, 0x35,
        0x90, 0xFF, 0x59, 0x9D, 0x81, 0xD6, 0x88, 0xCF,
        0xA7, 0xCC, 0x12, 0xA6, 0x8C, 0x50, 0xF5, 0x1A,
        0x10, 0x09, 0x41, 0x1B, 0x44, 0x85, 0x0F, 0x90,
        0x15, 0xDC, 0x84, 0xA9, 0x3B, 0x17, 0xC7, 0xA2,
        0x07, 0x55, 0x2C, 0x66, 0x1E, 0xA9, 0x83, 0x8E,
        0x31, 0xB9, 0x5E, 0xAD, 0x54, 0x62, 0x48, 0xE5,
        0x6B, 0xE7, 0xA5, 0x13, 0x05, 0x05, 0x26, 0x87,
        0x71, 0x19, 0x98, 0x80, 0xA1, 0x41, 0x77, 0x1A,
        0x9E, 0x47, 0xAC, 0xFE, 0xD5, 0x90, 0xCB, 0x3A,
        0xA7, 0xCB, 0x7C, 0x5F, 0x74, 0x91, 0x1D, 0x89,
        0x12, 0xC2, 0x9D, 0x62, 0x33, 0xF4, 0xD5, 0x3B,
        0xC6, 0x41, 0x39, 0xE2, 0xF5, 0x5B, 0xE7, 0x55,
        0x07, 0xDD, 0x77, 0x86, 0x8E, 0x38, 0x4A, 0xEC,
        0x58, 0x1F, 0x3F, 0x41, 0x1D, 0xB1, 0xA7, 0x42,
        0x97, 0x2D, 0x3E, 0xBF, 0xD3, 0x31, 0x5C, 0x84,
        0xA5, 0xAD, 0x63, 0xA0, 0xE7, 0x5C, 0x8B, 0xCA,
        0x3E, 0x30, 0x41, 0xE0, 0x5D, 0x90, 0x67, 0xAF,
        0xF3, 0xB1, 0x24, 0x4F, 0x76, 0x3E, 0x79, 0x83
    };
    static const byte seed_1024[WC_ML_KEM_ENC_RAND_SZ] = {
        0x59, 0xC5, 0x15, 0x4C, 0x04, 0xAE, 0x43, 0xAA,
        0xFF, 0x32, 0x70, 0x0F, 0x08, 0x17, 0x00, 0x38,
        0x9D, 0x54, 0xBE, 0xC4, 0xC3, 0x7C, 0x08, 0x8B,
        0x1C, 0x53, 0xF6, 0x62, 0x12, 0xB1, 0x2C, 0x72
    };
    static const byte c_1024[WC_ML_KEM_1024_CIPHER_TEXT_SIZE] = {
        0xE2, 0xD5, 0xFD, 0x4C, 0x13, 0xCE, 0xA0, 0xB5,
        0x2D, 0x87, 0x4F, 0xEA, 0x90, 0x12, 0xF3, 0xA5,
        0x17, 0x43, 0xA1, 0x09, 0x37, 0x10, 0xBB, 0xF2,
        0x39, 0x50, 0xF9, 0x14, 0x7A, 0x47, 0x2E, 0xE5,
        0x53, 0x39, 0x28, 0xA2, 0xF4, 0x6D, 0x59, 0x2F,
        0x35, 0xDA, 0x8B, 0x4F, 0x75, 0x8C, 0x89, 0x3B,
        0x0D, 0x7B, 0x98, 0x94, 0x8B, 0xE4, 0x47, 0xB1,
        0x7C, 0xB2, 0xAE, 0x58, 0xAF, 0x8A, 0x48, 0x9D,
        0xDD, 0x92, 0x32, 0xB9, 0x9B, 0x1C, 0x0D, 0x2D,
        0xE7, 0x7C, 0xAA, 0x47, 0x2B, 0xC3, 0xBB, 0xD4,
        0xA7, 0xC6, 0x0D, 0xBF, 0xDC, 0xA9, 0x2E, 0xBF,
        0x3A, 0x1C, 0xE1, 0xC2, 0x2D, 0xAD, 0x13, 0xE8,
        0x87, 0x00, 0x4E, 0x29, 0x24, 0xFD, 0x22, 0x65,
        0x6F, 0x5E, 0x50, 0x87, 0x91, 0xDE, 0x06, 0xD8,
        0x5E, 0x1A, 0x14, 0x26, 0x80, 0x8E, 0xD9, 0xA8,
        0x9F, 0x6E, 0x2F, 0xD3, 0xC2, 0x45, 0xD4, 0x75,
        0x8B, 0x22, 0xB0, 0x2C, 0xAD, 0xE3, 0x3B, 0x60,
        0xFC, 0x88, 0x9A, 0x33, 0xFC, 0x44, 0x47, 0xED,
        0xEB, 0xBF, 0xD4, 0x53, 0x0D, 0xE8, 0x65, 0x96,
        0xA3, 0x37, 0x89, 0xD5, 0xDB, 0xA6, 0xE6, 0xEC,
        0x9F, 0x89, 0x87, 0x9A, 0xF4, 0xBE, 0x49, 0x09,
        0xA6, 0x90, 0x17, 0xC9, 0xBB, 0x7A, 0x5E, 0x31,
        0x81, 0x5E, 0xA5, 0xF1, 0x32, 0xEE, 0xC4, 0x98,
        0x4F, 0xAA, 0x7C, 0xCF, 0x59, 0x4D, 0xD0, 0x0D,
        0x4D, 0x84, 0x87, 0xE4, 0x56, 0x21, 0xAF, 0x8F,
        0x6E, 0x33, 0x05, 0x51, 0x43, 0x9C, 0x93, 0xEC,
        0x07, 0x8A, 0x7A, 0x3C, 0xC1, 0x59, 0x4A, 0xF9,
        0x1F, 0x84, 0x17, 0x37, 0x5F, 0xD6, 0x08, 0x8C,
        0xEB, 0x5E, 0x85, 0xC6, 0x70, 0x99, 0x09, 0x1B,
        0xAC, 0x11, 0x49, 0x8A, 0x0D, 0x71, 0x14, 0x55,
        0xF5, 0xE0, 0xD9, 0x5C, 0xD7, 0xBB, 0xE5, 0xCD,
        0xD8, 0xFE, 0xCB, 0x31, 0x9E, 0x68, 0x53, 0xC2,
        0x3C, 0x9B, 0xE2, 0xC7, 0x63, 0xDF, 0x57, 0x86,
        0x66, 0xC4, 0x0A, 0x40, 0xA8, 0x74, 0x86, 0xE4,
        0x6B, 0xA8, 0x71, 0x61, 0x46, 0x19, 0x29, 0x04,
        0x51, 0x0A, 0x6D, 0xC5, 0x9D, 0xA8, 0x02, 0x58,
        0x25, 0x28, 0x3D, 0x68, 0x4D, 0xB9, 0x14, 0x10,
        0xB4, 0xF1, 0x2C, 0x6D, 0x8F, 0xBD, 0x0A, 0xDD,
        0x75, 0xD3, 0x09, 0x89, 0x18, 0xCB, 0x04, 0xAC,
        0x7B, 0xC4, 0xDB, 0x0D, 0x6B, 0xCD, 0xF1, 0x19,
        0x4D, 0xD8, 0x62, 0x92, 0xE0, 0x5B, 0x7B, 0x86,
        0x30, 0x62, 0x5B, 0x58, 0x9C, 0xC5, 0x09, 0xD2,
        0x15, 0xBB, 0xD0, 0x6A, 0x2E, 0x7C, 0x66, 0xF4,
        0x24, 0xCD, 0xF8, 0xC4, 0x0A, 0xC6, 0xC1, 0xE5,
        0xAE, 0x6C, 0x96, 0x4B, 0x7D, 0x9E, 0x92, 0xF9,
        0x5F, 0xC5, 0xC8, 0x85, 0x22, 0x81, 0x62, 0x8B,
        0x81, 0xB9, 0xAF, 0xAB, 0xC7, 0xF0, 0x3B, 0xE3,
        0xF6, 0x2E, 0x80, 0x47, 0xBB, 0x88, 0xD0, 0x1C,
        0x68, 0x68, 0x7B, 0x8D, 0xD4, 0xFE, 0x63, 0x82,
        0x00, 0x62, 0xB6, 0x78, 0x8A, 0x53, 0x72, 0x90,
        0x53, 0x82, 0x6E, 0xD3, 0xB7, 0xC7, 0xEF, 0x82,
        0x41, 0xE1, 0x9C, 0x85, 0x11, 0x7B, 0x3C, 0x53,
        0x41, 0x88, 0x1D, 0x4F, 0x29, 0x9E, 0x50, 0x37,
        0x4C, 0x8E, 0xEF, 0xD5, 0x56, 0x0B, 0xD1, 0x83,
        0x19, 0xA7, 0x96, 0x3A, 0x3D, 0x02, 0xF0, 0xFB,
        0xE8, 0x4B, 0xC4, 0x84, 0xB5, 0xA4, 0x01, 0x8B,
        0x97, 0xD2, 0x74, 0x19, 0x1C, 0x95, 0xF7, 0x02,
        0xBA, 0xB9, 0xB0, 0xD1, 0x05, 0xFA, 0xF9, 0xFD,
        0xCF, 0xF9, 0x7E, 0x43, 0x72, 0x36, 0x56, 0x75,
        0x99, 0xFA, 0xF7, 0x3B, 0x07, 0x5D, 0x40, 0x61,
        0x04, 0xD4, 0x03, 0xCD, 0xF8, 0x12, 0x24, 0xDA,
        0x59, 0x0B, 0xEC, 0x28, 0x97, 0xE3, 0x01, 0x09,
        0xE1, 0xF2, 0xE5, 0xAE, 0x46, 0x10, 0xC8, 0x09,
        0xA7, 0x3F, 0x63, 0x8C, 0x84, 0x21, 0x0B, 0x34,
        0x47, 0xA7, 0xC8, 0xB6, 0xDD, 0xDB, 0x5A, 0xE2,
        0x00, 0xBF, 0x20, 0xE2, 0xFE, 0x4D, 0x4B, 0xA6,
        0xC6, 0xB1, 0x27, 0x67, 0xFB, 0x87, 0x60, 0xF6,
        0x6C, 0x51, 0x18, 0xE7, 0xA9, 0x93, 0x5B, 0x41,
        0xC9, 0xA4, 0x71, 0xA1, 0xD3, 0x23, 0x76, 0x88,
        0xC1, 0xE6, 0x18, 0xCC, 0x3B, 0xE9, 0x36, 0xAA,
        0x3F, 0x5E, 0x44, 0xE0, 0x86, 0x82, 0x0B, 0x81,
        0x0E, 0x06, 0x32, 0x11, 0xFC, 0x21, 0xC4, 0x04,
        0x4B, 0x3A, 0xC4, 0xD0, 0x0D, 0xF1, 0xBC, 0xC7,
        0xB2, 0x4D, 0xC0, 0x7B, 0xA4, 0x8B, 0x23, 0xB0,
        0xFC, 0x12, 0xA3, 0xED, 0x3D, 0x0A, 0x5C, 0xF7,
        0x67, 0x14, 0x15, 0xAB, 0x9C, 0xF2, 0x12, 0x86,
        0xFE, 0x63, 0xFB, 0x41, 0x41, 0x85, 0x70, 0x55,
        0x5D, 0x47, 0x39, 0xB8, 0x81, 0x04, 0xA8, 0x59,
        0x3F, 0x29, 0x30, 0x25, 0xA4, 0xE3, 0xEE, 0x7C,
        0x67, 0xE4, 0xB4, 0x8E, 0x40, 0xF6, 0xBA, 0x8C,
        0x09, 0x86, 0x0C, 0x3F, 0xBB, 0xE5, 0x5D, 0x45,
        0xB4, 0x5F, 0xC9, 0xAB, 0x62, 0x9B, 0x17, 0xC2,
        0x76, 0xC9, 0xC9, 0xE2, 0xAF, 0x3A, 0x04, 0x3B,
        0xEA, 0xFC, 0x18, 0xFD, 0x4F, 0x25, 0xEE, 0x7F,
        0x83, 0xBD, 0xDC, 0xD2, 0xD9, 0x39, 0x14, 0xB7,
        0xED, 0x4F, 0x7C, 0x9A, 0xF1, 0x27, 0xF3, 0xF1,
        0x5C, 0x27, 0x7B, 0xE1, 0x65, 0x51, 0xFE, 0xF3,
        0xAE, 0x03, 0xD7, 0xB9, 0x14, 0x3F, 0x0C, 0x9C,
        0x01, 0x9A, 0xB9, 0x7E, 0xEA, 0x07, 0x63, 0x66,
        0x13, 0x1F, 0x51, 0x83, 0x63, 0x71, 0x1B, 0x34,
        0xE9, 0x6D, 0x3F, 0x8A, 0x51, 0x3F, 0x3E, 0x20,
        0xB1, 0xD4, 0x52, 0xC4, 0xB7, 0xAE, 0x3B, 0x97,
        0x5E, 0xA9, 0x4D, 0x88, 0x0D, 0xAC, 0x66, 0x93,
        0x39, 0x97, 0x50, 0xD0, 0x22, 0x20, 0x40, 0x3F,
        0x0D, 0x3E, 0x3F, 0xC1, 0x17, 0x2A, 0x4D, 0xE9,
        0xDC, 0x28, 0x0E, 0xAF, 0x0F, 0xEE, 0x28, 0x83,
        0xA6, 0x66, 0x0B, 0xF5, 0xA3, 0xD2, 0x46, 0xFF,
        0x41, 0xD2, 0x1B, 0x36, 0xEA, 0x52, 0x1C, 0xF7,
        0xAA, 0x68, 0x9F, 0x80, 0x0D, 0x0F, 0x86, 0xF4,
        0xFA, 0x10, 0x57, 0xD8, 0xA1, 0x3F, 0x9D, 0xA8,
        0xFF, 0xFD, 0x0D, 0xC1, 0xFA, 0xD3, 0xC0, 0x4B,
        0xB1, 0xCC, 0xCB, 0x7C, 0x83, 0x4D, 0xB0, 0x51,
        0xA7, 0xAC, 0x2E, 0x4C, 0x60, 0x30, 0x19, 0x96,
        0xC9, 0x30, 0x71, 0xEA, 0x41, 0x6B, 0x42, 0x17,
        0x59, 0x93, 0x56, 0x59, 0xCF, 0x62, 0xCA, 0x5F,
        0x13, 0xAE, 0x07, 0xC3, 0xB1, 0x95, 0xC1, 0x48,
        0x15, 0x9D, 0x8B, 0xEB, 0x03, 0xD4, 0x40, 0xB0,
        0x0F, 0x53, 0x05, 0x76, 0x5F, 0x20, 0xC0, 0xC4,
        0x6E, 0xEE, 0x59, 0xC6, 0xD1, 0x62, 0x06, 0x40,
        0x2D, 0xB1, 0xC7, 0x15, 0xE8, 0x88, 0xBD, 0xE5,
        0x9C, 0x78, 0x1F, 0x35, 0xA7, 0xCC, 0x7C, 0x1C,
        0x5E, 0xCB, 0x21, 0x55, 0xAE, 0x3E, 0x95, 0x9C,
        0x09, 0x64, 0xCC, 0x1E, 0xF8, 0xD7, 0xC6, 0x9D,
        0x14, 0x58, 0xA9, 0xA4, 0x2F, 0x95, 0xF4, 0xC6,
        0xB5, 0xB9, 0x96, 0x34, 0x57, 0x12, 0xAA, 0x29,
        0x0F, 0xBB, 0xF7, 0xDF, 0xD4, 0xA6, 0xE8, 0x64,
        0x63, 0x02, 0x2A, 0x3F, 0x47, 0x25, 0xF6, 0x51,
        0x1B, 0xF7, 0xEA, 0x5E, 0x95, 0xC7, 0x07, 0xCD,
        0x35, 0x73, 0x60, 0x9A, 0xAD, 0xEA, 0xF5, 0x40,
        0x15, 0x2C, 0x49, 0x5F, 0x37, 0xFE, 0x6E, 0xC8,
        0xBB, 0x9F, 0xA2, 0xAA, 0x61, 0xD1, 0x57, 0x35,
        0x93, 0x4F, 0x47, 0x37, 0x92, 0x8F, 0xDE, 0x90,
        0xBA, 0x99, 0x57, 0x22, 0x46, 0x5D, 0x4A, 0x64,
        0x50, 0x5A, 0x52, 0x01, 0xF0, 0x7A, 0xA5, 0x8C,
        0xFD, 0x8A, 0xE2, 0x26, 0xE0, 0x20, 0x70, 0xB2,
        0xDB, 0xF5, 0x12, 0xB9, 0x75, 0x31, 0x9A, 0x7E,
        0x87, 0x53, 0xB4, 0xFD, 0xAE, 0x0E, 0xB4, 0x92,
        0x28, 0x69, 0xCC, 0x8E, 0x25, 0xC4, 0xA5, 0x56,
        0x0C, 0x2A, 0x06, 0x85, 0xDE, 0x3A, 0xC3, 0x92,
        0xA8, 0x92, 0x5B, 0xA8, 0x82, 0x00, 0x48, 0x94,
        0x74, 0x2E, 0x43, 0xCC, 0xFC, 0x27, 0x74, 0x39,
        0xEC, 0x80, 0x50, 0xA9, 0xAE, 0xB4, 0x29, 0x32,
        0xE0, 0x1C, 0x84, 0x0D, 0xFC, 0xED, 0xCC, 0x34,
        0xD3, 0x99, 0x12, 0x89, 0xA6, 0x2C, 0x17, 0xD1,
        0x28, 0x4C, 0x83, 0x95, 0x14, 0xB9, 0x33, 0x51,
        0xDB, 0xB2, 0xDD, 0xA8, 0x1F, 0x92, 0x45, 0x65,
        0xD7, 0x0E, 0x70, 0x79, 0xD5, 0xB8, 0x12, 0x6C,
        0xAA, 0xB7, 0xA4, 0xA1, 0xC7, 0x31, 0x65, 0x5A,
        0x53, 0xBC, 0xC0, 0x9F, 0x5D, 0x63, 0xEC, 0x90,
        0x86, 0xDE, 0xA6, 0x50, 0x05, 0x59, 0x85, 0xED,
        0xFA, 0x82, 0x97, 0xD9, 0xC9, 0x54, 0x10, 0xC5,
        0xD1, 0x89, 0x4D, 0x17, 0xD5, 0x93, 0x05, 0x49,
        0xAD, 0xBC, 0x2B, 0x87, 0x33, 0xC9, 0x9F, 0xE6,
        0x2E, 0x17, 0xC4, 0xDE, 0x34, 0xA5, 0xD8, 0x9B,
        0x12, 0xD1, 0x8E, 0x42, 0xA4, 0x22, 0xD2, 0xCE,
        0x77, 0x9C, 0x2C, 0x28, 0xEB, 0x2D, 0x98, 0x00,
        0x3D, 0x5C, 0xD3, 0x23, 0xFC, 0xBE, 0xCF, 0x02,
        0xB5, 0x06, 0x6E, 0x0E, 0x73, 0x48, 0x10, 0xF0,
        0x9E, 0xD8, 0x90, 0x13, 0xC0, 0x0F, 0x01, 0x1B,
        0xD2, 0x20, 0xF2, 0xE5, 0xD6, 0xA3, 0x62, 0xDF,
        0x90, 0x59, 0x91, 0x98, 0xA0, 0x93, 0xB0, 0x3C,
        0x8D, 0x8E, 0xFB, 0xFE, 0x0B, 0x61, 0x75, 0x92,
        0xFA, 0xF1, 0xE6, 0x42, 0x20, 0xC4, 0x44, 0x0B,
        0x53, 0xFF, 0xB4, 0x71, 0x64, 0xF3, 0x69, 0xC9,
        0x52, 0x90, 0xBA, 0x9F, 0x31, 0x08, 0xD6, 0x86,
        0xC5, 0x7D, 0xB6, 0x45, 0xC5, 0x3C, 0x01, 0x2E,
        0x57, 0xAF, 0x25, 0xBD, 0x66, 0x93, 0xE2, 0xCC,
        0x6B, 0x57, 0x65, 0x1A, 0xF1, 0x59, 0x1F, 0xE5,
        0xD8, 0x91, 0x66, 0x40, 0xEC, 0x01, 0x7C, 0x25,
        0x3D, 0xF0, 0x60, 0x6B, 0xB6, 0xB3, 0x03, 0x5F,
        0xAE, 0x74, 0x8F, 0x3D, 0x40, 0x34, 0x22, 0x3B,
        0x1B, 0x5E, 0xFB, 0xF5, 0x28, 0x3E, 0x77, 0x8C,
        0x10, 0x94, 0x29, 0x1C, 0xF7, 0xB1, 0x9B, 0xE0,
        0xF3, 0x17, 0x35, 0x0E, 0x6F, 0x85, 0x18, 0xFD,
        0xE0, 0xEF, 0xB1, 0x38, 0x1F, 0xB6, 0xE1, 0x6C,
        0x24, 0x1F, 0x7F, 0x17, 0xA5, 0x21, 0x06, 0x93,
        0xA2, 0x74, 0x15, 0x9E, 0x7F, 0xAC, 0x86, 0x8C,
        0xD0, 0xDC, 0x43, 0x59, 0xC3, 0xD9, 0xEE, 0xFE,
        0xA0, 0xD9, 0xE3, 0x1E, 0x43, 0xFA, 0x65, 0x13,
        0x92, 0xC6, 0x5A, 0x54, 0x3A, 0x59, 0xB3, 0xEE,
        0xE3, 0xA6, 0x39, 0xDC, 0x94, 0x17, 0xD0, 0x56,
        0xA5, 0xFF, 0x0F, 0x16, 0x0B, 0xEE, 0xE2, 0xEA,
        0xC2, 0x9A, 0x7D, 0x88, 0xC0, 0x98, 0x2C, 0xF7,
        0x0B, 0x5A, 0x46, 0x37, 0x9F, 0x21, 0xE5, 0x06,
        0xAA, 0xC6, 0x1A, 0x9B, 0xB1, 0xB8, 0xC2, 0xB9,
        0xDA, 0xB0, 0xE4, 0x4A, 0x82, 0x3B, 0x61, 0xD0,
        0xAA, 0x11, 0xD9, 0x4F, 0x76, 0xA4, 0xA8, 0xE2,
        0x1F, 0x9D, 0x42, 0x80, 0x68, 0x32, 0x08, 0xF4,
        0xEA, 0x91, 0x11, 0x16, 0xF6, 0xFD, 0x6A, 0x97,
        0x42, 0x69, 0x34, 0xEC, 0x34, 0x26, 0xB8, 0xC8,
        0xF7, 0x03, 0xDA, 0x85, 0xE9, 0xDC, 0xF9, 0x93,
        0x36, 0x13, 0x60, 0x03, 0x72, 0x8B, 0x8E, 0xCD,
        0xD0, 0x4A, 0x38, 0x9F, 0x6A, 0x81, 0x7A, 0x78,
        0xBF, 0xA6, 0x1B, 0xA4, 0x60, 0x20, 0xBF, 0x3C,
        0x34, 0x82, 0x95, 0x08, 0xF9, 0xD0, 0x6D, 0x15,
        0x53, 0xCD, 0x98, 0x7A, 0xAC, 0x38, 0x0D, 0x86,
        0xF1, 0x68, 0x84, 0x3B, 0xA3, 0x90, 0x4D, 0xE5,
        0xF7, 0x05, 0x8A, 0x41, 0xB4, 0xCD, 0x38, 0x8B,
        0xC9, 0xCE, 0x3A, 0xBA, 0x7E, 0xE7, 0x13, 0x9B,
        0x7F, 0xC9, 0xE5, 0xB8, 0xCF, 0xAA, 0xA3, 0x89,
        0x90, 0xBD, 0x4A, 0x5D, 0xB3, 0x2E, 0x26, 0x13,
        0xE7, 0xEC, 0x4F, 0x5F, 0x8B, 0x12, 0x92, 0xA3,
        0x8C, 0x6F, 0x4F, 0xF5, 0xA4, 0x04, 0x90, 0xD7,
        0x6B, 0x12, 0x66, 0x52, 0xFC, 0xF8, 0x6E, 0x24,
        0x52, 0x35, 0xD6, 0x36, 0xC6, 0x5C, 0xD1, 0x02,
        0xB0, 0x1E, 0x22, 0x78, 0x1A, 0x72, 0x91, 0x8C
    };
    static const byte k_1024[WC_ML_KEM_SS_SZ] = {
        0x72, 0x64, 0xBD, 0xE5, 0xC6, 0xCE, 0xC1, 0x48,
        0x49, 0x69, 0x3E, 0x2C, 0x3C, 0x86, 0xE4, 0x8F,
        0x80, 0x95, 0x8A, 0x4F, 0x61, 0x86, 0xFC, 0x69,
        0x33, 0x3A, 0x41, 0x48, 0xE6, 0xE4, 0x97, 0xF3
    };
#endif
    static byte ct[WC_ML_KEM_MAX_CIPHER_TEXT_SIZE];
    static byte ss[WC_ML_KEM_SS_SZ];

    key = (MlKemKey*)XMALLOC(sizeof(MlKemKey), NULL, DYNAMIC_TYPE_TMP_BUFFER);
    ExpectNotNull(key);
    if (key != NULL) {
        XMEMSET(key, 0, sizeof(MlKemKey));
    }

#ifndef WOLFSSL_NO_ML_KEM_512
    ExpectIntEQ(wc_MlKemKey_Init(key, WC_ML_KEM_512, NULL, INVALID_DEVID), 0);
    ExpectIntEQ(wc_MlKemKey_DecodePublicKey(key, ek_512, sizeof(ek_512)), 0);
    ExpectIntEQ(wc_MlKemKey_EncapsulateWithRandom(key, ct, ss, seed_512,
        sizeof(seed_512)), 0);
    ExpectIntEQ(XMEMCMP(ct, c_512, WC_ML_KEM_512_CIPHER_TEXT_SIZE), 0);
    ExpectIntEQ(XMEMCMP(ss, k_512, WC_ML_KEM_SS_SZ), 0);
    wc_MlKemKey_Free(key);
#endif
#ifndef WOLFSSL_NO_ML_KEM_768
    ExpectIntEQ(wc_MlKemKey_Init(key, WC_ML_KEM_768, NULL, INVALID_DEVID), 0);
    ExpectIntEQ(wc_MlKemKey_DecodePublicKey(key, ek_768, sizeof(ek_768)), 0);
    ExpectIntEQ(wc_MlKemKey_EncapsulateWithRandom(key, ct, ss, seed_768,
        sizeof(seed_768)), 0);
    ExpectIntEQ(XMEMCMP(ct, c_768, WC_ML_KEM_768_CIPHER_TEXT_SIZE), 0);
    ExpectIntEQ(XMEMCMP(ss, k_768, WC_ML_KEM_SS_SZ), 0);
    wc_MlKemKey_Free(key);
#endif
#ifndef WOLFSSL_NO_ML_KEM_1024
    ExpectIntEQ(wc_MlKemKey_Init(key, WC_ML_KEM_1024, NULL, INVALID_DEVID), 0);
    ExpectIntEQ(wc_MlKemKey_DecodePublicKey(key, ek_1024, sizeof(ek_1024)), 0);
    ExpectIntEQ(wc_MlKemKey_EncapsulateWithRandom(key, ct, ss, seed_1024,
        sizeof(seed_1024)), 0);
    ExpectIntEQ(XMEMCMP(ct, c_1024, WC_ML_KEM_1024_CIPHER_TEXT_SIZE), 0);
    ExpectIntEQ(XMEMCMP(ss, k_1024, WC_ML_KEM_SS_SZ), 0);
    wc_MlKemKey_Free(key);
#endif

    XFREE(key, NULL, DYNAMIC_TYPE_TMP_BUFFER);
#endif
    return EXPECT_RESULT();
}

int test_wc_mlkem_decapsulate_kats(void)
{
    EXPECT_DECLS;
#if defined(WOLFSSL_HAVE_MLKEM) && defined(WOLFSSL_WC_MLKEM) && \
    !defined(WOLFSSL_NO_ML_KEM) && !defined(WOLFSSL_MLKEM_NO_DECAPSULATE)
    MlKemKey* key;
#ifndef WOLFSSL_NO_ML_KEM_512
    static const byte dk_512[WC_ML_KEM_512_PRIVATE_KEY_SIZE] = {
        0x69, 0xF9, 0xCB, 0xFD, 0x12, 0x37, 0xBA, 0x16,
        0x1C, 0xF6, 0xE6, 0xC1, 0x8F, 0x48, 0x8F, 0xC6,
        0xE3, 0x9A, 0xB4, 0xA5, 0xC9, 0xE6, 0xC2, 0x2E,
        0xA4, 0xE3, 0xAD, 0x8F, 0x26, 0x7A, 0x9C, 0x44,
        0x20, 0x10, 0xD3, 0x2E, 0x61, 0xF8, 0x3E, 0x6B,
        0xFA, 0x5C, 0x58, 0x70, 0x61, 0x45, 0x37, 0x6D,
        0xBB, 0x84, 0x95, 0x28, 0xF6, 0x80, 0x07, 0xC8,
        0x22, 0xB3, 0x3A, 0x95, 0xB8, 0x49, 0x04, 0xDC,
        0xD2, 0x70, 0x8D, 0x03, 0x40, 0xC8, 0xB8, 0x08,
        0xBC, 0xD3, 0xAA, 0xD0, 0xE4, 0x8B, 0x85, 0x84,
        0x95, 0x83, 0xA1, 0xB4, 0xE5, 0x94, 0x5D, 0xD9,
        0x51, 0x4A, 0x7F, 0x64, 0x61, 0xE0, 0x57, 0xB7,
        0xEC, 0xF6, 0x19, 0x57, 0xE9, 0x7C, 0xF6, 0x28,
        0x15, 0xF9, 0xC3, 0x22, 0x94, 0xB3, 0x26, 0xE1,
        0xA1, 0xC4, 0xE3, 0x60, 0xB9, 0x49, 0x8B, 0xA8,
        0x0F, 0x8C, 0xA9, 0x15, 0x32, 0xB1, 0x71, 0xD0,
        0xAE, 0xFC, 0x48, 0x49, 0xFA, 0x53, 0xBC, 0x61,
        0x79, 0x32, 0xE2, 0x08, 0xA6, 0x77, 0xC6, 0x04,
        0x4A, 0x66, 0x00, 0xB8, 0xD8, 0xB8, 0x3F, 0x26,
        0xA7, 0x47, 0xB1, 0x8C, 0xFB, 0x78, 0xBE, 0xAF,
        0xC5, 0x51, 0xAD, 0x52, 0xB7, 0xCA, 0x6C, 0xB8,
        0x8F, 0x3B, 0x5D, 0x9C, 0xE2, 0xAF, 0x6C, 0x67,
        0x95, 0x6C, 0x47, 0x8C, 0xEF, 0x49, 0x1F, 0x59,
        0xE0, 0x19, 0x1B, 0x3B, 0xBE, 0x92, 0x9B, 0x94,
        0xB6, 0x66, 0xC1, 0x76, 0x13, 0x8B, 0x00, 0xF4,
        0x97, 0x24, 0x34, 0x1E, 0xE2, 0xE1, 0x64, 0xB9,
        0x4C, 0x05, 0x3C, 0x18, 0x5A, 0x51, 0xF9, 0x3E,
        0x00, 0xF3, 0x68, 0x61, 0x61, 0x3A, 0x7F, 0xD7,
        0x2F, 0xEB, 0xD2, 0x3A, 0x8B, 0x96, 0xA2, 0x60,
        0x23, 0x42, 0x39, 0xC9, 0x62, 0x8F, 0x99, 0x5D,
        0xC1, 0x38, 0x07, 0xB4, 0x3A, 0x69, 0x46, 0x81,
        0x67, 0xCB, 0x1A, 0x8F, 0x9D, 0xD0, 0x7E, 0xE3,
        0xB3, 0x32, 0x38, 0xF6, 0x30, 0x96, 0xEB, 0xC4,
        0x9D, 0x50, 0x51, 0xC4, 0xB6, 0x59, 0x63, 0xD7,
        0x4A, 0x47, 0x66, 0xC2, 0x26, 0xF0, 0xB9, 0x4F,
        0x18, 0x62, 0xC2, 0x12, 0x4C, 0x8C, 0x74, 0x97,
        0x48, 0xC0, 0xBC, 0x4D, 0xC1, 0x4C, 0xB3, 0x49,
        0x06, 0xB8, 0x1C, 0x55, 0x24, 0xFB, 0x81, 0x00,
        0x79, 0x85, 0x42, 0xDC, 0x6C, 0xC2, 0xAA, 0x0A,
        0x70, 0x85, 0x75, 0xEA, 0xBC, 0xC1, 0x1F, 0x96,
        0xA9, 0xE6, 0x1C, 0x01, 0x7A, 0x96, 0xA7, 0xCE,
        0x93, 0xC4, 0x20, 0x91, 0x73, 0x71, 0x13, 0xAE,
        0x78, 0x3C, 0x0A, 0xE8, 0x75, 0x5E, 0x59, 0x41,
        0x11, 0xED, 0xFA, 0xBF, 0xD8, 0x6C, 0x32, 0x12,
        0xC6, 0x12, 0xA7, 0xB6, 0x2A, 0xFD, 0x3C, 0x7A,
        0x5C, 0x78, 0xB2, 0xF0, 0x73, 0x44, 0xB7, 0x89,
        0xC2, 0xB2, 0xDB, 0xB5, 0xF4, 0x44, 0x8B, 0xE9,
        0x7B, 0xBA, 0x42, 0x33, 0xC0, 0x03, 0x9C, 0x0F,
        0xE8, 0x43, 0x00, 0xF9, 0xB0, 0x3A, 0xC9, 0x94,
        0x97, 0xE6, 0xD4, 0x6B, 0x6E, 0x95, 0x30, 0x8F,
        0xF8, 0x47, 0x90, 0xF6, 0x12, 0xCF, 0x18, 0x6E,
        0xC1, 0x68, 0x11, 0xE8, 0x0C, 0x17, 0x93, 0x16,
        0xA6, 0x3B, 0x25, 0x70, 0x3F, 0x60, 0xB8, 0x42,
        0xB6, 0x19, 0x07, 0xE6, 0x28, 0x94, 0xE7, 0x36,
        0x64, 0x7B, 0x3C, 0x09, 0xDA, 0x6F, 0xEC, 0x59,
        0x32, 0x78, 0x2B, 0x36, 0xE0, 0x63, 0x50, 0x85,
        0xA3, 0x94, 0x9E, 0x69, 0x4D, 0x7E, 0x17, 0xCB,
        0xA3, 0xD9, 0x06, 0x43, 0x30, 0x43, 0x8C, 0x07,
        0x1B, 0x58, 0x36, 0xA7, 0x70, 0xC5, 0x5F, 0x62,
        0x13, 0xCC, 0x14, 0x25, 0x84, 0x5D, 0xE5, 0xA3,
        0x34, 0xD7, 0x5D, 0x3E, 0x50, 0x58, 0xC7, 0x80,
        0x9F, 0xDA, 0x4B, 0xCD, 0x78, 0x19, 0x1D, 0xA9,
        0x79, 0x73, 0x25, 0xE6, 0x23, 0x6C, 0x26, 0x50,
        0xFC, 0x60, 0x4E, 0xE4, 0x3A, 0x83, 0xCE, 0xB3,
        0x49, 0x80, 0x08, 0x44, 0x03, 0xA3, 0x32, 0x59,
        0x85, 0x79, 0x07, 0x79, 0x9A, 0x9D, 0x2A, 0x71,
        0x3A, 0x63, 0x3B, 0x5C, 0x90, 0x47, 0x27, 0xF6,
        0x1E, 0x42, 0x52, 0x09, 0x91, 0xD6, 0x55, 0x70,
        0x5C, 0xB6, 0xBC, 0x1B, 0x74, 0xAF, 0x60, 0x71,
        0x3E, 0xF8, 0x71, 0x2F, 0x14, 0x08, 0x68, 0x69,
        0xBE, 0x8E, 0xB2, 0x97, 0xD2, 0x28, 0xB3, 0x25,
        0xA0, 0x60, 0x9F, 0xD6, 0x15, 0xEA, 0xB7, 0x08,
        0x15, 0x40, 0xA6, 0x1A, 0x82, 0xAB, 0xF4, 0x3B,
        0x7D, 0xF9, 0x8A, 0x59, 0x5B, 0xE1, 0x1F, 0x41,
        0x6B, 0x41, 0xE1, 0xEB, 0x75, 0xBB, 0x57, 0x97,
        0x7C, 0x25, 0xC6, 0x4E, 0x97, 0x43, 0x7D, 0x88,
        0xCA, 0x5F, 0xDA, 0x61, 0x59, 0xD6, 0x68, 0xF6,
        0xBA, 0xB8, 0x15, 0x75, 0x55, 0xB5, 0xD5, 0x4C,
        0x0F, 0x47, 0xCB, 0xCD, 0x16, 0x84, 0x3B, 0x1A,
        0x0A, 0x0F, 0x02, 0x10, 0xEE, 0x31, 0x03, 0x13,
        0x96, 0x7F, 0x3D, 0x51, 0x64, 0x99, 0x01, 0x8F,
        0xDF, 0x31, 0x14, 0x77, 0x24, 0x70, 0xA1, 0x88,
        0x9C, 0xC0, 0x6C, 0xB6, 0xB6, 0x69, 0x0A, 0xC3,
        0x1A, 0xBC, 0xFA, 0xF4, 0xBC, 0x70, 0x76, 0x84,
        0x54, 0x5B, 0x00, 0x0B, 0x58, 0x0C, 0xCB, 0xFC,
        0xBC, 0xE9, 0xFA, 0x70, 0xAA, 0xEA, 0x0B, 0xBD,
        0x91, 0x10, 0x99, 0x2A, 0x7C, 0x6C, 0x06, 0xCB,
        0x36, 0x85, 0x27, 0xFD, 0x22, 0x90, 0x90, 0x75,
        0x7E, 0x6F, 0xE7, 0x57, 0x05, 0xFA, 0x59, 0x2A,
        0x76, 0x08, 0xF0, 0x50, 0xC6, 0xF8, 0x87, 0x03,
        0xCC, 0x28, 0xCB, 0x00, 0x0C, 0x1D, 0x7E, 0x77,
        0xB8, 0x97, 0xB7, 0x2C, 0x62, 0xBC, 0xC7, 0xAE,
        0xA2, 0x1A, 0x57, 0x72, 0x94, 0x83, 0xD2, 0x21,
        0x18, 0x32, 0xBE, 0xD6, 0x12, 0x43, 0x0C, 0x98,
        0x31, 0x03, 0xC6, 0x9E, 0x8C, 0x07, 0x2C, 0x0E,
        0xA7, 0x89, 0x8F, 0x22, 0x83, 0xBE, 0xC4, 0x8C,
        0x5A, 0xC8, 0x19, 0x84, 0xD4, 0xA5, 0xA8, 0x36,
        0x19, 0x73, 0x5A, 0x84, 0x2B, 0xD1, 0x72, 0xC0,
        0xD1, 0xB3, 0x9F, 0x43, 0x58, 0x8A, 0xF1, 0x70,
        0x45, 0x8B, 0xA9, 0xEE, 0x74, 0x92, 0xEA, 0xAA,
        0x94, 0xEA, 0x53, 0xA4, 0xD3, 0x84, 0x98, 0xEC,
        0xBB, 0x98, 0xA5, 0xF4, 0x07, 0xE7, 0xC9, 0x7B,
        0x4E, 0x16, 0x6E, 0x39, 0x71, 0x92, 0xC2, 0x16,
        0x03, 0x30, 0x14, 0xB8, 0x78, 0xE9, 0x38, 0x07,
        0x5C, 0x6C, 0x1F, 0x10, 0xA0, 0x06, 0x5A, 0xBC,
        0x31, 0x63, 0x72, 0x2F, 0x1A, 0x2E, 0xFF, 0xEC,
        0x8D, 0x6E, 0x3A, 0x0C, 0x4F, 0x71, 0x74, 0xFC,
        0x16, 0xB7, 0x9F, 0xB5, 0x18, 0x6A, 0x75, 0x16,
        0x8F, 0x81, 0xA5, 0x6A, 0xA4, 0x8A, 0x20, 0xA0,
        0x4B, 0xDD, 0xF1, 0x82, 0xC6, 0xE1, 0x79, 0xC3,
        0xF6, 0x90, 0x61, 0x55, 0x5E, 0xF7, 0x39, 0x6D,
        0xD0, 0xB7, 0x49, 0x96, 0x01, 0xA6, 0xEB, 0x3A,
        0x96, 0xA9, 0xA2, 0x2D, 0x04, 0xF1, 0x16, 0x8D,
        0xB5, 0x63, 0x55, 0xB0, 0x76, 0x00, 0xA2, 0x03,
        0x70, 0x63, 0x7B, 0x64, 0x59, 0x76, 0xBB, 0xD9,
        0x7B, 0x6D, 0x62, 0x88, 0xA0, 0xD3, 0x03, 0x63,
        0x60, 0x47, 0x2E, 0x3A, 0xC7, 0x1D, 0x56, 0x6D,
        0xB8, 0xFB, 0xB1, 0xB1, 0xD7, 0x6C, 0xB7, 0x55,
        0xCD, 0x0D, 0x68, 0xBD, 0xBF, 0xC0, 0x48, 0xEB,
        0xA2, 0x52, 0x5E, 0xEA, 0x9D, 0xD5, 0xB1, 0x44,
        0xFB, 0x3B, 0x60, 0xFB, 0xC3, 0x42, 0x39, 0x32,
        0x0C, 0xBC, 0x06, 0x9B, 0x35, 0xAB, 0x16, 0xB8,
        0x75, 0x65, 0x36, 0xFB, 0x33, 0xE8, 0xA6, 0xAF,
        0x1D, 0xD4, 0x2C, 0x79, 0xF4, 0x8A, 0xD1, 0x20,
        0xAE, 0x4B, 0x15, 0x9D, 0x3D, 0x8C, 0x31, 0x90,
        0x60, 0xCC, 0xE5, 0x69, 0xC3, 0xF6, 0x03, 0x53,
        0x65, 0x58, 0x5D, 0x34, 0x41, 0x37, 0x95, 0xA6,
        0xA1, 0x8E, 0xC5, 0x13, 0x6A, 0xB1, 0x3C, 0x90,
        0xE3, 0xAF, 0x14, 0xC0, 0xB8, 0xA4, 0x64, 0xC8,
        0x6B, 0x90, 0x73, 0x22, 0x2B, 0x56, 0xB3, 0xF7,
        0x32, 0x8A, 0xEA, 0x79, 0x81, 0x55, 0x32, 0x59,
        0x11, 0x25, 0x0E, 0xF0, 0x16, 0xD7, 0x28, 0x02,
        0xE3, 0x87, 0x8A, 0xA5, 0x05, 0x40, 0xCC, 0x98,
        0x39, 0x56, 0x97, 0x1D, 0x6E, 0xFA, 0x35, 0x2C,
        0x02, 0x55, 0x4D, 0xC7, 0x60, 0xA5, 0xA9, 0x13,
        0x58, 0xEA, 0x56, 0x37, 0x08, 0x84, 0xFD, 0x5B,
        0x3F, 0x85, 0xB7, 0x0E, 0x83, 0xE4, 0x69, 0x7D,
        0xEB, 0x17, 0x05, 0x16, 0x9E, 0x9C, 0x60, 0xA7,
        0x45, 0x28, 0xCF, 0x15, 0x28, 0x1C, 0xB1, 0xB1,
        0xC4, 0x57, 0xD4, 0x67, 0xB5, 0xF9, 0x3A, 0x60,
        0x37, 0x3D, 0x10, 0xE0, 0xCF, 0x6A, 0x83, 0x7A,
        0xA3, 0xC9, 0x59, 0x6A, 0x72, 0xBE, 0xC2, 0x9B,
        0x2D, 0x7E, 0x58, 0x65, 0x3D, 0x53, 0x30, 0x61,
        0xD3, 0x81, 0xD5, 0x17, 0x59, 0x75, 0x22, 0x17,
        0xEB, 0x46, 0xCA, 0xC7, 0x80, 0x7C, 0x4A, 0xD3,
        0x8B, 0x61, 0x16, 0x44, 0xAC, 0xF0, 0xA3, 0xF2,
        0x6B, 0x6B, 0x08, 0x4A, 0xB4, 0x7A, 0x83, 0xBF,
        0x0D, 0x69, 0x6F, 0x8A, 0x47, 0x68, 0xFC, 0x35,
        0xBC, 0xA6, 0xBC, 0x79, 0x03, 0xB2, 0xA2, 0x37,
        0xC2, 0x77, 0x49, 0xF5, 0x51, 0x0C, 0x86, 0x38,
        0x69, 0xE6, 0xAE, 0x56, 0xBB, 0x2A, 0xFE, 0x47,
        0x71, 0xC9, 0x22, 0x18, 0x74, 0xF5, 0x0F, 0x5B,
        0x14, 0xBA, 0xAD, 0x59, 0x93, 0xB4, 0x92, 0x38,
        0xFD, 0x0A, 0x0C, 0x9F, 0x79, 0xB7, 0xB4, 0x58,
        0x4E, 0x41, 0x30, 0x1F, 0x7A, 0x88, 0x5C, 0x9F,
        0x91, 0x81, 0x9B, 0xEA, 0x00, 0xD5, 0x12, 0x58,
        0x17, 0x30, 0x53, 0x9F, 0xB3, 0x7E, 0x59, 0xE8,
        0x6A, 0x6D, 0x19, 0xCA, 0x25, 0xF0, 0xA8, 0x11,
        0xC9, 0xB4, 0x28, 0xBA, 0x86, 0x14, 0xAA, 0x4F,
        0x94, 0x80, 0x7B, 0xC0, 0x31, 0xCB, 0xCC, 0x18,
        0x3F, 0x3B, 0xF0, 0x7F, 0xE2, 0xC1, 0xA6, 0xEB,
        0xA8, 0x0D, 0x5A, 0x70, 0x6E, 0xE0, 0xDA, 0xB2,
        0x7E, 0x23, 0x14, 0x58, 0x02, 0x5D, 0x84, 0xA7,
        0xA9, 0xB0, 0x23, 0x05, 0x01, 0x11, 0x6C, 0x29,
        0x0A, 0x6B, 0xB5, 0x06, 0x26, 0xD9, 0x7B, 0x93,
        0x98, 0x50, 0x94, 0x28, 0x28, 0x39, 0x0B, 0x0A,
        0x20, 0x01, 0xB7, 0x85, 0x3A, 0xD1, 0xAE, 0x9B,
        0x01, 0x1B, 0x2D, 0xB3, 0x6C, 0xAE, 0xEA, 0x73,
        0xA2, 0x32, 0x8E, 0x3C, 0x56, 0x48, 0x5B, 0x49,
        0x1C, 0x29, 0x91, 0x15, 0xA0, 0x17, 0xC9, 0x07,
        0xAB, 0x54, 0x31, 0x72, 0x60, 0xA5, 0x93, 0xA0,
        0xD7, 0xBA, 0x6D, 0x06, 0x61, 0x5D, 0x6E, 0x2C,
        0xA8, 0x4B, 0x86, 0x0E, 0xFF, 0x3C, 0xCB, 0x59,
        0x72, 0x11, 0xBF, 0xE3, 0x6B, 0xDE, 0xF8, 0x06,
        0x9A, 0xFA, 0x36, 0xC5, 0xA7, 0x33, 0x92, 0x72,
        0x26, 0x50, 0xE4, 0x95, 0x7D, 0xCA, 0x59, 0x7A,
        0xCB, 0xA5, 0x60, 0x5B, 0x63, 0xC1, 0x63, 0xCF,
        0xA9, 0x4B, 0x64, 0xDD, 0xD6, 0x23, 0x01, 0xA4,
        0x33, 0x20, 0x83, 0x36, 0x19, 0x72, 0x58, 0x9D,
        0xB0, 0x59, 0x9A, 0x69, 0x4D, 0xD4, 0x54, 0x7A,
        0x5E, 0xE9, 0x19, 0x65, 0x77, 0xC2, 0x2E, 0xD4,
        0x27, 0xAC, 0x89, 0xBB, 0x8B, 0xA3, 0x75, 0x3E,
        0xB7, 0x6C, 0x41, 0xF2, 0xC1, 0x12, 0x9C, 0x8A,
        0x77, 0xD6, 0x80, 0x5F, 0xA7, 0x19, 0xB1, 0xB6,
        0xCA, 0x11, 0xB7, 0x40, 0xA7, 0x8A, 0x3D, 0x41,
        0xB5, 0x33, 0x05, 0x26, 0xAB, 0x87, 0xD5, 0x8D,
        0x59, 0x25, 0x31, 0x5A, 0x14, 0x85, 0xED, 0xC6,
        0x47, 0xC1, 0x60, 0x4E, 0xB3, 0x81, 0x38, 0xDE,
        0x63, 0x7A, 0xD2, 0xC6, 0xCA, 0x5B, 0xE4, 0x4E,
        0x10, 0x08, 0xB2, 0xC0, 0x86, 0x7B, 0x22, 0x9C,
        0xCC, 0x36, 0x61, 0x9E, 0x27, 0x58, 0xC4, 0xC2,
        0x02, 0x9E, 0xAE, 0xB2, 0x6E, 0x7A, 0x80, 0x3F,
        0xCA, 0x30, 0x5A, 0x59, 0xCD, 0x58, 0x5E, 0x11,
        0x7D, 0x69, 0x8E, 0xCE, 0x01, 0x1C, 0xC3, 0xFC,
        0xE5, 0x4D, 0x2E, 0x11, 0x45, 0x45, 0xA2, 0x1A,
        0xC5, 0xBE, 0x67, 0x71, 0xAB, 0x8F, 0x13, 0x12,
        0x2F, 0xAD, 0x29, 0x5E, 0x74, 0x5A, 0x50, 0x3B,
        0x14, 0x2F, 0x91, 0xAE, 0xF7, 0xBD, 0xE9, 0x99,
        0x98, 0x84, 0x5F, 0xDA, 0x04, 0x35, 0x55, 0xC9,
        0xC1, 0xEE, 0x53, 0x5B, 0xE1, 0x25, 0xE5, 0xDC,
        0xE5, 0xD2, 0x66, 0x66, 0x7E, 0x72, 0x3E, 0x67,
        0xB6, 0xBA, 0x89, 0x1C, 0x16, 0xCB, 0xA1, 0x74,
        0x09, 0x8A, 0x3F, 0x35, 0x17, 0x78, 0xB0, 0x88,
        0x8C, 0x95, 0x90, 0xA9, 0x09, 0x0C, 0xD4, 0x04
    };
    static const byte c_512[WC_ML_KEM_512_CIPHER_TEXT_SIZE] = {
        0x16, 0x1C, 0xD2, 0x59, 0xFE, 0xAA, 0x7E, 0xC6,
        0xB2, 0x86, 0x49, 0x8A, 0x9A, 0x6F, 0x69, 0xF8,
        0xB2, 0x62, 0xA2, 0xE2, 0x09, 0x3D, 0x0F, 0xBD,
        0x76, 0xD5, 0xDC, 0x1C, 0x9F, 0xDE, 0x0D, 0xED,
        0xB3, 0x65, 0x81, 0x00, 0x4C, 0xB4, 0x81, 0x12,
        0xF8, 0x52, 0xE7, 0xF8, 0x7F, 0x64, 0x9E, 0x8A,
        0x42, 0xCD, 0x9E, 0x03, 0x49, 0xE7, 0xDA, 0xBD,
        0xF0, 0xA9, 0xAC, 0x1B, 0x52, 0x1C, 0x37, 0xEA,
        0x52, 0x41, 0x37, 0x0A, 0x8A, 0xB2, 0x91, 0x1C,
        0xC7, 0x99, 0x02, 0xC9, 0x5D, 0x28, 0x22, 0x4F,
        0xA8, 0x89, 0x6A, 0xD7, 0x15, 0x20, 0x9E, 0xCD,
        0xD5, 0xD7, 0x84, 0xE9, 0x1D, 0xD9, 0xD0, 0xBE,
        0x91, 0x6B, 0x45, 0x65, 0xF4, 0xD5, 0x66, 0x9A,
        0xEE, 0x0D, 0xEF, 0x93, 0x1E, 0x97, 0x68, 0x29,
        0x4E, 0xEC, 0x52, 0x58, 0xDE, 0x83, 0x91, 0xEC,
        0xE2, 0x71, 0xE7, 0xE4, 0xCF, 0xD9, 0xD2, 0x3A,
        0x79, 0xFA, 0xC3, 0xA8, 0xE0, 0xDB, 0x5D, 0xDD,
        0x6E, 0x01, 0x07, 0x23, 0x56, 0x88, 0xBB, 0xDF,
        0x7B, 0xC5, 0xD5, 0x63, 0x2F, 0x20, 0x6C, 0x63,
        0xA0, 0xC9, 0x56, 0x4F, 0x30, 0x96, 0x5C, 0xA5,
        0x8C, 0x69, 0xFF, 0x92, 0xD2, 0x5A, 0x4F, 0x93,
        0xA0, 0x9E, 0xAB, 0x9B, 0x90, 0x85, 0x94, 0x7E,
        0x07, 0x8A, 0x23, 0xE4, 0xD9, 0xC1, 0x3B, 0x8A,
        0x56, 0xE7, 0x3E, 0x18, 0xDF, 0x42, 0xD6, 0x94,
        0x9F, 0xAF, 0x59, 0x21, 0xF2, 0xE3, 0x73, 0xD4,
        0x50, 0xC8, 0xC0, 0x9D, 0x07, 0xB1, 0x52, 0xA9,
        0x7C, 0x24, 0x54, 0x47, 0x42, 0x94, 0x81, 0xD4,
        0x98, 0xBE, 0xB7, 0x25, 0x6B, 0xC4, 0x7F, 0x68,
        0xF9, 0x92, 0x2B, 0x0B, 0x1C, 0x62, 0xD9, 0xC2,
        0x3F, 0x9F, 0x73, 0x3D, 0xD7, 0x37, 0x92, 0xCF,
        0xC7, 0xB4, 0x3C, 0xBC, 0xEA, 0x27, 0x7D, 0x51,
        0xB2, 0xB8, 0xAD, 0x4A, 0x4F, 0x52, 0x2F, 0x64,
        0x2C, 0xAD, 0x5C, 0x5D, 0xEB, 0x21, 0xF3, 0x62,
        0x7F, 0x8A, 0xF4, 0xD3, 0xE5, 0xBC, 0x9E, 0x91,
        0xD4, 0xCB, 0x2F, 0x12, 0x4B, 0x5B, 0xD7, 0xC2,
        0xF4, 0xA0, 0x50, 0xCA, 0x75, 0x5B, 0xDB, 0x80,
        0x56, 0x60, 0x96, 0x63, 0xFB, 0x95, 0x11, 0xC9,
        0xAD, 0x83, 0xB5, 0x03, 0x90, 0x88, 0xCC, 0x01,
        0xF0, 0xDD, 0x54, 0x35, 0x3B, 0x0D, 0xD7, 0x43,
        0x3F, 0x0C, 0x6C, 0xEE, 0x0D, 0x07, 0x59, 0x59,
        0x81, 0x0D, 0xEC, 0x54, 0x16, 0x52, 0x2B, 0xB1,
        0xF1, 0xF6, 0x55, 0x47, 0xA0, 0xC2, 0xE9, 0xCC,
        0x9B, 0xC1, 0x7F, 0x8D, 0x39, 0xD2, 0x93, 0x09,
        0xEB, 0xE7, 0x9F, 0x21, 0x33, 0x1B, 0x75, 0xE1,
        0x2A, 0xF2, 0xE9, 0x3F, 0x03, 0xF7, 0x4F, 0x7F,
        0x87, 0xD3, 0x60, 0xF1, 0xDA, 0xF8, 0x6C, 0xED,
        0x73, 0x60, 0x92, 0xA2, 0x11, 0xA8, 0x15, 0x88,
        0x59, 0xC4, 0x2E, 0x22, 0x3C, 0xFE, 0x2E, 0x6E,
        0x55, 0x34, 0x37, 0xD8, 0x05, 0x76, 0xCF, 0xD1,
        0x94, 0x4E, 0x97, 0xEE, 0xFF, 0x9B, 0x49, 0xE5,
        0xEC, 0xCF, 0xC6, 0x78, 0xEE, 0x16, 0x52, 0x68,
        0xDF, 0xE3, 0xD3, 0x59, 0x6B, 0x4B, 0x86, 0x20,
        0x4A, 0x81, 0xC6, 0x06, 0x3B, 0x0C, 0xDC, 0xE6,
        0x19, 0xFD, 0xBB, 0x96, 0xDF, 0x7D, 0xE6, 0xE0,
        0xBD, 0x52, 0x70, 0xB4, 0xD5, 0x9C, 0x4D, 0xC5,
        0x08, 0x47, 0x6E, 0x7F, 0x07, 0x08, 0xF9, 0x8C,
        0x7A, 0x4F, 0x66, 0x45, 0xC4, 0x9D, 0x06, 0x10,
        0x0C, 0x76, 0x0C, 0x59, 0x95, 0x28, 0xD1, 0xB8,
        0xBB, 0xFE, 0x62, 0x81, 0x91, 0xCC, 0x08, 0x3C,
        0x8D, 0x22, 0x5A, 0x09, 0x3F, 0x9F, 0x17, 0xE3,
        0x55, 0x74, 0x98, 0x6F, 0x86, 0xBA, 0xA4, 0x68,
        0x98, 0xB5, 0x89, 0xF3, 0xCB, 0x7D, 0xB4, 0x6A,
        0x45, 0xF3, 0xED, 0xD4, 0xFA, 0xC2, 0x08, 0x08,
        0xF4, 0xCD, 0x02, 0x49, 0xDA, 0x69, 0x3F, 0x8F,
        0xAB, 0xFB, 0xD4, 0xE1, 0x0C, 0x02, 0xC6, 0x5B,
        0xA8, 0xC8, 0x61, 0x0F, 0xA8, 0xC6, 0xDF, 0x3D,
        0xBA, 0xEB, 0x67, 0x63, 0xDD, 0x48, 0x2A, 0xF4,
        0x15, 0x58, 0xB1, 0xE1, 0x5C, 0xC9, 0xC7, 0xA7,
        0x2E, 0x07, 0x16, 0x85, 0xAC, 0x19, 0xA0, 0x51,
        0xF1, 0x92, 0x45, 0xB9, 0xF7, 0x7C, 0x30, 0x38,
        0xA5, 0x4E, 0x29, 0x58, 0x62, 0x3E, 0xB8, 0x10,
        0x59, 0x55, 0x60, 0x9E, 0x27, 0xD6, 0x7C, 0xF7,
        0x2E, 0xC5, 0xC4, 0xA8, 0xE9, 0xB9, 0xC2, 0x92,
        0x4A, 0x9E, 0x22, 0x98, 0x50, 0x8B, 0xAB, 0xA1,
        0x3C, 0xF1, 0x11, 0xFD, 0xFB, 0x06, 0x2C, 0x96,
        0x07, 0xAC, 0x1A, 0xAA, 0x6C, 0x63, 0x73, 0x10,
        0xA8, 0x89, 0x4B, 0xF0, 0xB9, 0x6F, 0x0C, 0x19,
        0x13, 0x61, 0x86, 0xB6, 0x18, 0xDF, 0xFB, 0x27,
        0x55, 0x28, 0xBE, 0xD1, 0xCC, 0x27, 0x15, 0xDE,
        0xF4, 0x12, 0xF7, 0x7A, 0x3C, 0xF9, 0x66, 0x45,
        0x73, 0x3B, 0x04, 0x8A, 0x78, 0x47, 0x43, 0x20,
        0xD1, 0xA3, 0x80, 0xF5, 0xEE, 0xDB, 0xDA, 0x21,
        0xFA, 0x01, 0x25, 0xC9, 0x1D, 0x3C, 0x37, 0xC5,
        0x4B, 0xF3, 0x75, 0x2A, 0x1F, 0x84, 0x71, 0xC8,
        0x1F, 0xCA, 0xE2, 0xD3, 0xED, 0xA9, 0x66, 0xE1,
        0x4E, 0x66, 0xF2, 0x23, 0xB0, 0x54, 0xD7, 0x98,
        0x48, 0xFF, 0x94, 0x11, 0xD6, 0x34, 0x02, 0x4A,
        0x09, 0x89, 0x70, 0xAD, 0xE6, 0xA8, 0x8B, 0x5F,
        0x90, 0x69, 0xF7, 0x60, 0x58, 0x4D, 0xC4, 0xCF,
        0xFF, 0xCE, 0xA8, 0xEC, 0xE1, 0x1B, 0xB5, 0x56,
        0x6B, 0xD2, 0x36, 0x0A, 0xB7, 0x07, 0xDF, 0x2D,
        0x21, 0xB6, 0x74, 0x88, 0xD9, 0x31, 0xF0, 0x20,
        0x06, 0x91, 0x76, 0x42, 0x3E, 0x69, 0x44, 0x49,
        0x0C, 0xB3, 0x85, 0xE7, 0x0B, 0x35, 0x8A, 0x25,
        0x34, 0x6B, 0xAF, 0xCD, 0xD0, 0x6D, 0x40, 0x2F,
        0xF2, 0x4D, 0x6C, 0x1E, 0x5F, 0x61, 0xA8, 0x5D
    };
    static const byte kprime_512[WC_ML_KEM_SS_SZ] = {
        0xDF, 0x46, 0x2A, 0xD6, 0x8F, 0x1E, 0xC8, 0x97,
        0x2E, 0xD9, 0xB0, 0x2D, 0x6D, 0xE0, 0x60, 0x4B,
        0xDE, 0xC7, 0x57, 0x20, 0xE0, 0x50, 0x49, 0x73,
        0x51, 0xE6, 0xEC, 0x93, 0x3E, 0x71, 0xF8, 0x82
    };
#endif
#ifndef WOLFSSL_NO_ML_KEM_768
    static const byte dk_768[WC_ML_KEM_768_PRIVATE_KEY_SIZE] = {
        0x1E, 0x4A, 0xC8, 0x7B, 0x1A, 0x69, 0x2A, 0x52,
        0x9F, 0xDB, 0xBA, 0xB9, 0x33, 0x74, 0xC5, 0x7D,
        0x11, 0x0B, 0x10, 0xF2, 0xB1, 0xDD, 0xEB, 0xAC,
        0x0D, 0x19, 0x6B, 0x7B, 0xA6, 0x31, 0xB8, 0xE9,
        0x29, 0x30, 0x28, 0xA8, 0xF3, 0x79, 0x88, 0x8C,
        0x42, 0x2D, 0xC8, 0xD3, 0x2B, 0xBF, 0x22, 0x60,
        0x10, 0xC2, 0xC1, 0xEC, 0x73, 0x18, 0x90, 0x80,
        0x45, 0x6B, 0x05, 0x64, 0xB2, 0x58, 0xB0, 0xF2,
        0x31, 0x31, 0xBC, 0x79, 0xC8, 0xE8, 0xC1, 0x1C,
        0xEF, 0x39, 0x38, 0xB2, 0x43, 0xC5, 0xCE, 0x9C,
        0x0E, 0xDD, 0x37, 0xC8, 0xF9, 0xD2, 0x98, 0x77,
        0xDB, 0xBB, 0x61, 0x5B, 0x9B, 0x5A, 0xC3, 0xC9,
        0x48, 0x48, 0x7E, 0x46, 0x71, 0x96, 0xA9, 0x14,
        0x3E, 0xFB, 0xC7, 0xCE, 0xDB, 0x64, 0xB4, 0x5D,
        0x4A, 0xCD, 0xA2, 0x66, 0x6C, 0xBC, 0x28, 0x04,
        0xF2, 0xC8, 0x66, 0x2E, 0x12, 0x8F, 0x6A, 0x99,
        0x69, 0xEC, 0x15, 0xBC, 0x0B, 0x93, 0x51, 0xF6,
        0xF9, 0x63, 0x46, 0xAA, 0x7A, 0xBC, 0x74, 0x3A,
        0x14, 0xFA, 0x03, 0x0E, 0x37, 0xA2, 0xE7, 0x59,
        0x7B, 0xDD, 0xFC, 0x5A, 0x22, 0xF9, 0xCE, 0xDA,
        0xF8, 0x61, 0x48, 0x32, 0x52, 0x72, 0x10, 0xB2,
        0x6F, 0x02, 0x4C, 0x7F, 0x6C, 0x0D, 0xCF, 0x55,
        0x1E, 0x97, 0xA4, 0x85, 0x87, 0x64, 0xC3, 0x21,
        0xD1, 0x83, 0x4A, 0xD5, 0x1D, 0x75, 0xBB, 0x24,
        0x6D, 0x27, 0x72, 0x37, 0xB7, 0xBD, 0x41, 0xDC,
        0x43, 0x62, 0xD0, 0x63, 0xF4, 0x29, 0x82, 0x92,
        0x27, 0x2D, 0x01, 0x01, 0x17, 0x80, 0xB7, 0x98,
        0x56, 0xB2, 0x96, 0xC4, 0xE9, 0x46, 0x65, 0x8B,
        0x79, 0x60, 0x31, 0x97, 0xC9, 0xB2, 0xA9, 0x9E,
        0xC6, 0x6A, 0xCB, 0x06, 0xCE, 0x2F, 0x69, 0xB5,
        0xA5, 0xA6, 0x1E, 0x9B, 0xD0, 0x6A, 0xD4, 0x43,
        0xCE, 0xB0, 0xC7, 0x4E, 0xD6, 0x53, 0x45, 0xA9,
        0x03, 0xB6, 0x14, 0xE8, 0x13, 0x68, 0xAA, 0xC2,
        0xB3, 0xD2, 0xA7, 0x9C, 0xA8, 0xCC, 0xAA, 0x1C,
        0x3B, 0x88, 0xFB, 0x82, 0xA3, 0x66, 0x32, 0x86,
        0x0B, 0x3F, 0x79, 0x50, 0x83, 0x3F, 0xD0, 0x21,
        0x2E, 0xC9, 0x6E, 0xDE, 0x4A, 0xB6, 0xF5, 0xA0,
        0xBD, 0xA3, 0xEC, 0x60, 0x60, 0xA6, 0x58, 0xF9,
        0x45, 0x7F, 0x6C, 0xC8, 0x7C, 0x6B, 0x62, 0x0C,
        0x1A, 0x14, 0x51, 0x98, 0x74, 0x86, 0xE4, 0x96,
        0x61, 0x2A, 0x10, 0x1D, 0x0E, 0x9C, 0x20, 0x57,
        0x7C, 0x57, 0x1E, 0xDB, 0x52, 0x82, 0x60, 0x8B,
        0xF4, 0xE1, 0xAC, 0x92, 0x6C, 0x0D, 0xB1, 0xC8,
        0x2A, 0x50, 0x4A, 0x79, 0x9D, 0x89, 0x88, 0x5C,
        0xA6, 0x25, 0x2B, 0xD5, 0xB1, 0xC1, 0x83, 0xAF,
        0x70, 0x13, 0x92, 0xA4, 0x07, 0xC0, 0x5B, 0x84,
        0x8C, 0x2A, 0x30, 0x16, 0xC4, 0x06, 0x13, 0xF0,
        0x2A, 0x44, 0x9B, 0x3C, 0x79, 0x26, 0xDA, 0x06,
        0x7A, 0x53, 0x31, 0x16, 0x50, 0x68, 0x40, 0x09,
        0x75, 0x10, 0x46, 0x0B, 0xBF, 0xD3, 0x60, 0x73,
        0xDC, 0xB0, 0xBF, 0xA0, 0x09, 0xB3, 0x6A, 0x91,
        0x23, 0xEA, 0xA6, 0x8F, 0x83, 0x5F, 0x74, 0xA0,
        0x1B, 0x00, 0xD2, 0x09, 0x78, 0x35, 0x96, 0x4D,
        0xF5, 0x21, 0xCE, 0x92, 0x10, 0x78, 0x9C, 0x30,
        0xB7, 0xF0, 0x6E, 0x58, 0x44, 0xB4, 0x44, 0xC5,
        0x33, 0x22, 0x39, 0x6E, 0x47, 0x99, 0xBA, 0xF6,
        0xA8, 0x8A, 0xF7, 0x31, 0x58, 0x60, 0xD0, 0x19,
        0x2D, 0x48, 0xC2, 0xC0, 0xDA, 0x6B, 0x5B, 0xA6,
        0x43, 0x25, 0x54, 0x3A, 0xCD, 0xF5, 0x90, 0x0E,
        0x8B, 0xC4, 0x77, 0xAB, 0x05, 0x82, 0x00, 0x72,
        0xD4, 0x63, 0xAF, 0xFE, 0xD0, 0x97, 0xE0, 0x62,
        0xBD, 0x78, 0xC9, 0x9D, 0x12, 0xB3, 0x85, 0x13,
        0x1A, 0x24, 0x1B, 0x70, 0x88, 0x65, 0xB4, 0x19,
        0x0A, 0xF6, 0x9E, 0xA0, 0xA6, 0x4D, 0xB7, 0x14,
        0x48, 0xA6, 0x08, 0x29, 0x36, 0x9C, 0x75, 0x55,
        0x19, 0x8E, 0x43, 0x8C, 0x9A, 0xBC, 0x31, 0x0B,
        0xC7, 0x01, 0x01, 0x91, 0x3B, 0xB1, 0x2F, 0xAA,
        0x5B, 0xEE, 0xF9, 0x75, 0x84, 0x16, 0x17, 0xC8,
        0x47, 0xCD, 0x6B, 0x33, 0x6F, 0x87, 0x79, 0x87,
        0x75, 0x38, 0x22, 0x02, 0x0B, 0x92, 0xC4, 0xCC,
        0x97, 0x05, 0x5C, 0x9B, 0x1E, 0x0B, 0x12, 0x8B,
        0xF1, 0x1F, 0x50, 0x50, 0x05, 0xB6, 0xAB, 0x0E,
        0x62, 0x77, 0x95, 0xA2, 0x06, 0x09, 0xEF, 0xA9,
        0x91, 0xE5, 0x98, 0xB8, 0x0F, 0x37, 0xB1, 0xC6,
        0xA1, 0xC3, 0xA1, 0xE9, 0xAE, 0xE7, 0x02, 0x8F,
        0x77, 0x57, 0x0A, 0xB2, 0x13, 0x91, 0x28, 0xA0,
        0x01, 0x08, 0xC5, 0x0E, 0xB3, 0x05, 0xCD, 0xB8,
        0xF9, 0xA6, 0x03, 0xA6, 0xB0, 0x78, 0x41, 0x3F,
        0x6F, 0x9B, 0x14, 0xC6, 0xD8, 0x2B, 0x51, 0x99,
        0xCE, 0x59, 0xD8, 0x87, 0x90, 0x2A, 0x28, 0x1A,
        0x02, 0x7B, 0x71, 0x74, 0x95, 0xFE, 0x12, 0x67,
        0x2A, 0x12, 0x7B, 0xBF, 0x9B, 0x25, 0x6C, 0x43,
        0x72, 0x0D, 0x7C, 0x16, 0x0B, 0x28, 0x1C, 0x12,
        0x75, 0x7D, 0xA1, 0x35, 0xB1, 0x93, 0x33, 0x52,
        0xBE, 0x4A, 0xB6, 0x7E, 0x40, 0x24, 0x8A, 0xFC,
        0x31, 0x8E, 0x23, 0x70, 0xC3, 0xB8, 0x20, 0x8E,
        0x69, 0x5B, 0xDF, 0x33, 0x74, 0x59, 0xB9, 0xAC,
        0xBF, 0xE5, 0xB4, 0x87, 0xF7, 0x6E, 0x9B, 0x4B,
        0x40, 0x01, 0xD6, 0xCF, 0x90, 0xCA, 0x8C, 0x69,
        0x9A, 0x17, 0x4D, 0x42, 0x97, 0x2D, 0xC7, 0x33,
        0xF3, 0x33, 0x89, 0xFD, 0xF5, 0x9A, 0x1D, 0xAB,
        0xA8, 0x1D, 0x83, 0x49, 0x55, 0x02, 0x73, 0x34,
        0x18, 0x5A, 0xD0, 0x2C, 0x76, 0xCF, 0x29, 0x48,
        0x46, 0xCA, 0x92, 0x94, 0xBA, 0x0E, 0xD6, 0x67,
        0x41, 0xDD, 0xEC, 0x79, 0x1C, 0xAB, 0x34, 0x19,
        0x6A, 0xC5, 0x65, 0x7C, 0x5A, 0x78, 0x32, 0x1B,
        0x56, 0xC3, 0x33, 0x06, 0xB5, 0x10, 0x23, 0x97,
        0xA5, 0xC0, 0x9C, 0x35, 0x08, 0xF7, 0x6B, 0x48,
        0x28, 0x24, 0x59, 0xF8, 0x1D, 0x0C, 0x72, 0xA4,
        0x3F, 0x73, 0x7B, 0xC2, 0xF1, 0x2F, 0x45, 0x42,
        0x26, 0x28, 0xB6, 0x7D, 0xB5, 0x1A, 0xC1, 0x42,
        0x42, 0x76, 0xA6, 0xC0, 0x8C, 0x3F, 0x76, 0x15,
        0x66, 0x5B, 0xBB, 0x8E, 0x92, 0x81, 0x48, 0xA2,
        0x70, 0xF9, 0x91, 0xBC, 0xF3, 0x65, 0xA9, 0x0F,
        0x87, 0xC3, 0x06, 0x87, 0xB6, 0x88, 0x09, 0xC9,
        0x1F, 0x23, 0x18, 0x13, 0xB8, 0x66, 0xBE, 0xA8,
        0x2E, 0x30, 0x37, 0x4D, 0x80, 0xAA, 0x0C, 0x02,
        0x97, 0x34, 0x37, 0x49, 0x8A, 0x53, 0xB1, 0x4B,
        0xF6, 0xB6, 0xCA, 0x1E, 0xD7, 0x6A, 0xB8, 0xA2,
        0x0D, 0x54, 0xA0, 0x83, 0xF4, 0xA2, 0x6B, 0x7C,
        0x03, 0x8D, 0x81, 0x96, 0x76, 0x40, 0xC2, 0x0B,
        0xF4, 0x43, 0x1E, 0x71, 0xDA, 0xCC, 0xE8, 0x57,
        0x7B, 0x21, 0x24, 0x0E, 0x49, 0x4C, 0x31, 0xF2,
        0xD8, 0x77, 0xDA, 0xF4, 0x92, 0x4F, 0xD3, 0x9D,
        0x82, 0xD6, 0x16, 0x7F, 0xBC, 0xC1, 0xF9, 0xC5,
        0xA2, 0x59, 0xF8, 0x43, 0xE3, 0x09, 0x87, 0xCC,
        0xC4, 0xBC, 0xE7, 0x49, 0x3A, 0x24, 0x04, 0xB5,
        0xE4, 0x43, 0x87, 0xF7, 0x07, 0x42, 0x57, 0x81,
        0xB7, 0x43, 0xFB, 0x55, 0x56, 0x85, 0x58, 0x4E,
        0x25, 0x57, 0xCC, 0x03, 0x8B, 0x1A, 0x9B, 0x3F,
        0x40, 0x43, 0x12, 0x1F, 0x54, 0x72, 0xEB, 0x2B,
        0x96, 0xE5, 0x94, 0x1F, 0xEC, 0x01, 0x1C, 0xEE,
        0xA5, 0x07, 0x91, 0x63, 0x6C, 0x6A, 0xBC, 0x26,
        0xC1, 0x37, 0x7E, 0xE3, 0xB5, 0x14, 0x6F, 0xC7,
        0xC8, 0x5C, 0xB3, 0x35, 0xB1, 0xE7, 0x95, 0xEE,
        0xC2, 0x03, 0x3E, 0xE4, 0x4B, 0x9A, 0xA9, 0x06,
        0x85, 0x24, 0x5E, 0xF7, 0xB4, 0x43, 0x6C, 0x00,
        0x0E, 0x66, 0xBC, 0x8B, 0xCB, 0xF1, 0xCD, 0xB8,
        0x03, 0xAC, 0x14, 0x21, 0xB1, 0xFD, 0xB2, 0x66,
        0xD5, 0x29, 0x1C, 0x83, 0x10, 0x37, 0x3A, 0x8A,
        0x3C, 0xE9, 0x56, 0x2A, 0xB1, 0x97, 0x95, 0x38,
        0x71, 0xAB, 0x99, 0xF3, 0x82, 0xCC, 0x5A, 0xA9,
        0xC0, 0xF2, 0x73, 0xD1, 0xDC, 0xA5, 0x5D, 0x27,
        0x12, 0x85, 0x38, 0x71, 0xE1, 0xA8, 0x3C, 0xB3,
        0xB8, 0x54, 0x50, 0xF7, 0x6D, 0x3F, 0x3C, 0x42,
        0xBA, 0xB5, 0x50, 0x5F, 0x72, 0x12, 0xFD, 0xB6,
        0xB8, 0xB7, 0xF6, 0x02, 0x99, 0x72, 0xA8, 0xF3,
        0x75, 0x1E, 0x4C, 0x94, 0xC1, 0x10, 0x8B, 0x02,
        0xD6, 0xAC, 0x79, 0xF8, 0xD9, 0x38, 0xF0, 0x5A,
        0x1B, 0x2C, 0x22, 0x9B, 0x14, 0xB4, 0x2B, 0x31,
        0xB0, 0x1A, 0x36, 0x40, 0x17, 0xE5, 0x95, 0x78,
        0xC6, 0xB0, 0x33, 0x83, 0x37, 0x74, 0xCB, 0x9B,
        0x57, 0x0F, 0x90, 0x86, 0xB7, 0x22, 0x90, 0x3B,
        0x37, 0x54, 0x46, 0xB4, 0x95, 0xD8, 0xA2, 0x9B,
        0xF8, 0x07, 0x51, 0x87, 0x7A, 0x80, 0xFB, 0x72,
        0x4A, 0x02, 0x10, 0xC3, 0xE1, 0x69, 0x2F, 0x39,
        0x7C, 0x2F, 0x1D, 0xDC, 0x2E, 0x6B, 0xA1, 0x7A,
        0xF8, 0x1B, 0x92, 0xAC, 0xFA, 0xBE, 0xF5, 0xF7,
        0x57, 0x3C, 0xB4, 0x93, 0xD1, 0x84, 0x02, 0x7B,
        0x71, 0x82, 0x38, 0xC8, 0x9A, 0x35, 0x49, 0xB8,
        0x90, 0x5B, 0x28, 0xA8, 0x33, 0x62, 0x86, 0x7C,
        0x08, 0x2D, 0x30, 0x19, 0xD3, 0xCA, 0x70, 0x70,
        0x07, 0x31, 0xCE, 0xB7, 0x3E, 0x84, 0x72, 0xC1,
        0xA3, 0xA0, 0x93, 0x36, 0x1C, 0x5F, 0xEA, 0x6A,
        0x7D, 0x40, 0x95, 0x5D, 0x07, 0xA4, 0x1B, 0x64,
        0xE5, 0x00, 0x81, 0xA3, 0x61, 0xB6, 0x04, 0xCC,
        0x51, 0x84, 0x47, 0xC8, 0xE2, 0x57, 0x65, 0xAB,
        0x7D, 0x68, 0xB2, 0x43, 0x27, 0x52, 0x07, 0xAF,
        0x8C, 0xA6, 0x56, 0x4A, 0x4C, 0xB1, 0xE9, 0x41,
        0x99, 0xDB, 0xA1, 0x87, 0x8C, 0x59, 0xBE, 0xC8,
        0x09, 0xAB, 0x48, 0xB2, 0xF2, 0x11, 0xBA, 0xDC,
        0x6A, 0x19, 0x98, 0xD9, 0xC7, 0x22, 0x7C, 0x13,
        0x03, 0xF4, 0x69, 0xD4, 0x6A, 0x9C, 0x7E, 0x53,
        0x03, 0xF9, 0x8A, 0xBA, 0x67, 0x56, 0x9A, 0xE8,
        0x22, 0x7C, 0x16, 0xBA, 0x1F, 0xB3, 0x24, 0x44,
        0x66, 0xA2, 0x5E, 0x7F, 0x82, 0x36, 0x71, 0x81,
        0x0C, 0xC2, 0x62, 0x06, 0xFE, 0xB2, 0x9C, 0x7E,
        0x2A, 0x1A, 0x91, 0x95, 0x9E, 0xEB, 0x03, 0xA9,
        0x82, 0x52, 0xA4, 0xF7, 0x41, 0x26, 0x74, 0xEB,
        0x9A, 0x4B, 0x27, 0x7E, 0x1F, 0x25, 0x95, 0xFC,
        0xA6, 0x40, 0x33, 0xB4, 0x1B, 0x40, 0x33, 0x08,
        0x12, 0xE9, 0x73, 0x5B, 0x7C, 0x60, 0x75, 0x01,
        0xCD, 0x81, 0x83, 0xA2, 0x2A, 0xFC, 0x33, 0x92,
        0x55, 0x37, 0x44, 0xF3, 0x3C, 0x4D, 0x20, 0x25,
        0x26, 0x94, 0x5C, 0x6D, 0x78, 0xA6, 0x0E, 0x20,
        0x1A, 0x16, 0x98, 0x7A, 0x6F, 0xA5, 0x9D, 0x94,
        0x46, 0x4B, 0x56, 0x50, 0x65, 0x56, 0x78, 0x48,
        0x24, 0xA0, 0x70, 0x58, 0xF5, 0x73, 0x20, 0xE7,
        0x6C, 0x82, 0x5B, 0x93, 0x47, 0xF2, 0x93, 0x6F,
        0x4A, 0x0E, 0x5C, 0xDA, 0xA1, 0x8C, 0xF8, 0x83,
        0x39, 0x45, 0xAE, 0x31, 0x2A, 0x36, 0xB5, 0xF5,
        0xA3, 0x81, 0x0A, 0xAC, 0x82, 0x38, 0x1F, 0xDA,
        0xE4, 0xCB, 0x9C, 0x68, 0x31, 0xD8, 0xEB, 0x8A,
        0xBA, 0xB8, 0x50, 0x41, 0x64, 0x43, 0xD7, 0x39,
        0x08, 0x6B, 0x1C, 0x32, 0x6F, 0xC2, 0xA3, 0x97,
        0x57, 0x04, 0xE3, 0x96, 0xA5, 0x96, 0x80, 0xC3,
        0xB5, 0xF3, 0x60, 0xF5, 0x48, 0x0D, 0x2B, 0x62,
        0x16, 0x9C, 0xD9, 0x4C, 0xA7, 0x1B, 0x37, 0xBC,
        0x58, 0x78, 0xBA, 0x29, 0x85, 0xE0, 0x68, 0xBA,
        0x05, 0x0B, 0x2C, 0xE5, 0x07, 0x26, 0xD4, 0xB4,
        0x45, 0x1B, 0x77, 0xAA, 0xA8, 0x67, 0x6E, 0xAE,
        0x09, 0x49, 0x82, 0x21, 0x01, 0x92, 0x19, 0x7B,
        0x1E, 0x92, 0xA2, 0x7F, 0x59, 0x86, 0x8B, 0x78,
        0x86, 0x78, 0x87, 0xB9, 0xA7, 0x0C, 0x32, 0xAF,
        0x84, 0x63, 0x0A, 0xA9, 0x08, 0x81, 0x43, 0x79,
        0xE6, 0x51, 0x91, 0x50, 0xBA, 0x16, 0x43, 0x9B,
        0x5E, 0x2B, 0x06, 0x03, 0xD0, 0x6A, 0xA6, 0x67,
        0x45, 0x57, 0xF5, 0xB0, 0x98, 0x3E, 0x5C, 0xB6,
        0xA9, 0x75, 0x96, 0x06, 0x9B, 0x01, 0xBB, 0x31,
        0x28, 0xC4, 0x16, 0x68, 0x06, 0x57, 0x20, 0x4F,
        0xD0, 0x76, 0x40, 0x39, 0x2E, 0x16, 0xB1, 0x9F,
        0x33, 0x7A, 0x99, 0xA3, 0x04, 0x84, 0x4E, 0x1A,
        0xA4, 0x74, 0xE9, 0xC7, 0x99, 0x06, 0x29, 0x71,
        0xF6, 0x72, 0x26, 0x89, 0x60, 0xF5, 0xA8, 0x2F,
        0x95, 0x00, 0x70, 0xBB, 0xE9, 0xC2, 0xA7, 0x19,
        0x50, 0xA3, 0x78, 0x5B, 0xDF, 0x0B, 0x84, 0x40,
        0x25, 0x5E, 0xD6, 0x39, 0x28, 0xD2, 0x57, 0x84,
        0x51, 0x68, 0xB1, 0xEC, 0xCC, 0x41, 0x91, 0x32,
        0x5A, 0xA7, 0x66, 0x45, 0x71, 0x9B, 0x28, 0xEB,
        0xD8, 0x93, 0x02, 0xDC, 0x67, 0x23, 0xC7, 0x86,
        0xDF, 0x52, 0x17, 0xB2, 0x43, 0x09, 0x9C, 0xA7,
        0x82, 0x38, 0xE5, 0x7E, 0x64, 0x69, 0x2F, 0x20,
        0x6B, 0x17, 0x7A, 0xBC, 0x25, 0x96, 0x60, 0x39,
        0x5C, 0xD7, 0x86, 0x0F, 0xB3, 0x5A, 0x16, 0xF6,
        0xB2, 0xFE, 0x65, 0x48, 0xC8, 0x5A, 0xB6, 0x63,
        0x30, 0xC5, 0x17, 0xFA, 0x74, 0xCD, 0xF3, 0xCB,
        0x49, 0xD2, 0x6B, 0x11, 0x81, 0x90, 0x1A, 0xF7,
        0x75, 0xA1, 0xE1, 0x80, 0x81, 0x3B, 0x6A, 0x24,
        0xC4, 0x56, 0x82, 0x9B, 0x5C, 0x38, 0x10, 0x4E,
        0xCE, 0x43, 0xC7, 0x6A, 0x43, 0x7A, 0x6A, 0x33,
        0xB6, 0xFC, 0x6C, 0x5E, 0x65, 0xC8, 0xA8, 0x94,
        0x66, 0xC1, 0x42, 0x54, 0x85, 0xB2, 0x9B, 0x9E,
        0x18, 0x54, 0x36, 0x8A, 0xFC, 0xA3, 0x53, 0xE1,
        0x43, 0xD0, 0xA9, 0x0A, 0x6C, 0x6C, 0x9E, 0x7F,
        0xDB, 0x62, 0xA6, 0x06, 0x85, 0x6B, 0x56, 0x14,
        0xF1, 0x2B, 0x64, 0xB7, 0x96, 0x02, 0x0C, 0x35,
        0x34, 0xC3, 0x60, 0x5C, 0xFD, 0xC7, 0x3B, 0x86,
        0x71, 0x4F, 0x41, 0x18, 0x50, 0x22, 0x8A, 0x28,
        0xB8, 0xF4, 0xB4, 0x9E, 0x66, 0x34, 0x16, 0xC8,
        0x4F, 0x7E, 0x38, 0x1F, 0x6A, 0xF1, 0x07, 0x13,
        0x43, 0xBF, 0x9D, 0x39, 0xB4, 0x54, 0x39, 0x24,
        0x0C, 0xC0, 0x38, 0x97, 0x29, 0x5F, 0xEA, 0x08,
        0x0B, 0x14, 0xBB, 0x2D, 0x81, 0x19, 0xA8, 0x80,
        0xE1, 0x64, 0x49, 0x5C, 0x61, 0xBE, 0xBC, 0x71,
        0x39, 0xC1, 0x18, 0x57, 0xC8, 0x5E, 0x17, 0x50,
        0x33, 0x8D, 0x63, 0x43, 0x91, 0x37, 0x06, 0xA5,
        0x07, 0xC9, 0x56, 0x64, 0x64, 0xCD, 0x28, 0x37,
        0xCF, 0x91, 0x4D, 0x1A, 0x3C, 0x35, 0xE8, 0x9B,
        0x23, 0x5C, 0x6A, 0xB7, 0xED, 0x07, 0x8B, 0xED,
        0x23, 0x47, 0x57, 0xC0, 0x2E, 0xF6, 0x99, 0x3D,
        0x4A, 0x27, 0x3C, 0xB8, 0x15, 0x05, 0x28, 0xDA,
        0x4D, 0x76, 0x70, 0x81, 0x77, 0xE9, 0x42, 0x55,
        0x46, 0xC8, 0x3E, 0x14, 0x70, 0x39, 0x76, 0x66,
        0x03, 0xB3, 0x0D, 0xA6, 0x26, 0x8F, 0x45, 0x98,
        0xA5, 0x31, 0x94, 0x24, 0x0A, 0x28, 0x32, 0xA3,
        0xD6, 0x75, 0x33, 0xB5, 0x05, 0x6F, 0x9A, 0xAA,
        0xC6, 0x1B, 0x4B, 0x17, 0xB9, 0xA2, 0x69, 0x3A,
        0xA0, 0xD5, 0x88, 0x91, 0xE6, 0xCC, 0x56, 0xCD,
        0xD7, 0x72, 0x41, 0x09, 0x00, 0xC4, 0x05, 0xAF,
        0x20, 0xB9, 0x03, 0x79, 0x7C, 0x64, 0x87, 0x69,
        0x15, 0xC3, 0x7B, 0x84, 0x87, 0xA1, 0x44, 0x9C,
        0xE9, 0x24, 0xCD, 0x34, 0x5C, 0x29, 0xA3, 0x6E,
        0x08, 0x23, 0x8F, 0x7A, 0x15, 0x7C, 0xC7, 0xE5,
        0x16, 0xAB, 0x5B, 0xA7, 0x3C, 0x80, 0x63, 0xF7,
        0x26, 0xBB, 0x5A, 0x0A, 0x03, 0x19, 0xE5, 0x71,
        0x27, 0x43, 0x8C, 0x7F, 0xC6, 0x01, 0xC9, 0x9C,
        0xCA, 0xAE, 0x4C, 0x1A, 0x83, 0x72, 0x6F, 0xDC,
        0xB5, 0x04, 0x5E, 0xD1, 0xA8, 0x2A, 0x98, 0x5E,
        0xA9, 0x95, 0x39, 0x6D, 0x77, 0x27, 0x2C, 0x66,
        0xCE, 0x49, 0x32, 0x89, 0xF6, 0x11, 0x09, 0x10,
        0xF3, 0x7C, 0x27, 0x41, 0xCE, 0x47, 0x02, 0x6A,
        0x6F, 0x82, 0x61, 0x99, 0x9C, 0x64, 0x82, 0x57,
        0x2B, 0x16, 0x93, 0x91, 0x2E, 0xF1, 0x2E, 0xEB,
        0xEA, 0x7A, 0xCF, 0x92, 0x34, 0xFB, 0x40, 0x9F,
        0x2A, 0x60, 0x90, 0xE6, 0xB0, 0xBF, 0xD8, 0x95,
        0x46, 0x9D, 0x0B, 0x2A, 0x92, 0x1B, 0xB7, 0x23,
        0xF8, 0x7A, 0x33, 0xEA, 0x54, 0x65, 0xAB, 0x90,
        0xF5, 0x14, 0xB6, 0x76, 0x98, 0xC0, 0x76, 0x8B,
        0x6C, 0xA4, 0x98, 0xB0, 0x22, 0xC5, 0x12, 0xFA,
        0x08, 0x75, 0xF0, 0x54, 0xAA, 0x22, 0x65, 0x86,
        0x7E, 0x31, 0xC0, 0xE5, 0x22, 0x65, 0x1E, 0x02,
        0x4A, 0x07, 0xD6, 0x0D, 0xD9, 0xF6, 0x33, 0x16,
        0x69, 0x21, 0xF4, 0x12, 0x6B, 0xC2, 0xB6, 0xAA,
        0x01, 0xCC, 0x15, 0xA0, 0x9B, 0x85, 0xBF, 0xF8,
        0x21, 0x8C, 0x5A, 0xAE, 0x95, 0xBC, 0x1F, 0xFB,
        0x26, 0xAE, 0x5A, 0x13, 0x76, 0x70, 0xF0, 0x49,
        0x10, 0xCA, 0x9D, 0x72, 0x41, 0xB6, 0x66, 0x0C,
        0x39, 0x4C, 0x54, 0x55, 0x91, 0x77, 0x46, 0xA2,
        0x66, 0x82, 0xFB, 0x71, 0xA4, 0x32, 0xEA, 0x95,
        0x30, 0xE8, 0x39, 0xBD, 0xEB, 0x07, 0x43, 0x30,
        0x04, 0xF4, 0x5A, 0x0D, 0xDA, 0xA0, 0xB2, 0x4E,
        0x3A, 0x56, 0x6A, 0x54, 0x08, 0x15, 0xF2, 0x81,
        0xE3, 0xFC, 0x25, 0x9A, 0xC6, 0xCB, 0xC0, 0xAC,
        0xB8, 0xD6, 0x22, 0x68, 0xB6, 0x03, 0xBC, 0x67,
        0x6A, 0xB4, 0x15, 0xC4, 0x74, 0xBB, 0x94, 0x87,
        0x3E, 0x44, 0x87, 0xAE, 0x31, 0xA4, 0xE3, 0x84,
        0x5C, 0x79, 0x90, 0x15, 0x50, 0x89, 0x0E, 0xE8,
        0x78, 0x4E, 0xEF, 0x90, 0x4F, 0xEE, 0x62, 0xBA,
        0x8C, 0x5F, 0x95, 0x2C, 0x68, 0x41, 0x30, 0x52,
        0xE0, 0xA7, 0xE3, 0x38, 0x8B, 0xB8, 0xFF, 0x0A,
        0xD6, 0x02, 0xAE, 0x3E, 0xA1, 0x4D, 0x9D, 0xF6,
        0xDD, 0x5E, 0x4C, 0xC6, 0xA3, 0x81, 0xA4, 0x1D,
        0xA5, 0xC1, 0x37, 0xEC, 0xC4, 0x9D, 0xF5, 0x87,
        0xE1, 0x78, 0xEA, 0xF4, 0x77, 0x02, 0xEC, 0x62,
        0x37, 0x80, 0x69, 0x1A, 0x32, 0x33, 0xF6, 0x9F,
        0x12, 0xBD, 0x9C, 0x9B, 0x96, 0x37, 0xC5, 0x13,
        0x78, 0xAD, 0x71, 0xA8, 0x31, 0x05, 0x52, 0x77,
        0x25, 0x4C, 0xC6, 0x3C, 0x5A, 0xD4, 0xCB, 0x76,
        0xB4, 0xAB, 0x82, 0xE5, 0xFC, 0xA1, 0x35, 0xE8,
        0xD2, 0x6A, 0x6B, 0x3A, 0x89, 0xFA, 0x5B, 0x6F
    };
    static const byte c_768[WC_ML_KEM_768_CIPHER_TEXT_SIZE] = {
        0xA5, 0xC8, 0x1C, 0x76, 0xC2, 0x43, 0x05, 0xE1,
        0xCE, 0x5D, 0x81, 0x35, 0xD4, 0x15, 0x23, 0x68,
        0x2E, 0x9E, 0xE6, 0xD7, 0xB4, 0x0A, 0xD4, 0x1D,
        0xF1, 0xF3, 0x7C, 0x9B, 0x17, 0xDC, 0xE7, 0x80,
        0x76, 0x01, 0x9A, 0x6B, 0x0B, 0x7C, 0x95, 0xC9,
        0xBE, 0x7A, 0xF2, 0x95, 0x07, 0xB2, 0xD5, 0xA6,
        0x98, 0x7C, 0x8E, 0xE3, 0x25, 0x91, 0x90, 0x85,
        0x52, 0x43, 0xE6, 0xE5, 0x6F, 0x56, 0x20, 0x60,
        0x8C, 0x52, 0xD9, 0x6F, 0xAB, 0x10, 0x3A, 0x87,
        0x00, 0xFB, 0xA1, 0xA8, 0x7D, 0xCA, 0x60, 0x78,
        0x11, 0x8A, 0x08, 0x71, 0x76, 0x2C, 0x95, 0x34,
        0xC0, 0xC0, 0xC3, 0x97, 0x8C, 0x91, 0xC3, 0xA0,
        0x1F, 0x0F, 0x60, 0x8D, 0xCF, 0x75, 0x78, 0x15,
        0x43, 0x8F, 0xE8, 0x95, 0x7C, 0x8A, 0x85, 0x91,
        0x83, 0xB1, 0xB6, 0x72, 0x1A, 0x08, 0x65, 0xBE,
        0xBC, 0x79, 0x9D, 0x4E, 0x5C, 0x0E, 0x7B, 0xD3,
        0xEA, 0xE4, 0x85, 0x8E, 0x6A, 0xB6, 0xA2, 0xE7,
        0x65, 0x8E, 0xD8, 0x0D, 0x4E, 0xD1, 0x58, 0xB0,
        0x36, 0xB9, 0x3F, 0xA0, 0x3A, 0xFA, 0x6A, 0xE3,
        0x13, 0x6C, 0xF3, 0xD6, 0x93, 0xC9, 0x11, 0xBC,
        0xC7, 0x59, 0x05, 0xE5, 0xB0, 0xCB, 0x28, 0x65,
        0xB9, 0xE9, 0x88, 0x45, 0x22, 0xA7, 0x77, 0x77,
        0x61, 0x3E, 0x53, 0x11, 0x1D, 0x5A, 0x1C, 0x7D,
        0x3D, 0xAB, 0x73, 0x4C, 0xEB, 0x03, 0x65, 0x7A,
        0xE0, 0xC8, 0x97, 0x63, 0xE9, 0x94, 0x71, 0x05,
        0x47, 0x76, 0xBA, 0xE7, 0xD5, 0x1B, 0x0E, 0x73,
        0xA5, 0xBB, 0x35, 0xAE, 0xC3, 0x0F, 0xF6, 0xBC,
        0x93, 0x68, 0x49, 0x16, 0xFE, 0xF1, 0x16, 0x25,
        0x86, 0x45, 0x2F, 0x42, 0x66, 0x53, 0xE2, 0xCA,
        0x84, 0x4D, 0x57, 0x44, 0x30, 0x7F, 0xF9, 0xAE,
        0xB2, 0x87, 0xA6, 0x44, 0x77, 0x83, 0xB2, 0x1A,
        0x0E, 0x93, 0x9C, 0x81, 0x42, 0x1D, 0x63, 0x1F,
        0x5D, 0xCB, 0x45, 0x2E, 0x51, 0xED, 0x34, 0xE3,
        0xDA, 0xD1, 0xCF, 0x50, 0x4E, 0x0A, 0x3B, 0x0F,
        0x47, 0x11, 0xA8, 0xDC, 0x64, 0x99, 0xD1, 0x69,
        0x1D, 0x10, 0x95, 0x69, 0x33, 0x6C, 0xE1, 0x55,
        0x8A, 0x4C, 0x0A, 0x46, 0x4E, 0x20, 0x87, 0xEA,
        0x8F, 0x9E, 0x3B, 0x18, 0xF7, 0x47, 0xEF, 0x61,
        0xF4, 0x57, 0x6A, 0xEB, 0x42, 0xB1, 0x7C, 0xAD,
        0xB7, 0xF0, 0xFD, 0x84, 0xDA, 0x8E, 0x3A, 0x6F,
        0x47, 0x1D, 0x95, 0xED, 0xFA, 0x65, 0xBE, 0x9E,
        0x6C, 0x9F, 0x6A, 0xE7, 0x56, 0xA2, 0x2A, 0x4F,
        0x1A, 0x5C, 0x54, 0x3C, 0x26, 0xBA, 0x7B, 0xAD,
        0x88, 0xE1, 0x6D, 0x5F, 0x5B, 0x7E, 0x12, 0xE2,
        0xD4, 0xCA, 0x34, 0xB3, 0xA6, 0x4D, 0x17, 0xF8,
        0x7C, 0xCF, 0xC4, 0xFF, 0x8C, 0x5E, 0x4F, 0x53,
        0x75, 0x2A, 0x07, 0x7C, 0x68, 0x72, 0x1E, 0x8C,
        0xC8, 0x17, 0xF9, 0xFF, 0x24, 0x87, 0x61, 0x70,
        0xFF, 0x2A, 0xF8, 0x9F, 0xA9, 0x58, 0x55, 0xA5,
        0xB1, 0xDE, 0x34, 0x7C, 0x07, 0xFD, 0xDB, 0xCF,
        0xE7, 0x26, 0x4A, 0xA5, 0xED, 0x64, 0x01, 0x49,
        0x15, 0x61, 0xD8, 0x31, 0x53, 0x8F, 0x85, 0x2B,
        0x0E, 0xD7, 0xB9, 0xE8, 0xEB, 0xAF, 0xFC, 0x06,
        0x02, 0x84, 0xF2, 0x2D, 0x2B, 0xAE, 0xE5, 0x6F,
        0xA9, 0xF6, 0xD0, 0x14, 0x32, 0xA1, 0x15, 0xA2,
        0xD6, 0xA6, 0x4C, 0x38, 0xAE, 0x0A, 0x50, 0xBA,
        0x36, 0x2F, 0xB5, 0x7B, 0x53, 0xE3, 0xE8, 0x55,
        0xB8, 0x3C, 0xE8, 0xC4, 0x22, 0x74, 0x04, 0x55,
        0x99, 0xF6, 0x5F, 0xA6, 0xA8, 0x92, 0x1D, 0x85,
        0xF9, 0x4E, 0xD2, 0x30, 0xB5, 0x16, 0x71, 0x2D,
        0xB6, 0xFD, 0x2F, 0xF2, 0x8B, 0x3A, 0x33, 0x71,
        0xD9, 0xBE, 0x05, 0x8A, 0xE7, 0x5C, 0x2F, 0xA5,
        0x91, 0xB7, 0xEC, 0x3C, 0x3D, 0xAA, 0x1F, 0x76,
        0x42, 0xBC, 0x26, 0xC3, 0x24, 0xC0, 0x80, 0x90,
        0x60, 0x7E, 0x66, 0x62, 0x15, 0x4D, 0xB3, 0x7C,
        0xF7, 0x47, 0x96, 0x7A, 0x1F, 0x9F, 0xC2, 0x90,
        0x89, 0xF5, 0x70, 0xEB, 0xE6, 0x0E, 0xEE, 0xF8,
        0x9F, 0xD2, 0x44, 0x81, 0x02, 0x8C, 0x85, 0xAE,
        0xF1, 0xDC, 0x3B, 0x09, 0xF2, 0x2C, 0xD3, 0x69,
        0x1B, 0xBB, 0xB8, 0x21, 0xC7, 0xA8, 0xA0, 0xF3,
        0x5A, 0xD1, 0x2B, 0xE1, 0xDD, 0x19, 0x9B, 0x97,
        0x70, 0x48, 0xF3, 0xD4, 0x8C, 0x16, 0xBB, 0x2C,
        0xA9, 0x4C, 0xEC, 0xB8, 0x92, 0x87, 0x70, 0xD5,
        0xBB, 0x32, 0x9A, 0x03, 0x27, 0xE0, 0xB2, 0x86,
        0xFA, 0xA1, 0xC6, 0x52, 0x81, 0x03, 0x1A, 0x31,
        0xC8, 0x4F, 0x2E, 0xDC, 0x9C, 0x04, 0xD4, 0x75,
        0xED, 0x4E, 0x12, 0x8E, 0x51, 0xEF, 0xA9, 0x7D,
        0x01, 0x48, 0xCB, 0xA6, 0xC9, 0x5F, 0x67, 0x4C,
        0x58, 0x9F, 0x30, 0x1C, 0x26, 0x5B, 0xED, 0x70,
        0x8E, 0x9A, 0xD8, 0xDA, 0x3C, 0x5C, 0xEC, 0xBD,
        0xEE, 0xED, 0x35, 0xEF, 0x1E, 0x25, 0x31, 0x32,
        0xBA, 0x89, 0x92, 0x0D, 0x78, 0x6B, 0x88, 0x23,
        0x0B, 0x01, 0x3B, 0xCF, 0x2D, 0xC9, 0x2D, 0x6B,
        0x15, 0x7A, 0xFA, 0x8D, 0xA8, 0x59, 0x2C, 0xD0,
        0x74, 0x3D, 0x49, 0x82, 0xBE, 0x60, 0xD7, 0xC2,
        0xD5, 0xC4, 0x72, 0xAB, 0x9F, 0xA7, 0xF4, 0xCC,
        0x3D, 0x12, 0xB0, 0xEB, 0xAF, 0x0A, 0xBE, 0x55,
        0x5C, 0x75, 0x80, 0x54, 0x26, 0x84, 0x4D, 0xD9,
        0x42, 0x86, 0x43, 0xF8, 0x44, 0x06, 0xA1, 0xB8,
        0xD6, 0xFA, 0xED, 0xFD, 0x8A, 0xE6, 0xE7, 0x3A,
        0x72, 0x77, 0x2A, 0x21, 0x59, 0xAC, 0xAB, 0xD9,
        0x72, 0xAE, 0xB6, 0xF7, 0xDE, 0x09, 0x1A, 0xC5,
        0xFD, 0xD7, 0xF4, 0x9A, 0x3D, 0xC6, 0x64, 0x1C,
        0xDF, 0x62, 0x44, 0x6B, 0x4B, 0x04, 0xA3, 0x1F,
        0x73, 0xB8, 0x0A, 0x62, 0xF8, 0x0A, 0x40, 0x4A,
        0x8C, 0xB1, 0x8C, 0xE3, 0xE6, 0x54, 0x80, 0xEF,
        0x7B, 0x52, 0xBF, 0x00, 0x91, 0x11, 0x7E, 0x5D,
        0x08, 0xEA, 0xE1, 0xB0, 0xAA, 0xBB, 0x72, 0xE6,
        0xDF, 0xFF, 0xF7, 0x6F, 0x6E, 0x44, 0xBB, 0xD7,
        0xEA, 0x57, 0x0D, 0x66, 0x04, 0xBC, 0x2E, 0x74,
        0x31, 0x8B, 0xAF, 0xA3, 0x15, 0xA3, 0x88, 0x61,
        0xAA, 0x1B, 0x21, 0xAF, 0xB2, 0xA5, 0x3F, 0x26,
        0x14, 0xF1, 0xD6, 0x40, 0x07, 0x59, 0x84, 0xAE,
        0x62, 0xE2, 0xFC, 0xA1, 0xD1, 0xB4, 0xDB, 0x36,
        0x9F, 0x15, 0x70, 0x5C, 0xE7, 0xD4, 0xDF, 0x8A,
        0xE9, 0x82, 0x64, 0x50, 0x10, 0x51, 0xC0, 0xDE,
        0xF2, 0x1D, 0x64, 0x5D, 0x49, 0x62, 0x5A, 0xF0,
        0x2C, 0xA4, 0x28, 0xD9, 0xF0, 0xC2, 0xCD, 0x9F,
        0xBA, 0xEE, 0xAB, 0x97, 0xE8, 0xE9, 0x15, 0x16,
        0x62, 0xB6, 0x99, 0x2B, 0x4C, 0x99, 0xAB, 0x1B,
        0x92, 0x5D, 0x08, 0x92, 0x03, 0x63, 0x37, 0x3F,
        0x76, 0xD3, 0xFD, 0xF0, 0x82, 0x8C, 0xAA, 0x69,
        0xC8, 0xB1, 0xBD, 0xC6, 0xF5, 0x21, 0xDF, 0x64,
        0x1C, 0xF1, 0xC8, 0xA4, 0xE7, 0xEF, 0x0C, 0x23,
        0x28, 0x9A, 0x4E, 0x2C, 0xF1, 0x8A, 0xCE, 0xBB,
        0xE4, 0xC1, 0xE6, 0x83, 0x69, 0xBD, 0x52, 0x35,
        0x12, 0x01, 0x42, 0xEC, 0xDD, 0x1A, 0x73, 0x81,
        0x1E, 0x2E, 0x53, 0x3A, 0x64, 0x7D, 0x7A, 0xEE,
        0x16, 0xDA, 0xA0, 0x3B, 0x68, 0x36, 0x39, 0xDC,
        0xF1, 0xE1, 0xF1, 0xE7, 0x1C, 0xFA, 0xED, 0x48,
        0xF6, 0x9A, 0xEC, 0x3E, 0x83, 0x17, 0x33, 0xDA,
        0x19, 0xCE, 0xBE, 0xC1, 0xDD, 0xBF, 0x71, 0xCB,
        0xAE, 0x08, 0x00, 0xF2, 0xF6, 0xD6, 0x4A, 0x09,
        0x6E, 0xC4, 0x95, 0xD6, 0x2F, 0x43, 0x44, 0xF7,
        0xAA, 0x56, 0x21, 0xB3, 0x22, 0x35, 0x3A, 0x79,
        0x5A, 0xA0, 0x99, 0xEA, 0x3A, 0x07, 0x02, 0x72,
        0xD0, 0x53, 0xD4, 0x65, 0x3A, 0x20, 0xCF, 0x21,
        0x0E, 0xAA, 0xF1, 0x2C, 0xAE, 0x60, 0x23, 0xD8,
        0xE5, 0x11, 0x8D, 0xF0, 0x4B, 0x38, 0x4A, 0x44,
        0xD1, 0xED, 0xB9, 0x1C, 0x44, 0x98, 0x9E, 0xF7,
        0xEE, 0x57, 0xF2, 0xBF, 0x81, 0xA2, 0x4B, 0xDC,
        0x76, 0x80, 0x7D, 0xA9, 0x67, 0xEE, 0x65, 0x25,
        0x41, 0x0C, 0x5C, 0x48, 0x50, 0x67, 0xEF, 0xC3,
        0xD3, 0x9A, 0x9A, 0xD4, 0x2C, 0xC7, 0x53, 0xBA,
        0xA5, 0x9A, 0x1F, 0xD2, 0x8A, 0xF3, 0x5C, 0x00,
        0xD1, 0x8A, 0x40, 0x6A, 0x28, 0xFC, 0x79, 0xBA
    };
    static const byte kprime_768[WC_ML_KEM_SS_SZ] = {
        0xDC, 0x5B, 0x88, 0x88, 0xBC, 0x1E, 0xBA, 0x5C,
        0x19, 0x69, 0xC2, 0x11, 0x64, 0xEA, 0x43, 0xE2,
        0x2E, 0x7A, 0xC0, 0xCD, 0x01, 0x2A, 0x2F, 0x26,
        0xCB, 0x8C, 0x48, 0x7E, 0x69, 0xEF, 0x7C, 0xE4
    };
#endif
#ifndef WOLFSSL_NO_ML_KEM_1024
    static const byte dk_1024[WC_ML_KEM_1024_PRIVATE_KEY_SIZE] = {
        0x84, 0x45, 0xC3, 0x36, 0xF3, 0x51, 0x8B, 0x29,
        0x81, 0x63, 0xDC, 0xBB, 0x63, 0x57, 0x59, 0x79,
        0x83, 0xCA, 0x2E, 0x87, 0x3D, 0xCB, 0x49, 0x61,
        0x0C, 0xF5, 0x2F, 0x14, 0xDB, 0xCB, 0x94, 0x7C,
        0x1F, 0x3E, 0xE9, 0x26, 0x69, 0x67, 0x27, 0x6B,
        0x0C, 0x57, 0x6C, 0xF7, 0xC3, 0x0E, 0xE6, 0xB9,
        0x3D, 0xEA, 0x51, 0x18, 0x67, 0x6C, 0xBE, 0xE1,
        0xB1, 0xD4, 0x79, 0x42, 0x06, 0xFB, 0x36, 0x9A,
        0xBA, 0x41, 0x16, 0x7B, 0x43, 0x93, 0x85, 0x5C,
        0x84, 0xEB, 0xA8, 0xF3, 0x23, 0x73, 0xC0, 0x5B,
        0xAE, 0x76, 0x31, 0xC8, 0x02, 0x74, 0x4A, 0xAD,
        0xB6, 0xC2, 0xDE, 0x41, 0x25, 0x0C, 0x49, 0x43,
        0x15, 0x23, 0x0B, 0x52, 0x82, 0x6C, 0x34, 0x58,
        0x7C, 0xB2, 0x1B, 0x18, 0x3B, 0x49, 0xB2, 0xA5,
        0xAC, 0x04, 0x92, 0x1A, 0xC6, 0xBF, 0xAC, 0x1B,
        0x24, 0xA4, 0xB3, 0x7A, 0x93, 0xA4, 0xB1, 0x68,
        0xCC, 0xE7, 0x59, 0x1B, 0xE6, 0x11, 0x1F, 0x47,
        0x62, 0x60, 0xF2, 0x76, 0x29, 0x59, 0xF5, 0xC1,
        0x64, 0x01, 0x18, 0xC2, 0x42, 0x37, 0x72, 0xE2,
        0xAD, 0x03, 0xDC, 0x71, 0x68, 0xA3, 0x8C, 0x6D,
        0xD3, 0x9F, 0x5F, 0x72, 0x54, 0x26, 0x42, 0x80,
        0xC8, 0xBC, 0x10, 0xB9, 0x14, 0x16, 0x80, 0x70,
        0x47, 0x2F, 0xA8, 0x80, 0xAC, 0xB8, 0x60, 0x1A,
        0x8A, 0x08, 0x37, 0xF2, 0x5F, 0xE1, 0x94, 0x68,
        0x7C, 0xD6, 0x8B, 0x7D, 0xE2, 0x34, 0x0F, 0x03,
        0x6D, 0xAD, 0x89, 0x1D, 0x38, 0xD1, 0xB0, 0xCE,
        0x9C, 0x26, 0x33, 0x35, 0x5C, 0xF5, 0x7B, 0x50,
        0xB8, 0x96, 0x03, 0x6F, 0xCA, 0x26, 0x0D, 0x26,
        0x69, 0xF8, 0x5B, 0xAC, 0x79, 0x71, 0x4F, 0xDA,
        0xFB, 0x41, 0xEF, 0x80, 0xB8, 0xC3, 0x02, 0x64,
        0xC3, 0x13, 0x86, 0xAE, 0x60, 0xB0, 0x5F, 0xAA,
        0x54, 0x2A, 0x26, 0xB4, 0x1E, 0xB8, 0x5F, 0x67,
        0x06, 0x8F, 0x08, 0x80, 0x34, 0xFF, 0x67, 0xAA,
        0x2E, 0x81, 0x5A, 0xAB, 0x8B, 0xCA, 0x6B, 0xF7,
        0x1F, 0x70, 0xEC, 0xC3, 0xCB, 0xCB, 0xC4, 0x5E,
        0xF7, 0x01, 0xFC, 0xD5, 0x42, 0xBD, 0x21, 0xC7,
        0xB0, 0x95, 0x68, 0xF3, 0x69, 0xC6, 0x69, 0xF3,
        0x96, 0x47, 0x38, 0x44, 0xFB, 0xA1, 0x49, 0x57,
        0xF5, 0x19, 0x74, 0xD8, 0x52, 0xB9, 0x78, 0x01,
        0x46, 0x03, 0xA2, 0x10, 0xC0, 0x19, 0x03, 0x62,
        0x87, 0x00, 0x89, 0x94, 0xF2, 0x12, 0x55, 0xB2,
        0x50, 0x99, 0xAD, 0x82, 0xAA, 0x13, 0x24, 0x38,
        0x96, 0x3B, 0x2C, 0x0A, 0x47, 0xCD, 0xF5, 0xF3,
        0x2B, 0xA4, 0x6B, 0x76, 0xC7, 0xA6, 0x55, 0x9F,
        0x18, 0xBF, 0xD5, 0x55, 0xB7, 0x62, 0xE4, 0x87,
        0xB6, 0xAC, 0x99, 0x2F, 0xE2, 0x0E, 0x28, 0x3C,
        0xA0, 0xB3, 0xF6, 0x16, 0x44, 0x96, 0x95, 0x59,
        0x95, 0xC3, 0xB2, 0x8A, 0x57, 0xBB, 0xC2, 0x98,
        0x26, 0xF0, 0x6F, 0xB3, 0x8B, 0x25, 0x34, 0x70,
        0xAF, 0x63, 0x1B, 0xC4, 0x6C, 0x3A, 0x8F, 0x9C,
        0xE8, 0x24, 0x32, 0x19, 0x85, 0xDD, 0x01, 0xC0,
        0x5F, 0x69, 0xB8, 0x24, 0xF9, 0x16, 0x63, 0x3B,
        0x40, 0x65, 0x4C, 0x75, 0xAA, 0xEB, 0x93, 0x85,
        0x57, 0x6F, 0xFD, 0xE2, 0x99, 0x0A, 0x6B, 0x0A,
        0x3B, 0xE8, 0x29, 0xD6, 0xD8, 0x4E, 0x34, 0xF1,
        0x78, 0x05, 0x89, 0xC7, 0x92, 0x04, 0xC6, 0x3C,
        0x79, 0x8F, 0x55, 0xD2, 0x31, 0x87, 0xE4, 0x61,
        0xD4, 0x8C, 0x21, 0xE5, 0xC0, 0x47, 0xE5, 0x35,
        0xB1, 0x9F, 0x45, 0x8B, 0xBA, 0x13, 0x45, 0xB9,
        0xE4, 0x1E, 0x0C, 0xB4, 0xA9, 0xC2, 0xD8, 0xC4,
        0x0B, 0x49, 0x0A, 0x3B, 0xAB, 0xC5, 0x53, 0xB3,
        0x02, 0x6B, 0x16, 0x72, 0xD2, 0x8C, 0xBC, 0x8B,
        0x49, 0x8A, 0x3A, 0x99, 0x57, 0x9A, 0x83, 0x2F,
        0xEA, 0xE7, 0x46, 0x10, 0xF0, 0xB6, 0x25, 0x0C,
        0xC3, 0x33, 0xE9, 0x49, 0x3E, 0xB1, 0x62, 0x1E,
        0xD3, 0x4A, 0xA4, 0xAB, 0x17, 0x5F, 0x2C, 0xA2,
        0x31, 0x15, 0x25, 0x09, 0xAC, 0xB6, 0xAC, 0x86,
        0xB2, 0x0F, 0x6B, 0x39, 0x10, 0x84, 0x39, 0xE5,
        0xEC, 0x12, 0xD4, 0x65, 0xA0, 0xFE, 0xF3, 0x50,
        0x03, 0xE1, 0x42, 0x77, 0xA2, 0x18, 0x12, 0x14,
        0x6B, 0x25, 0x44, 0x71, 0x6D, 0x6A, 0xB8, 0x2D,
        0x1B, 0x07, 0x26, 0xC2, 0x7A, 0x98, 0xD5, 0x89,
        0xEB, 0xDA, 0xCC, 0x4C, 0x54, 0xBA, 0x77, 0xB2,
        0x49, 0x8F, 0x21, 0x7E, 0x14, 0xE3, 0x4E, 0x66,
        0x02, 0x5A, 0x2A, 0x14, 0x3A, 0x99, 0x25, 0x20,
        0xA6, 0x1C, 0x06, 0x72, 0xCC, 0x9C, 0xCE, 0xD7,
        0xC9, 0x45, 0x0C, 0x68, 0x3E, 0x90, 0xA3, 0xE4,
        0x65, 0x1D, 0xB6, 0x23, 0xA6, 0xDB, 0x39, 0xAC,
        0x26, 0x12, 0x5B, 0x7F, 0xC1, 0x98, 0x6D, 0x7B,
        0x04, 0x93, 0xB8, 0xB7, 0x2D, 0xE7, 0x70, 0x7D,
        0xC2, 0x0B, 0xBD, 0xD4, 0x37, 0x13, 0x15, 0x6A,
        0xF7, 0xD9, 0x43, 0x0E, 0xF4, 0x53, 0x99, 0x66,
        0x3C, 0x22, 0x02, 0x73, 0x91, 0x68, 0x69, 0x2D,
        0xD6, 0x57, 0x54, 0x5B, 0x05, 0x6D, 0x9C, 0x92,
        0x38, 0x5A, 0x7F, 0x41, 0x4B, 0x34, 0xB9, 0x0C,
        0x79, 0x60, 0xD5, 0x7B, 0x35, 0xBA, 0x7D, 0xDE,
        0x7B, 0x81, 0xFC, 0xA0, 0x11, 0x9D, 0x74, 0x1B,
        0x12, 0x78, 0x09, 0x26, 0x01, 0x8F, 0xE4, 0xC8,
        0x03, 0x0B, 0xF0, 0x38, 0xE1, 0x8B, 0x4F, 0xA3,
        0x37, 0x43, 0xD0, 0xD3, 0xC8, 0x46, 0x41, 0x7E,
        0x9D, 0x59, 0x15, 0xC2, 0x46, 0x31, 0x59, 0x38,
        0xB1, 0xE2, 0x33, 0x61, 0x45, 0x01, 0xD0, 0x26,
        0x95, 0x95, 0x51, 0x25, 0x8B, 0x23, 0x32, 0x30,
        0xD4, 0x28, 0xB1, 0x81, 0xB1, 0x32, 0xF1, 0xD0,
        0xB0, 0x26, 0x06, 0x7B, 0xA8, 0x16, 0x99, 0x9B,
        0xC0, 0xCD, 0x6B, 0x54, 0x7E, 0x54, 0x8B, 0x63,
        0xC9, 0xEA, 0xA0, 0x91, 0xBA, 0xC4, 0x93, 0xDC,
        0x59, 0x8D, 0xBC, 0x2B, 0x0E, 0x14, 0x6A, 0x25,
        0x91, 0xC2, 0xA8, 0xC0, 0x09, 0xDD, 0x51, 0x70,
        0xAA, 0xE0, 0x27, 0xC5, 0x41, 0xA1, 0xB5, 0xE6,
        0x6E, 0x45, 0xC6, 0x56, 0x12, 0x98, 0x4C, 0x46,
        0x77, 0x04, 0x93, 0xEC, 0x89, 0x6E, 0xF2, 0x5A,
        0xA9, 0x30, 0x5E, 0x9F, 0x06, 0x69, 0x2C, 0xD0,
        0xB2, 0xF0, 0x69, 0x62, 0xE2, 0x05, 0xBE, 0xBE,
        0x11, 0x3A, 0x34, 0xEB, 0xB1, 0xA4, 0x83, 0x0A,
        0x9B, 0x37, 0x49, 0x64, 0x1B, 0xB9, 0x35, 0x00,
        0x7B, 0x23, 0xB2, 0x4B, 0xFE, 0x57, 0x69, 0x56,
        0x25, 0x4D, 0x7A, 0x35, 0xAA, 0x49, 0x6A, 0xC4,
        0x46, 0xC6, 0x7A, 0x7F, 0xEC, 0x85, 0xA6, 0x00,
        0x57, 0xE8, 0x58, 0x06, 0x17, 0xBC, 0xB3, 0xFA,
        0xD1, 0x5C, 0x76, 0x44, 0x0F, 0xED, 0x54, 0xCC,
        0x78, 0x93, 0x94, 0xFE, 0xA2, 0x44, 0x52, 0xCC,
        0x6B, 0x05, 0x85, 0xB7, 0xEB, 0x0A, 0x88, 0xBB,
        0xA9, 0x50, 0x0D, 0x98, 0x00, 0xE6, 0x24, 0x1A,
        0xFE, 0xB5, 0x23, 0xB5, 0x5A, 0x96, 0xA5, 0x35,
        0x15, 0x1D, 0x10, 0x49, 0x57, 0x32, 0x06, 0xE5,
        0x9C, 0x7F, 0xEB, 0x07, 0x09, 0x66, 0x82, 0x36,
        0x34, 0xF7, 0x7D, 0x5F, 0x12, 0x91, 0x75, 0x5A,
        0x24, 0x31, 0x19, 0x62, 0x1A, 0xF8, 0x08, 0x4A,
        0xB7, 0xAC, 0x1E, 0x22, 0xA0, 0x56, 0x8C, 0x62,
        0x01, 0x41, 0x7C, 0xBE, 0x36, 0x55, 0xD8, 0xA0,
        0x8D, 0xD5, 0xB5, 0x13, 0x88, 0x4C, 0x98, 0xD5,
        0xA4, 0x93, 0xFD, 0x49, 0x38, 0x2E, 0xA4, 0x18,
        0x60, 0xF1, 0x33, 0xCC, 0xD6, 0x01, 0xE8, 0x85,
        0x96, 0x64, 0x26, 0xA2, 0xB1, 0xF2, 0x3D, 0x42,
        0xD8, 0x2E, 0x24, 0x58, 0x2D, 0x99, 0x72, 0x51,
        0x92, 0xC2, 0x17, 0x77, 0x46, 0x7B, 0x14, 0x57,
        0xB1, 0xDD, 0x42, 0x9A, 0x0C, 0x41, 0xA5, 0xC3,
        0xD7, 0x04, 0xCE, 0xA0, 0x62, 0x78, 0xC5, 0x99,
        0x41, 0xB4, 0x38, 0xC6, 0x27, 0x27, 0x09, 0x78,
        0x09, 0xB4, 0x53, 0x0D, 0xBE, 0x83, 0x7E, 0xA3,
        0x96, 0xB6, 0xD3, 0x10, 0x77, 0xFA, 0xD3, 0x73,
        0x30, 0x53, 0x98, 0x9A, 0x84, 0x42, 0xAA, 0xC4,
        0x25, 0x5C, 0xB1, 0x63, 0xB8, 0xCA, 0x2F, 0x27,
        0x50, 0x1E, 0xA9, 0x67, 0x30, 0x56, 0x95, 0xAB,
        0xD6, 0x59, 0xAA, 0x02, 0xC8, 0x3E, 0xE6, 0x0B,
        0xB5, 0x74, 0x20, 0x3E, 0x99, 0x37, 0xAE, 0x1C,
        0x62, 0x1C, 0x8E, 0xCB, 0x5C, 0xC1, 0xD2, 0x1D,
        0x55, 0x69, 0x60, 0xB5, 0xB9, 0x16, 0x1E, 0xA9,
        0x6F, 0xFF, 0xEB, 0xAC, 0x72, 0xE1, 0xB8, 0xA6,
        0x15, 0x4F, 0xC4, 0xD8, 0x8B, 0x56, 0xC0, 0x47,
        0x41, 0xF0, 0x90, 0xCB, 0xB1, 0x56, 0xA7, 0x37,
        0xC9, 0xE6, 0xA2, 0x2B, 0xA8, 0xAC, 0x70, 0x4B,
        0xC3, 0x04, 0xF8, 0xE1, 0x7E, 0x5E, 0xA8, 0x45,
        0xFD, 0xE5, 0x9F, 0xBF, 0x78, 0x8C, 0xCE, 0x0B,
        0x97, 0xC8, 0x76, 0x1F, 0x89, 0xA2, 0x42, 0xF3,
        0x05, 0x25, 0x83, 0xC6, 0x84, 0x4A, 0x63, 0x20,
        0x31, 0xC9, 0x64, 0xA6, 0xC4, 0xA8, 0x5A, 0x12,
        0x8A, 0x28, 0x61, 0x9B, 0xA1, 0xBB, 0x3D, 0x1B,
        0xEA, 0x4B, 0x49, 0x84, 0x1F, 0xC8, 0x47, 0x61,
        0x4A, 0x06, 0x68, 0x41, 0xF5, 0x2E, 0xD0, 0xEB,
        0x8A, 0xE0, 0xB8, 0xB0, 0x96, 0xE9, 0x2B, 0x81,
        0x95, 0x40, 0x58, 0x15, 0xB2, 0x31, 0x26, 0x6F,
        0x36, 0xB1, 0x8C, 0x1A, 0x53, 0x33, 0x3D, 0xAB,
        0x95, 0xD2, 0xA9, 0xA3, 0x74, 0xB5, 0x47, 0x8A,
        0x4A, 0x41, 0xFB, 0x87, 0x59, 0x95, 0x7C, 0x9A,
        0xB2, 0x2C, 0xAE, 0x54, 0x5A, 0xB5, 0x44, 0xBA,
        0x8D, 0xD0, 0x5B, 0x83, 0xF3, 0xA6, 0x13, 0xA2,
        0x43, 0x7A, 0xDB, 0x07, 0x3A, 0x96, 0x35, 0xCB,
        0x4B, 0xBC, 0x96, 0x5F, 0xB4, 0x54, 0xCF, 0x27,
        0xB2, 0x98, 0xA4, 0x0C, 0xD0, 0xDA, 0x3B, 0x8F,
        0x9C, 0xA9, 0x9D, 0x8C, 0xB4, 0x28, 0x6C, 0x5E,
        0xB4, 0x76, 0x41, 0x67, 0x96, 0x07, 0x0B, 0xA5,
        0x35, 0xAA, 0xA5, 0x8C, 0xDB, 0x45, 0x1C, 0xD6,
        0xDB, 0x5C, 0xBB, 0x0C, 0xA2, 0x0F, 0x0C, 0x71,
        0xDE, 0x97, 0xC3, 0x0D, 0xA9, 0x7E, 0xC7, 0x90,
        0x6D, 0x06, 0xB4, 0xB9, 0x39, 0x39, 0x60, 0x28,
        0xC4, 0x6B, 0xA0, 0xE7, 0xA8, 0x65, 0xBC, 0x83,
        0x08, 0xA3, 0x81, 0x0F, 0x12, 0x12, 0x00, 0x63,
        0x39, 0xF7, 0xBC, 0x16, 0x9B, 0x16, 0x66, 0xFD,
        0xF4, 0x75, 0x91, 0x1B, 0xBC, 0x8A, 0xAA, 0xB4,
        0x17, 0x55, 0xC9, 0xA8, 0xAA, 0xBF, 0xA2, 0x3C,
        0x0E, 0x37, 0xF8, 0x4F, 0xE4, 0x69, 0x99, 0xE0,
        0x30, 0x49, 0x4B, 0x92, 0x98, 0xEF, 0x99, 0x34,
        0xE8, 0xA6, 0x49, 0xC0, 0xA5, 0xCC, 0xE2, 0xB2,
        0x2F, 0x31, 0x80, 0x9A, 0xFE, 0xD2, 0x39, 0x55,
        0xD8, 0x78, 0x81, 0xD9, 0x9F, 0xC1, 0xD3, 0x52,
        0x89, 0x6C, 0xAC, 0x90, 0x55, 0xBE, 0xA0, 0xD0,
        0x16, 0xCC, 0xBA, 0x78, 0x05, 0xA3, 0xA5, 0x0E,
        0x22, 0x16, 0x30, 0x37, 0x9B, 0xD0, 0x11, 0x35,
        0x22, 0x1C, 0xAD, 0x5D, 0x95, 0x17, 0xC8, 0xCC,
        0x42, 0x63, 0x7B, 0x9F, 0xC0, 0x71, 0x8E, 0x9A,
        0x9B, 0xB4, 0x94, 0x5C, 0x72, 0xD8, 0xD1, 0x1D,
        0x3D, 0x65, 0x9D, 0x83, 0xA3, 0xC4, 0x19, 0x50,
        0x9A, 0xF5, 0xB4, 0x70, 0xDD, 0x89, 0xB7, 0xF3,
        0xAC, 0xCF, 0x5F, 0x35, 0xCF, 0xC3, 0x22, 0x11,
        0x5F, 0xD6, 0x6A, 0x5C, 0xD2, 0x87, 0x56, 0x51,
        0x32, 0x6F, 0x9B, 0x31, 0x68, 0x91, 0x3B, 0xE5,
        0xB9, 0xC8, 0x7A, 0xE0, 0xB0, 0x25, 0xEC, 0x7A,
        0x2F, 0x4A, 0x07, 0x27, 0x50, 0x94, 0x6A, 0xC6,
        0x11, 0x70, 0xA7, 0x82, 0x6D, 0x97, 0x04, 0xC5,
        0xA2, 0x3A, 0x1C, 0x0A, 0x23, 0x25, 0x14, 0x6C,
        0x3B, 0xC1, 0x85, 0x88, 0x26, 0xC6, 0xB3, 0x92,
        0x79, 0xC2, 0xDA, 0x74, 0x38, 0xA3, 0x70, 0xED,
        0x8A, 0x0A, 0xA5, 0x16, 0x9E, 0x3B, 0xEC, 0x29,
        0xED, 0x88, 0x47, 0x87, 0x32, 0x75, 0x8D, 0x45,
        0x41, 0x43, 0xE2, 0x27, 0xF8, 0x59, 0x58, 0x83,
        0x29, 0x78, 0x42, 0xE6, 0xAF, 0x13, 0x3B, 0x17,
        0xE4, 0x81, 0x1B, 0x0F, 0x57, 0x13, 0xAC, 0x73,
        0xB7, 0xE3, 0x47, 0x42, 0x3E, 0xB9, 0x28, 0x22,
        0xD2, 0x30, 0x6F, 0xA1, 0x45, 0x00, 0xA7, 0x20,
        0x7A, 0x06, 0x72, 0x67, 0x20, 0x46, 0x54, 0x4A,
        0xCC, 0x4E, 0xA9, 0xC1, 0x6E, 0xD7, 0x42, 0x1A,
        0x06, 0x9E, 0x0D, 0x73, 0x7A, 0x98, 0x62, 0x85,
        0x19, 0xC6, 0xA2, 0x9A, 0x42, 0x4A, 0x86, 0x8B,
        0x46, 0xD9, 0xA0, 0xCC, 0x7C, 0x6C, 0x9D, 0xDD,
        0x8B, 0x8B, 0xCB, 0xF4, 0x22, 0xC8, 0xF4, 0x8A,
        0x73, 0x14, 0x3D, 0x5A, 0xBB, 0x66, 0xBC, 0x55,
        0x49, 0x94, 0x18, 0x43, 0x08, 0x02, 0xBA, 0xC5,
        0x44, 0x46, 0x3C, 0xC7, 0x31, 0x9D, 0x17, 0x99,
        0x8F, 0x29, 0x41, 0x13, 0x65, 0x76, 0x6D, 0x04,
        0xC8, 0x47, 0xF3, 0x12, 0x9D, 0x90, 0x77, 0xB7,
        0xD8, 0x33, 0x9B, 0xFB, 0x96, 0xA6, 0x73, 0x9C,
        0x3F, 0x6B, 0x74, 0xA8, 0xF0, 0x5F, 0x91, 0x38,
        0xAB, 0x2F, 0xE3, 0x7A, 0xCB, 0x57, 0x63, 0x4D,
        0x18, 0x20, 0xB5, 0x01, 0x76, 0xF5, 0xA0, 0xB6,
        0xBC, 0x29, 0x40, 0xF1, 0xD5, 0x93, 0x8F, 0x19,
        0x36, 0xB5, 0xF9, 0x58, 0x28, 0xB9, 0x2E, 0xB7,
        0x29, 0x73, 0xC1, 0x59, 0x0A, 0xEB, 0x7A, 0x55,
        0x2C, 0xEC, 0xA1, 0x0B, 0x00, 0xC3, 0x03, 0xB7,
        0xC7, 0x5D, 0x40, 0x20, 0x71, 0xA7, 0x9E, 0x2C,
        0x81, 0x0A, 0xF7, 0xC7, 0x45, 0xE3, 0x33, 0x67,
        0x12, 0x49, 0x2A, 0x42, 0x04, 0x3F, 0x29, 0x03,
        0xA3, 0x7C, 0x64, 0x34, 0xCE, 0xE2, 0x0B, 0x1D,
        0x15, 0x9B, 0x05, 0x76, 0x99, 0xFF, 0x9C, 0x1D,
        0x3B, 0xD6, 0x80, 0x29, 0x83, 0x9A, 0x08, 0xF4,
        0x3E, 0x6C, 0x1C, 0x81, 0x99, 0x13, 0x53, 0x2F,
        0x91, 0x1D, 0xD3, 0x70, 0xC7, 0x02, 0x14, 0x88,
        0xE1, 0x1C, 0xB5, 0x04, 0xCB, 0x9C, 0x70, 0x57,
        0x0F, 0xFF, 0x35, 0xB4, 0xB4, 0x60, 0x11, 0x91,
        0xDC, 0x1A, 0xD9, 0xE6, 0xAD, 0xC5, 0xFA, 0x96,
        0x18, 0x79, 0x8D, 0x7C, 0xC8, 0x60, 0xC8, 0x7A,
        0x93, 0x9E, 0x4C, 0xCF, 0x85, 0x33, 0x63, 0x22,
        0x68, 0xCF, 0x1A, 0x51, 0xAF, 0xF0, 0xCB, 0x81,
        0x1C, 0x55, 0x45, 0xCB, 0x16, 0x56, 0xE6, 0x52,
        0x69, 0x47, 0x74, 0x30, 0x69, 0x9C, 0xCD, 0xEA,
        0x38, 0x00, 0x63, 0x0B, 0x78, 0xCD, 0x58, 0x10,
        0x33, 0x4C, 0xCF, 0x02, 0xE0, 0x13, 0xF3, 0xB8,
        0x02, 0x44, 0xE7, 0x0A, 0xCD, 0xB0, 0x60, 0xBB,
        0xE7, 0xA5, 0x53, 0xB0, 0x63, 0x45, 0x6B, 0x2E,
        0xA8, 0x07, 0x47, 0x34, 0x13, 0x16, 0x5C, 0xE5,
        0x7D, 0xD5, 0x63, 0x47, 0x3C, 0xFB, 0xC9, 0x06,
        0x18, 0xAD, 0xE1, 0xF0, 0xB8, 0x88, 0xAA, 0x48,
        0xE7, 0x22, 0xBB, 0x27, 0x51, 0x85, 0x8F, 0xE1,
        0x96, 0x87, 0x44, 0x2A, 0x48, 0xE7, 0xCA, 0x0D,
        0x2A, 0x29, 0xCD, 0x51, 0xBF, 0xD8, 0xF7, 0x8C,
        0x17, 0xB9, 0x66, 0x0B, 0xFB, 0x54, 0xA4, 0x70,
        0xB2, 0xAE, 0x9A, 0x95, 0x5C, 0x6A, 0xB8, 0xD6,
        0xE5, 0xCC, 0x92, 0xAC, 0x8E, 0xD3, 0xC1, 0x85,
        0xDA, 0xA8, 0xBC, 0x29, 0xF0, 0x57, 0x8E, 0xBB,
        0x81, 0x2B, 0x97, 0xC9, 0xE5, 0xA8, 0x48, 0xA6,
        0x38, 0x4D, 0xE4, 0xE7, 0x5A, 0x31, 0x47, 0x0B,
        0x53, 0x06, 0x6A, 0x8D, 0x02, 0x7B, 0xA4, 0x4B,
        0x21, 0x74, 0x9C, 0x04, 0x92, 0x46, 0x5F, 0x90,
        0x72, 0xB2, 0x83, 0x76, 0xC4, 0xE2, 0x90, 0xB3,
        0x0C, 0x18, 0x63, 0xF9, 0xE5, 0xB7, 0x99, 0x96,
        0x08, 0x34, 0x22, 0xBD, 0x8C, 0x27, 0x2C, 0x10,
        0xEC, 0xC6, 0xEB, 0x9A, 0x0A, 0x82, 0x25, 0xB3,
        0x1A, 0xA0, 0xA6, 0x6E, 0x35, 0xB9, 0xC0, 0xB9,
        0xA7, 0x95, 0x82, 0xBA, 0x20, 0xA3, 0xC0, 0x4C,
        0xD2, 0x99, 0x14, 0xF0, 0x83, 0xA0, 0x15, 0x82,
        0x88, 0xBA, 0x4D, 0x6E, 0xB6, 0x2D, 0x87, 0x26,
        0x4B, 0x91, 0x2B, 0xCA, 0x39, 0x73, 0x2F, 0xBD,
        0xE5, 0x36, 0xA3, 0x77, 0xAD, 0x02, 0xB8, 0xC8,
        0x35, 0xD4, 0xA2, 0xF4, 0xE7, 0xB1, 0xCE, 0x11,
        0x5D, 0x0C, 0x86, 0x0B, 0xEA, 0xA7, 0x95, 0x5A,
        0x49, 0xAD, 0x68, 0x95, 0x86, 0xA8, 0x9A, 0x2B,
        0x9F, 0x9B, 0x10, 0xD1, 0x59, 0x5D, 0x2F, 0xC0,
        0x65, 0xAD, 0x01, 0x8A, 0x7D, 0x56, 0xC6, 0x14,
        0x47, 0x1F, 0x8E, 0x94, 0x6F, 0xE8, 0xAB, 0x49,
        0xE8, 0x22, 0x65, 0x91, 0x11, 0x9F, 0xCA, 0xDB,
        0x4F, 0x9A, 0x86, 0x16, 0x31, 0x37, 0x87, 0x36,
        0xB6, 0x68, 0x8B, 0x78, 0x2D, 0x58, 0xE9, 0x7E,
        0x45, 0x72, 0x75, 0x3A, 0x96, 0x64, 0xB6, 0xB8,
        0x53, 0x68, 0x12, 0xB2, 0x59, 0x11, 0xAA, 0x76,
        0xA2, 0x42, 0x37, 0x54, 0x33, 0x19, 0x27, 0x38,
        0xEE, 0xE7, 0x62, 0xF6, 0xB8, 0x43, 0x15, 0xBB,
        0x34, 0x36, 0x23, 0x1E, 0x0A, 0x9B, 0x27, 0x7E,
        0xD2, 0x8A, 0xE0, 0x05, 0x07, 0x28, 0x34, 0x64,
        0x57, 0xE1, 0x34, 0x05, 0x06, 0x2D, 0xB2, 0x80,
        0x4B, 0x8D, 0xA6, 0x0B, 0xB5, 0xC7, 0x93, 0xD4,
        0xCC, 0x0E, 0x10, 0x1C, 0xBA, 0x2D, 0x91, 0x82,
        0xFD, 0x71, 0x24, 0xFF, 0x52, 0xBF, 0x4C, 0xA2,
        0x82, 0x92, 0xAC, 0x26, 0xD6, 0x78, 0x08, 0x89,
        0x53, 0x97, 0x1D, 0xBA, 0x0B, 0x6F, 0xEC, 0x2C,
        0x96, 0x59, 0x35, 0x32, 0x91, 0xC7, 0x0C, 0x5B,
        0x92, 0x45, 0xA0, 0xCA, 0x25, 0x33, 0x04, 0xAF,
        0xD3, 0xC9, 0x51, 0x02, 0xBE, 0xA6, 0x68, 0x75,
        0xC6, 0x20, 0x16, 0x80, 0xB4, 0xBD, 0xA3, 0x86,
        0x87, 0xB6, 0x48, 0xC2, 0x8E, 0xB3, 0x74, 0x78,
        0xE3, 0xBC, 0x00, 0xCA, 0x8A, 0x3C, 0xC2, 0x72,
        0x04, 0x64, 0x2B, 0x42, 0xB6, 0x8F, 0xCB, 0xE7,
        0xB2, 0x1A, 0x36, 0x6D, 0x06, 0x68, 0xA5, 0x02,
        0x9A, 0x7D, 0xEE, 0xF9, 0x4C, 0xDD, 0x6A, 0x95,
        0xD7, 0xEA, 0x89, 0x31, 0x67, 0x3B, 0xF7, 0x11,
        0x2D, 0x40, 0x42, 0x10, 0x7B, 0x1B, 0x8B, 0x97,
        0x00, 0xC9, 0x74, 0xF9, 0xC4, 0xE8, 0x3A, 0x8F,
        0xAC, 0xD8, 0x9B, 0xFE, 0x0C, 0xA3, 0xCC, 0x4C,
        0x2F, 0xCE, 0x80, 0xA0, 0x3D, 0x35, 0x76, 0xC2,
        0x22, 0xA7, 0x92, 0xB7, 0x2B, 0x1F, 0x07, 0x0A,
        0xB7, 0xF6, 0xB6, 0xF2, 0xB5, 0xCA, 0x2A, 0xF5,
        0x05, 0x4A, 0xFA, 0x70, 0xA8, 0x96, 0x99, 0x01,
        0x59, 0xB4, 0x5D, 0x10, 0x03, 0xE2, 0xA0, 0x56,
        0x48, 0x67, 0x5E, 0x59, 0x60, 0x16, 0xF1, 0xB7,
        0x1D, 0xD0, 0xF7, 0xBD, 0xA7, 0xE2, 0x09, 0x7F,
        0xC7, 0x3B, 0x3A, 0x14, 0x3D, 0x12, 0xC7, 0x26,
        0x02, 0x0A, 0xC3, 0x49, 0x58, 0xAD, 0x70, 0x62,
        0xB9, 0x2B, 0x9A, 0xBF, 0x3C, 0xA6, 0xBE, 0x5A,
        0xE2, 0x9F, 0x57, 0x13, 0x5E, 0x62, 0x5A, 0x36,
        0x79, 0x71, 0x83, 0x7E, 0x63, 0x63, 0xD1, 0x53,
        0x20, 0x94, 0xE0, 0x22, 0xA2, 0x34, 0x67, 0xCF,
        0x93, 0x2E, 0x1F, 0x89, 0xB5, 0xB0, 0x80, 0x3C,
        0x1E, 0xC9, 0x9B, 0x58, 0x5A, 0x78, 0xB5, 0x86,
        0x50, 0x96, 0x74, 0x6F, 0x32, 0x25, 0x82, 0x14,
        0xEC, 0xB3, 0x80, 0x65, 0xC9, 0x7F, 0x45, 0x5E,
        0x15, 0x5A, 0xCC, 0x2D, 0xD0, 0x05, 0xA9, 0xC7,
        0x6B, 0xED, 0x59, 0xCD, 0xA7, 0x38, 0x37, 0xD3,
        0x03, 0x50, 0x4E, 0x6C, 0x97, 0x6A, 0x60, 0x6A,
        0x2B, 0xE7, 0xBB, 0xEC, 0x59, 0x48, 0xB9, 0x1A,
        0x34, 0x9E, 0x89, 0x36, 0x68, 0x8C, 0xC0, 0x27,
        0x97, 0x54, 0xB7, 0x43, 0xAB, 0xC5, 0x86, 0x66,
        0xB1, 0x9B, 0x6C, 0x32, 0x60, 0x05, 0x1F, 0x19,
        0x20, 0x6B, 0xB9, 0x62, 0xBB, 0x66, 0x33, 0xEB,
        0x00, 0x48, 0xE3, 0x2B, 0xAA, 0xCC, 0x5B, 0x02,
        0x0D, 0x02, 0xC8, 0x6C, 0xA9, 0x77, 0x0A, 0xD4,
        0x69, 0xDB, 0x54, 0xA1, 0x06, 0xAC, 0x73, 0xA3,
        0x5B, 0x80, 0x57, 0x42, 0x2B, 0x3D, 0xB2, 0x02,
        0xC5, 0xA5, 0xB4, 0xE3, 0xD5, 0x35, 0xF0, 0xFC,
        0x99, 0x32, 0x6C, 0x4B, 0x8B, 0x7B, 0x16, 0xF1,
        0xCB, 0x5A, 0xF9, 0x68, 0x03, 0xFA, 0x8C, 0x19,
        0x5F, 0xC0, 0xBC, 0xED, 0xDA, 0xAF, 0x01, 0x2A,
        0x51, 0x72, 0x8B, 0x76, 0x48, 0x90, 0x82, 0x37,
        0x3C, 0x91, 0xE9, 0x2C, 0x87, 0xAC, 0xCA, 0x79,
        0x51, 0x60, 0x78, 0x2E, 0x3B, 0x0D, 0xD6, 0x43,
        0x54, 0x4B, 0xB9, 0x6A, 0xBC, 0x27, 0x08, 0xD4,
        0x9B, 0x75, 0x9C, 0xF0, 0x57, 0xAA, 0x22, 0x3B,
        0xAF, 0xD9, 0x6A, 0x33, 0x0B, 0xAF, 0x39, 0x81,
        0x0F, 0xE8, 0x67, 0x1B, 0x43, 0x43, 0xC2, 0x97,
        0xDA, 0x1E, 0x19, 0x69, 0xC9, 0x96, 0x21, 0x6A,
        0xB5, 0x10, 0x6D, 0xA6, 0x68, 0x94, 0x1B, 0x16,
        0x0D, 0x44, 0x77, 0x01, 0x71, 0x36, 0xCB, 0xCA,
        0x5B, 0x5A, 0x8D, 0x44, 0xC4, 0xA8, 0xB1, 0xCF,
        0x3E, 0xF7, 0x97, 0x85, 0xE5, 0xAA, 0x25, 0xC3,
        0xA1, 0xAD, 0x6C, 0x24, 0xFD, 0x14, 0x0F, 0x79,
        0x20, 0x7D, 0xE5, 0xA4, 0x99, 0xF8, 0xA1, 0x53,
        0x4F, 0xFA, 0x80, 0x4A, 0xA7, 0xB3, 0x88, 0x9C,
        0xBE, 0x25, 0xC0, 0x41, 0x47, 0x04, 0xAA, 0x57,
        0x89, 0x7F, 0x17, 0x86, 0x23, 0x64, 0xEC, 0xA5,
        0x62, 0x58, 0x00, 0x72, 0x48, 0x81, 0x39, 0x12,
        0xB8, 0x36, 0x49, 0x7F, 0x03, 0x59, 0xC2, 0xF7,
        0x23, 0x8A, 0x05, 0xD3, 0x05, 0xA0, 0xEA, 0x15,
        0x2E, 0x72, 0xB4, 0x44, 0x17, 0xA8, 0x68, 0x13,
        0x4E, 0x91, 0xB3, 0xCA, 0x79, 0x31, 0x23, 0x2F,
        0xD4, 0xC2, 0x5F, 0x8C, 0x2A, 0x49, 0x2A, 0x33,
        0x9C, 0xDC, 0x0A, 0x13, 0x89, 0x67, 0x21, 0x14,
        0x51, 0xF2, 0x56, 0x26, 0x78, 0xFA, 0x14, 0x08,
        0x0A, 0x34, 0x43, 0x6C, 0x42, 0xB0, 0x78, 0x65,
        0xAC, 0x03, 0x6A, 0x81, 0xE9, 0x7A, 0x77, 0x87,
        0xA9, 0x38, 0x02, 0x5C, 0xAF, 0x81, 0x34, 0x50,
        0x36, 0x8B, 0xED, 0x0C, 0x94, 0xB1, 0x85, 0x76,
        0x04, 0x52, 0x64, 0x05, 0xD2, 0x7A, 0x1C, 0x1A,
        0xBC, 0x81, 0xB5, 0xB6, 0xEC, 0x13, 0xC7, 0x19,
        0x30, 0xA9, 0x7D, 0x92, 0x32, 0xCF, 0x70, 0x21,
        0xEF, 0x87, 0xA4, 0xD1, 0x55, 0x32, 0x8E, 0x62,
        0xB5, 0x83, 0xA8, 0x3B, 0x4A, 0xF2, 0x1F, 0x9F,
        0x57, 0x50, 0xF8, 0x57, 0x51, 0x50, 0x42, 0x4F,
        0x63, 0xB8, 0x99, 0xD7, 0x1C, 0xAD, 0x26, 0x7C,
        0x09, 0xE4, 0x46, 0x71, 0x46, 0xE1, 0x6E, 0x9B,
        0x6C, 0x65, 0x3F, 0x00, 0x8C, 0x31, 0x13, 0x75,
        0xE2, 0xE0, 0x06, 0xD4, 0x07, 0x6A, 0x54, 0x6B,
        0x82, 0xF5, 0x31, 0x42, 0x22, 0xF7, 0xC6, 0x54,
        0x31, 0x7E, 0x79, 0xEC, 0x60, 0x35, 0xB7, 0x3F,
        0xAF, 0x49, 0x17, 0x57, 0xE6, 0x1C, 0x82, 0x83,
        0x26, 0xD5, 0x30, 0x44, 0x54, 0x1C, 0x4D, 0x45,
        0x37, 0xAB, 0xD3, 0xEA, 0x1E, 0x67, 0x99, 0x8C,
        0x33, 0x82, 0x97, 0x4C, 0xA7, 0x8A, 0xE1, 0xB1,
        0x96, 0x0E, 0x4A, 0x92, 0x26, 0xB0, 0x21, 0x9A,
        0xB0, 0x70, 0xF0, 0xD7, 0xAA, 0x66, 0xD7, 0x6F,
        0x93, 0x16, 0xAD, 0xB8, 0x0C, 0x54, 0xD6, 0x49,
        0x97, 0x71, 0xB4, 0x71, 0xE8, 0x16, 0x8D, 0x47,
        0xBC, 0xAA, 0x08, 0x32, 0x4A, 0xB6, 0xBA, 0x92,
        0xC3, 0xA7, 0x02, 0x75, 0xF2, 0x4F, 0xA4, 0xDC,
        0x10, 0xE2, 0x51, 0x63, 0x3F, 0xB9, 0x8D, 0x16,
        0x2B, 0xB5, 0x53, 0x72, 0x02, 0xC6, 0xA5, 0x53,
        0xCE, 0x78, 0x41, 0xC4, 0xD4, 0x0B, 0x87, 0x3B,
        0x85, 0xCA, 0x03, 0xA0, 0xA1, 0xE1, 0xCF, 0xAD,
        0xE6, 0xBA, 0x51, 0x80, 0xAB, 0x13, 0x23, 0xCC,
        0xBA, 0x9A, 0x3E, 0x9C, 0x53, 0xD3, 0x75, 0x75,
        0xAB, 0x1F, 0xD9, 0xE7, 0x31, 0x6C, 0x6F, 0xEE,
        0xCB, 0x0A, 0x14, 0xDF, 0x6F, 0x2D, 0xA5, 0x6C,
        0x2F, 0x56, 0xF5, 0x5A, 0x89, 0x63, 0x5C, 0xFC,
        0xFD, 0xA4, 0x79, 0x27, 0xAF, 0x1F, 0x0A, 0x47,
        0xB2, 0xD4, 0xE4, 0xE6, 0x16, 0x34, 0xB1, 0xB5,
        0x1D, 0x37, 0xA3, 0xA3, 0x07, 0xA9, 0x72, 0x42,
        0x0D, 0xE1, 0xB7, 0xA4, 0x81, 0xB8, 0x3E, 0x58,
        0x3B, 0x6A, 0xF1, 0x6F, 0x63, 0xCB, 0x00, 0xC6
    };
    static const byte c_1024[WC_ML_KEM_1024_CIPHER_TEXT_SIZE] = {
        0x0C, 0x68, 0x1B, 0x4A, 0xA8, 0x1F, 0x26, 0xAD,
        0xFB, 0x64, 0x5E, 0xC2, 0x4B, 0x37, 0x52, 0xF6,
        0xB3, 0x2C, 0x68, 0x64, 0x5A, 0xA5, 0xE7, 0xA9,
        0x99, 0xB6, 0x20, 0x36, 0xA5, 0x3D, 0xC5, 0xCB,
        0x06, 0x0A, 0x47, 0x3C, 0x08, 0xE5, 0xDA, 0x5C,
        0x0F, 0x5A, 0xF0, 0xE5, 0x17, 0x0C, 0x65, 0x97,
        0xE5, 0x0E, 0xC0, 0x80, 0x60, 0xF9, 0x9B, 0x0C,
        0x00, 0xEE, 0x9B, 0xDD, 0xAD, 0x7E, 0x7D, 0x25,
        0xA2, 0x2B, 0x22, 0x6F, 0x90, 0x14, 0x9B, 0x4C,
        0xE8, 0x87, 0xC7, 0x2F, 0xB6, 0x0A, 0xFF, 0x21,
        0x44, 0xEA, 0x2A, 0x72, 0x38, 0x3B, 0x31, 0x18,
        0xF9, 0x22, 0xD0, 0x32, 0xA1, 0x6F, 0x55, 0x42,
        0x89, 0x90, 0x2A, 0x14, 0xCF, 0x77, 0x55, 0x51,
        0x2B, 0xB1, 0x18, 0x6B, 0xAF, 0xAF, 0xFE, 0x79,
        0x4D, 0x2B, 0x6C, 0xDE, 0x90, 0x10, 0x9E, 0x65,
        0x82, 0xD3, 0x9C, 0xE0, 0xC9, 0x61, 0x97, 0x48,
        0x4B, 0x3F, 0xA0, 0x7F, 0xC9, 0x1D, 0x39, 0x4F,
        0xC8, 0xD8, 0x8E, 0x7F, 0xC4, 0xBE, 0x00, 0x2E,
        0x2D, 0xB5, 0x6F, 0x0C, 0x4D, 0x9D, 0x3F, 0xBD,
        0xA2, 0x74, 0x53, 0x6A, 0x0B, 0x86, 0xAB, 0xC6,
        0xE3, 0x9B, 0xDA, 0x52, 0x93, 0x1A, 0xEB, 0xB8,
        0xF1, 0x08, 0x4C, 0x5C, 0x1F, 0x7C, 0xB3, 0x17,
        0x77, 0x88, 0xB7, 0xF3, 0x31, 0xB7, 0x07, 0x43,
        0x61, 0x16, 0x34, 0x91, 0xD4, 0x28, 0xE7, 0x8B,
        0xCB, 0xB5, 0x7B, 0x63, 0x08, 0x41, 0xAA, 0x98,
        0x73, 0x33, 0x37, 0x7C, 0xF0, 0x95, 0x69, 0xCF,
        0xD1, 0x4C, 0xC2, 0xA1, 0x1C, 0x50, 0x1B, 0xDF,
        0x82, 0xC9, 0x3D, 0xE0, 0x5B, 0xEA, 0x20, 0x06,
        0x0D, 0xE8, 0x9C, 0x68, 0x6B, 0x82, 0x45, 0x71,
        0xCE, 0xF9, 0x4A, 0xB3, 0xFD, 0xAF, 0xA8, 0x51,
        0x26, 0x19, 0x81, 0x36, 0x69, 0xD4, 0xF5, 0x36,
        0x37, 0xFE, 0xFA, 0x4D, 0x02, 0x8C, 0xB2, 0x33,
        0xE5, 0x69, 0x30, 0xE2, 0x23, 0x5F, 0x7E, 0x60,
        0x34, 0xCA, 0x94, 0xB1, 0x43, 0xB7, 0x7A, 0xD4,
        0xA6, 0x87, 0x56, 0xE8, 0xA9, 0x18, 0x4D, 0xBA,
        0x61, 0xA8, 0x9F, 0x91, 0xED, 0xFB, 0x51, 0xA3,
        0x92, 0x11, 0x40, 0x24, 0x73, 0xA5, 0xF8, 0x91,
        0x45, 0x73, 0x6B, 0x2B, 0xF8, 0x56, 0x9C, 0x70,
        0x5B, 0x0C, 0xDB, 0x89, 0x80, 0xA4, 0x47, 0xE4,
        0xE1, 0xEA, 0xAD, 0x3E, 0x7E, 0x05, 0x78, 0xF5,
        0xF8, 0x6B, 0x8D, 0x03, 0xC9, 0xDA, 0xFE, 0x87,
        0x5E, 0x33, 0x9B, 0x44, 0x23, 0x84, 0x56, 0x16,
        0x79, 0x9E, 0xDC, 0xE0, 0x5F, 0x31, 0xB9, 0x26,
        0x64, 0xC5, 0xA5, 0x92, 0x53, 0xA6, 0x0E, 0x9D,
        0x89, 0x54, 0x8A, 0x30, 0x0C, 0x1A, 0xDB, 0x6D,
        0x19, 0x0A, 0x77, 0x5C, 0x5E, 0xE6, 0xE8, 0xA8,
        0x9B, 0x6E, 0x77, 0x9B, 0x03, 0x4C, 0x34, 0x00,
        0xA6, 0x25, 0xF4, 0xBB, 0xED, 0xBF, 0x91, 0x9C,
        0x45, 0xB2, 0xBC, 0xD1, 0x4C, 0x66, 0x92, 0x48,
        0xFC, 0x43, 0xC3, 0xEF, 0x47, 0xE1, 0x00, 0x75,
        0x89, 0x42, 0xE7, 0x5E, 0x8E, 0xD6, 0x07, 0x5A,
        0x96, 0xD7, 0x0D, 0x4E, 0xBD, 0x2B, 0x61, 0x35,
        0x82, 0x24, 0xDD, 0xA1, 0xEC, 0x4C, 0x19, 0xC2,
        0xA9, 0x28, 0x98, 0x17, 0x6F, 0xEB, 0x3C, 0x02,
        0xED, 0xCB, 0x99, 0x08, 0xBA, 0xE4, 0x9B, 0xD9,
        0x4A, 0xF0, 0x28, 0xED, 0xF8, 0xCF, 0xC2, 0xE5,
        0xF2, 0xE0, 0xBD, 0x37, 0x50, 0x06, 0x98, 0x6A,
        0xD4, 0x9E, 0x71, 0x75, 0x48, 0xE7, 0x46, 0xFE,
        0xF4, 0x9C, 0x86, 0x8B, 0xCE, 0xA2, 0x79, 0x0A,
        0xA9, 0x7E, 0x04, 0x06, 0x1B, 0x75, 0x60, 0x5C,
        0xB3, 0x9E, 0xFD, 0x46, 0x3D, 0x7B, 0x3D, 0x68,
        0xBA, 0x57, 0x44, 0x34, 0xFF, 0x7B, 0xE8, 0xE2,
        0xB8, 0x4B, 0xFC, 0x47, 0xE6, 0x7E, 0x9C, 0xD1,
        0x5F, 0x3E, 0xD4, 0x50, 0xC6, 0x1A, 0xFB, 0xA7,
        0x9A, 0x20, 0xB0, 0xB6, 0xF2, 0x87, 0x77, 0x7C,
        0x72, 0xF4, 0xAD, 0x24, 0x81, 0x74, 0xF1, 0x95,
        0x94, 0x77, 0xAA, 0x7A, 0x7C, 0x97, 0xF1, 0x22,
        0xC5, 0x04, 0x47, 0xC7, 0x48, 0x4F, 0x38, 0x2B,
        0xC4, 0x7D, 0x81, 0xFC, 0xC9, 0xC7, 0xE8, 0x92,
        0xC8, 0x83, 0x9D, 0x37, 0xB3, 0x53, 0x94, 0xB5,
        0x3E, 0x6B, 0x2B, 0x18, 0x95, 0xAB, 0xB0, 0xDE,
        0x8C, 0x98, 0xF2, 0x63, 0x3D, 0xC4, 0x41, 0x3A,
        0x8D, 0x57, 0x35, 0xDF, 0xC9, 0xA6, 0x40, 0x26,
        0xB6, 0xF3, 0x47, 0x79, 0xD6, 0xAC, 0x8A, 0xD9,
        0x9C, 0xC3, 0x1A, 0xA8, 0x98, 0xC2, 0xE7, 0x05,
        0x7F, 0x3D, 0xB8, 0xA1, 0xA8, 0xA9, 0x85, 0x27,
        0xA7, 0x9E, 0x43, 0x55, 0x2F, 0x28, 0xD1, 0x02,
        0x3E, 0x1F, 0x6A, 0x6B, 0x84, 0x85, 0x5C, 0xF5,
        0xE6, 0xDF, 0x88, 0x9B, 0xA2, 0x69, 0xF0, 0x48,
        0x94, 0x6E, 0x84, 0x02, 0x1C, 0x65, 0xC5, 0xA9,
        0x3B, 0x00, 0x7B, 0x07, 0x74, 0x1C, 0x1E, 0xE1,
        0x76, 0xC7, 0x39, 0x49, 0x11, 0x0F, 0x54, 0x8E,
        0xF4, 0x33, 0x2D, 0xCD, 0xD4, 0x91, 0xD2, 0xCE,
        0xFD, 0x02, 0x48, 0x88, 0x3F, 0x5E, 0x95, 0x25,
        0xBC, 0x91, 0xF3, 0x0A, 0xF1, 0x7C, 0xF5, 0xA9,
        0x8D, 0xD4, 0x4E, 0xF9, 0xA7, 0x1F, 0x99, 0xBB,
        0x73, 0x29, 0x85, 0xBA, 0x10, 0xA7, 0x23, 0xEF,
        0x47, 0x6F, 0xCF, 0x96, 0x6D, 0xA9, 0x45, 0x6B,
        0x24, 0x97, 0x8E, 0x33, 0x05, 0x0D, 0x0E, 0xC9,
        0x0D, 0x3C, 0xE4, 0x63, 0x78, 0x85, 0x1C, 0x9E,
        0xCF, 0xCF, 0xD3, 0x6C, 0x89, 0x5D, 0x44, 0xE9,
        0xE5, 0x06, 0x99, 0x30, 0x82, 0x52, 0x3D, 0x26,
        0x18, 0x57, 0x66, 0xB2, 0x35, 0x68, 0xCB, 0x95,
        0xE6, 0x41, 0x08, 0xF8, 0x9D, 0x10, 0x14, 0x74,
        0x7C, 0x67, 0xB6, 0xF3, 0xC8, 0x76, 0x7B, 0xE5,
        0xFC, 0x34, 0x12, 0x27, 0xDE, 0x94, 0x88, 0x86,
        0x1C, 0x5F, 0xE8, 0x11, 0x40, 0x9F, 0x80, 0x95,
        0x7D, 0x07, 0x52, 0x2A, 0x72, 0xCF, 0x6A, 0xB0,
        0x37, 0x8D, 0x0F, 0x2F, 0x28, 0xAF, 0x54, 0x81,
        0x85, 0xC3, 0x93, 0x67, 0x77, 0x99, 0x44, 0x66,
        0xA0, 0x19, 0xD3, 0x3B, 0x18, 0xA5, 0x4F, 0x38,
        0x0A, 0x33, 0x89, 0x2A, 0xB4, 0xD4, 0xBD, 0x50,
        0x7B, 0x5A, 0x61, 0xD0, 0xD3, 0x58, 0x34, 0x1A,
        0xC9, 0x2F, 0x07, 0xB4, 0x3B, 0x8F, 0x6A, 0xFC,
        0x69, 0x91, 0xBB, 0x6A, 0x1E, 0xAC, 0x23, 0xCA,
        0x6F, 0x73, 0xE9, 0x1F, 0x24, 0x64, 0xBD, 0x11,
        0x90, 0x98, 0xD7, 0xE7, 0x68, 0xE7, 0x7E, 0xCE,
        0x53, 0xFB, 0x89, 0x9B, 0xEB, 0x42, 0x26, 0x5E,
        0xCF, 0x7B, 0x27, 0x1F, 0x66, 0x54, 0x62, 0x82,
        0xD4, 0x72, 0xC3, 0x62, 0x39, 0x00, 0x6B, 0xB0,
        0xAB, 0xAB, 0xCC, 0xA2, 0x45, 0x50, 0xBA, 0xA0,
        0xA6, 0x01, 0x34, 0x8C, 0x81, 0x0F, 0xF5, 0xF9,
        0xEE, 0x50, 0x4B, 0xF7, 0x15, 0x5D, 0xEE, 0x41,
        0x41, 0xA1, 0x16, 0x05, 0xA4, 0xF3, 0x50, 0x9A,
        0xC9, 0xCA, 0xEF, 0x66, 0x24, 0xD2, 0x1D, 0xE3,
        0x32, 0xD5, 0xD5, 0x08, 0x28, 0xB5, 0x2E, 0x92,
        0x88, 0x5D, 0x3B, 0x90, 0x55, 0x3B, 0x14, 0x46,
        0x3A, 0xFB, 0x1E, 0xDC, 0xCD, 0x3B, 0x56, 0x9B,
        0x5A, 0x7F, 0x00, 0xBB, 0x66, 0x76, 0x9D, 0xAD,
        0xAC, 0x23, 0xAD, 0x8B, 0xB5, 0xD7, 0x3A, 0x6F,
        0x39, 0x0E, 0x6F, 0xC2, 0xF6, 0xF8, 0xEE, 0x3C,
        0xF4, 0x00, 0x9A, 0x5C, 0x3E, 0x1E, 0xF6, 0x0E,
        0x8F, 0x04, 0x06, 0x72, 0xD2, 0x62, 0xE6, 0x49,
        0x03, 0x79, 0xBB, 0xC7, 0x04, 0x95, 0xDF, 0xF2,
        0x37, 0xBE, 0xCD, 0x99, 0x52, 0xCD, 0x7E, 0xDE,
        0xB6, 0xD1, 0xDF, 0xC3, 0x60, 0xB3, 0xFC, 0x8B,
        0x0A, 0xF4, 0x80, 0xFF, 0xE0, 0x24, 0xAE, 0xEF,
        0xCD, 0x4E, 0x9C, 0xE9, 0x5D, 0x9B, 0x46, 0x9C,
        0x9A, 0x70, 0xE5, 0x11, 0x0D, 0xA0, 0xBA, 0xC1,
        0x24, 0xFC, 0x37, 0x41, 0xDC, 0xF4, 0x91, 0x16,
        0x26, 0x17, 0x96, 0x50, 0x4D, 0x5F, 0x49, 0x0B,
        0x43, 0x3C, 0x33, 0xC4, 0x0E, 0xDC, 0xE2, 0xB7,
        0x51, 0x51, 0xDA, 0x25, 0x6A, 0x86, 0x8A, 0x5E,
        0x35, 0xF8, 0x62, 0x26, 0xB8, 0x15, 0x1C, 0x91,
        0x93, 0x4C, 0xCC, 0x3D, 0xAC, 0xA3, 0x91, 0xDE,
        0xCC, 0xA7, 0x45, 0x37, 0x56, 0x60, 0xB6, 0xEC,
        0x41, 0xAE, 0x5D, 0x81, 0x08, 0x38, 0xCB, 0xEE,
        0xFF, 0xA1, 0x25, 0x57, 0x88, 0x44, 0x12, 0x35,
        0x7B, 0x10, 0x08, 0x36, 0x3D, 0x32, 0xB2, 0x37,
        0xAA, 0x1D, 0xD8, 0xE2, 0xD9, 0xC6, 0x36, 0x7A,
        0xDA, 0x09, 0xB2, 0xC9, 0x50, 0x60, 0x20, 0x6C,
        0xEC, 0x3E, 0xED, 0x39, 0x1F, 0xDC, 0x5D, 0xBE,
        0xF6, 0xF0, 0x8B, 0xDF, 0x04, 0x08, 0xE5, 0x85,
        0xAE, 0x5E, 0xBC, 0x8E, 0x97, 0x45, 0xD4, 0x4F,
        0xEC, 0xA9, 0x75, 0xAB, 0xBC, 0x14, 0x0B, 0xB3,
        0x7B, 0x8A, 0xDD, 0x16, 0xFC, 0xC2, 0x95, 0x69,
        0x10, 0xDC, 0x72, 0xBB, 0x3F, 0x02, 0xE9, 0xA1,
        0x30, 0xC9, 0xA8, 0x4F, 0x9C, 0xCB, 0x74, 0xD1,
        0x34, 0xCD, 0xF4, 0x0A, 0xFC, 0xBA, 0x20, 0x09,
        0xC8, 0xF0, 0x04, 0x02, 0x39, 0xBC, 0x99, 0x22,
        0x0E, 0xF6, 0x4C, 0x4D, 0xCC, 0xDE, 0x2E, 0x2E,
        0x5C, 0x9B, 0x68, 0x60, 0x2F, 0xBE, 0x8E, 0xF4,
        0xC9, 0x8B, 0x34, 0x68, 0xC7, 0x9D, 0xF4, 0xE0,
        0x78, 0x51, 0x1B, 0xFB, 0x8A, 0xA3, 0xDA, 0x09,
        0x59, 0x7A, 0x02, 0x51, 0x1E, 0x7C, 0x21, 0xA7,
        0xCF, 0x66, 0xA9, 0x38, 0x43, 0xA9, 0x48, 0x68,
        0xF1, 0x9E, 0x85, 0x52, 0x55, 0x2E, 0x3A, 0xCD,
        0xF6, 0xCB, 0x81, 0x06, 0x34, 0xDB, 0x97, 0xCB,
        0xC4, 0xBB, 0x56, 0x97, 0x09, 0xDA, 0xD4, 0x84,
        0x56, 0x45, 0x44, 0x6F, 0xA8, 0xD2, 0x89, 0xFC,
        0x59, 0x30, 0x7B, 0x80, 0x1E, 0x60, 0xCE, 0x2A,
        0x91, 0xE0, 0x6E, 0x9C, 0x22, 0xC1, 0x6E, 0x2E,
        0x59, 0xBD, 0xE3, 0x8A, 0x41, 0x6B, 0xB1, 0xB4,
        0xAC, 0x54, 0x57, 0x43, 0x8F, 0xDC, 0x5D, 0x64,
        0x45, 0x0A, 0x89, 0xEC, 0xB8, 0x32, 0xC1, 0xBB,
        0x27, 0x9D, 0xBF, 0x59, 0x33, 0x46, 0x81, 0x77,
        0x6A, 0xC0, 0x04, 0x09, 0x84, 0x6D, 0x09, 0xD6,
        0xF6, 0x87, 0x77, 0x2E, 0x34, 0x08, 0x50, 0xAB,
        0x86, 0x73, 0x38, 0x42, 0x15, 0xE1, 0x2C, 0x8D,
        0x0F, 0x53, 0x1C, 0x45, 0x1E, 0x58, 0x49, 0x3E,
        0x0E, 0xE4, 0x15, 0xAD, 0x59, 0x4D, 0xF3, 0x8C,
        0x34, 0x40, 0x8C, 0x7E, 0xD9, 0xF0, 0xC3, 0x92,
        0xF1, 0x53, 0x46, 0x04, 0xEA, 0xC3, 0xD9, 0xC1,
        0x54, 0x65, 0xA9, 0xA4, 0x66, 0x32, 0x21, 0x4B,
        0x53, 0x69, 0x90, 0xD7, 0x80, 0x78, 0xE5, 0xBD,
        0x7E, 0xAE, 0x20, 0x13, 0xFF, 0xF8, 0xFD, 0xD8,
        0xB2, 0x75, 0xC8, 0x9D, 0x97, 0xC9, 0x35, 0x3D,
        0xF3, 0xC4, 0x2A, 0x28, 0xE8, 0x14, 0xD8, 0x46,
        0x8E, 0x2B, 0x48, 0xDB, 0x09, 0x76, 0xD8, 0x8F,
        0x5E, 0xEC, 0xEF, 0xEA, 0xFB, 0x8F, 0x7F, 0x4A,
        0xF2, 0x91, 0xA7, 0x28, 0xF6, 0x24, 0x9E, 0xCF,
        0x56, 0x22, 0x33, 0x92, 0x69, 0xAA, 0x94, 0x53,
        0x29, 0xE9, 0x19, 0xF8, 0xB4, 0x41, 0xC8, 0x3D,
        0x55, 0x07, 0xF3, 0x0D, 0xF0, 0xFD, 0x2B, 0x13,
        0xFF, 0x80, 0x6F, 0x52, 0x2D, 0xAA, 0x11, 0xAF,
        0x67, 0x6A, 0x51, 0x3C, 0x14, 0x9C, 0x70, 0xF0,
        0xD6, 0xE9, 0x9A, 0x88, 0x04, 0x50, 0xA5, 0x4E,
        0x04, 0x17, 0xFE, 0x3C, 0x1E, 0x51, 0x3E, 0x9D,
        0x92, 0x0E, 0x30, 0xA8, 0xB4, 0x28, 0x91, 0x26,
        0x7A, 0x2D, 0xC5, 0x0A, 0xD8, 0x1F, 0x98, 0x04,
        0x49, 0x20, 0xC0, 0x99, 0xDF, 0x22, 0xC7, 0x39,
        0x98, 0xA2, 0x5C, 0x58, 0x1A, 0x51, 0x78, 0xC7,
        0x2B, 0x17, 0xAC, 0x87, 0x5B, 0xC6, 0x85, 0x48,
        0xA0, 0xFB, 0x0C, 0xBE, 0xE3, 0x8F, 0x05, 0x01,
        0x7B, 0x12, 0x43, 0x33, 0x43, 0xA6, 0x58, 0xF1,
        0x98, 0x0C, 0x81, 0x24, 0xEA, 0x6D, 0xD8, 0x1F
    };
    static const byte kprime_1024[WC_ML_KEM_SS_SZ] = {
        0x8F, 0x33, 0x6E, 0x9C, 0x28, 0xDF, 0x34, 0x9E,
        0x03, 0x22, 0x0A, 0xF0, 0x1C, 0x42, 0x83, 0x2F,
        0xEF, 0xAB, 0x1F, 0x2A, 0x74, 0xC1, 0x6F, 0xAF,
        0x6F, 0x64, 0xAD, 0x07, 0x1C, 0x1A, 0x33, 0x94
    };
#endif
    static byte ss[WC_ML_KEM_SS_SZ];

    key = (MlKemKey*)XMALLOC(sizeof(MlKemKey), NULL, DYNAMIC_TYPE_TMP_BUFFER);
    ExpectNotNull(key);
    if (key != NULL) {
        XMEMSET(key, 0, sizeof(MlKemKey));
    }

#ifndef WOLFSSL_NO_ML_KEM_512
    ExpectIntEQ(wc_MlKemKey_Init(key, WC_ML_KEM_512, NULL, INVALID_DEVID), 0);
    ExpectIntEQ(wc_MlKemKey_DecodePrivateKey(key, dk_512, sizeof(dk_512)), 0);
    ExpectIntEQ(wc_MlKemKey_Decapsulate(key, ss, c_512, sizeof(c_512)), 0);
    ExpectIntEQ(XMEMCMP(ss, kprime_512, WC_ML_KEM_SS_SZ), 0);
    wc_MlKemKey_Free(key);
#endif
#ifndef WOLFSSL_NO_ML_KEM_768
    ExpectIntEQ(wc_MlKemKey_Init(key, WC_ML_KEM_768, NULL, INVALID_DEVID), 0);
    ExpectIntEQ(wc_MlKemKey_DecodePrivateKey(key, dk_768, sizeof(dk_768)), 0);
    ExpectIntEQ(wc_MlKemKey_Decapsulate(key, ss, c_768, sizeof(c_768)), 0);
    ExpectIntEQ(XMEMCMP(ss, kprime_768, WC_ML_KEM_SS_SZ), 0);
    wc_MlKemKey_Free(key);
#endif
#ifndef WOLFSSL_NO_ML_KEM_1024
    ExpectIntEQ(wc_MlKemKey_Init(key, WC_ML_KEM_1024, NULL, INVALID_DEVID), 0);
    ExpectIntEQ(wc_MlKemKey_DecodePrivateKey(key, dk_1024, sizeof(dk_1024)), 0);
    ExpectIntEQ(wc_MlKemKey_Decapsulate(key, ss, c_1024, sizeof(c_1024)), 0);
    ExpectIntEQ(XMEMCMP(ss, kprime_1024, WC_ML_KEM_SS_SZ), 0);
    wc_MlKemKey_Free(key);
#endif

    XFREE(key, NULL, DYNAMIC_TYPE_TMP_BUFFER);
#endif
    return EXPECT_RESULT();
}

