#!/usr/bin/env perl

# gencertbuf.pl
# version 1.1
# Updated 07/01/2014
#
# Copyright (C) 2006-2015 wolfSSL Inc.
#

use strict;
use warnings;

# ---- SCRIPT SETTINGS -------------------------------------------------------

# output C header file to write cert/key buffers to
my $outputFile = "./wolfssl/certs_test.h";

# ecc keys and certs to be converted
# Used with HAVE_ECC && USE_CERT_BUFFERS_256

my @fileList_ecc = (
        [ "./certs/ecc-client-key.der",              "ecc_clikey_der_256" ],
        [ "./certs/ecc-client-keyPub.der",           "ecc_clikeypub_der_256" ],
        [ "./certs/client-ecc-cert.der",             "cliecc_cert_der_256" ],
        [ "./certs/ecc-key.der",                     "ecc_key_der_256" ],
        [ "./certs/ecc-keyPub.der",                  "ecc_key_pub_der_256" ],
        [ "./certs/statickeys/ecc-secp256r1.der",    "ecc_secp_r1_statickey_der_256" ],
        [ "./certs/server-ecc-comp.der",             "serv_ecc_comp_der_256" ],
        [ "./certs/server-ecc-rsa.der",              "serv_ecc_rsa_der_256" ],
        [ "./certs/server-ecc.der",                  "serv_ecc_der_256" ],
        [ "./certs/ca-ecc-key.der",                  "ca_ecc_key_der_256" ],
        [ "./certs/ca-ecc-cert.der",                 "ca_ecc_cert_der_256" ],
        [ "./certs/ca-ecc384-key.der",               "ca_ecc_key_der_384" ],
        [ "./certs/ca-ecc384-cert.der",              "ca_ecc_cert_der_384" ]
        );


# ed25519 keys and certs
# Used with HAVE_ED25519 define.
my @fileList_ed = (
        [ "./certs/ed25519/server-ed25519.der",     "server_ed25519_cert" ],
        [ "./certs/ed25519/server-ed25519-key.der", "server_ed25519_key" ],
        [ "./certs/ed25519/ca-ed25519.der",         "ca_ed25519_cert" ],
        [ "./certs/ed25519/client-ed25519.der",     "client_ed25519_cert" ],
        [ "./certs/ed25519/client-ed25519-key.der", "client_ed25519_key" ]
        );

# x25519 keys and certs
# Used with USE_CERT_BUFFERS_25519 define.
my @fileList_x = (
        [ "./certs/statickeys/x25519.der",      "x25519_statickey_der" ],
        [ "./certs/statickeys/x25519-pub.der",  "x25519_pub_statickey_der" ]
        );


# 1024-bit certs/keys to be converted
# Used with USE_CERT_BUFFERS_1024 define.

my @fileList_1024 = (
        [ "./certs/1024/client-key.der", "client_key_der_1024" ],
        [ "./certs/1024/client-keyPub.der", "client_keypub_der_1024" ],
        [ "./certs/1024/client-cert.der", "client_cert_der_1024" ],
        [ "./certs/1024/dh1024.der", "dh_key_der_1024" ],
        [ "./certs/1024/dsa1024.der", "dsa_key_der_1024" ],
        [ "./certs/1024/rsa1024.der", "rsa_key_der_1024" ],
        [ "./certs/1024/ca-key.der", "ca_key_der_1024"],
        [ "./certs/1024/ca-cert.der", "ca_cert_der_1024" ],
        [ "./certs/1024/server-key.der", "server_key_der_1024" ],
        [ "./certs/1024/server-cert.der", "server_cert_der_1024" ]
        );

# 2048-bit certs/keys to be converted
# Used with USE_CERT_BUFFERS_2048 define.
my @fileList_2048 = (
        [ "./certs/client-key.der", "client_key_der_2048" ],
        [ "./certs/client-keyPub.der", "client_keypub_der_2048" ],
        [ "./certs/client-cert.der", "client_cert_der_2048" ],
        [ "./certs/dh2048.der", "dh_key_der_2048" ],
        [ "./certs/dh-pubkey-2048.der", "dh_pub_key_der_2048" ],
        [ "./certs/statickeys/dh-ffdhe2048.der", "dh_ffdhe_statickey_der_2048" ],
        [ "./certs/statickeys/dh-ffdhe2048-pub.der", "dh_ffdhe_pub_statickey_der_2048" ],
        [ "./certs/dsa-pubkey-2048.der", "dsa_pub_key_der_2048" ],
        [ "./certs/dsa2048.der", "dsa_key_der_2048" ],
        [ "./certs/rsa2048.der", "rsa_key_der_2048" ],
        [ "./certs/ca-key.der", "ca_key_der_2048" ],
        [ "./certs/ca-cert.der", "ca_cert_der_2048" ],
        [ "./certs/ca-cert-chain.der", "ca_cert_chain_der" ],
        [ "./certs/server-key.der", "server_key_der_2048" ],
        [ "./certs/server-cert.der", "server_cert_der_2048" ]
        );

# 3072-bit certs/keys to be converted
# Used with USE_CERT_BUFFERS_3072 define.
my @fileList_3072 = (
        [ "./certs/dh3072.der", "dh_key_der_3072" ],
        [ "./certs/dsa3072.der", "dsa_key_der_3072" ],
        [ "./certs/rsa3072.der", "rsa_key_der_3072" ],
        [ "./certs/3072/client-key.der", "client_key_der_3072" ],
        [ "./certs/3072/client-keyPub.der", "client_keypub_der_3072" ],
        [ "./certs/3072/client-cert.der", "client_cert_der_3072" ],
        );

# 4096-bit certs/keys to be converted
# Used with USE_CERT_BUFFERS_4096 define.
my @fileList_4096 = (
        [ "./certs/4096/client-key.der", "client_key_der_4096" ],
        [ "./certs/4096/client-keyPub.der", "client_keypub_der_4096" ],
        [ "./certs/4096/client-cert.der", "client_cert_der_4096" ],
        [ "./certs/dh4096.der", "dh_key_der_4096" ],
        );

#Falcon Post-Quantum Keys
#Used with HAVE_PQC
my @fileList_falcon = (
        ["certs/falcon/bench_falcon_level1_key.der", "bench_falcon_level1_key" ],
        ["certs/falcon/bench_falcon_level5_key.der", "bench_falcon_level5_key" ],
        );

#Sphincs+ Post-Quantum Keys
#Used with HAVE_PQC
my @fileList_sphincs = (
        ["certs/sphincs/bench_sphincs_fast_level1_key.der", "bench_sphincs_fast_level1_key" ],
        ["certs/sphincs/bench_sphincs_fast_level3_key.der", "bench_sphincs_fast_level3_key" ],
        ["certs/sphincs/bench_sphincs_fast_level5_key.der", "bench_sphincs_fast_level5_key" ],
        ["certs/sphincs/bench_sphincs_small_level1_key.der", "bench_sphincs_small_level1_key" ],
        ["certs/sphincs/bench_sphincs_small_level3_key.der", "bench_sphincs_small_level3_key" ],
        ["certs/sphincs/bench_sphincs_small_level5_key.der", "bench_sphincs_small_level5_key" ],
        );


# ----------------------------------------------------------------------------

my $num_ecc = @fileList_ecc;
my $num_ed = @fileList_ed;
my $num_x = @fileList_x;
my $num_1024 = @fileList_1024;
my $num_2048 = @fileList_2048;
my $num_3072 = @fileList_3072;
my $num_4096 = @fileList_4096;
my $num_falcon = @fileList_falcon;
my $num_sphincs = @fileList_sphincs;

# open our output file, "+>" creates and/or truncates
open OUT_FILE, "+>", $outputFile  or die $!;

print OUT_FILE "/* certs_test.h */\n";
print OUT_FILE "/* This file was generated using: ./gencertbuf.pl */\n\n";
print OUT_FILE "#ifndef WOLFSSL_CERTS_TEST_H\n";
print OUT_FILE "#define WOLFSSL_CERTS_TEST_H\n\n";

# convert and print 1024-bit cert/keys
print OUT_FILE "#ifdef USE_CERT_BUFFERS_1024\n\n";
for (my $i = 0; $i < $num_1024; $i++) {

    my $fname = $fileList_1024[$i][0];
    my $sname = $fileList_1024[$i][1];

    print OUT_FILE "/* $fname, 1024-bit */\n";
    print OUT_FILE "static const unsigned char $sname\[] =\n";
    print OUT_FILE "{\n";
    file_to_hex($fname);
    print OUT_FILE "};\n";
    print OUT_FILE "static const int sizeof_$sname = sizeof($sname);\n\n";
}
print OUT_FILE "#endif /* USE_CERT_BUFFERS_1024 */\n\n";


# convert and print 2048-bit certs/keys
print OUT_FILE "#ifdef USE_CERT_BUFFERS_2048\n\n";
for (my $i = 0; $i < $num_2048; $i++) {

    my $fname = $fileList_2048[$i][0];
    my $sname = $fileList_2048[$i][1];

    print OUT_FILE "/* $fname, 2048-bit */\n";
    print OUT_FILE "static const unsigned char $sname\[] =\n";
    print OUT_FILE "{\n";
    file_to_hex($fname);
    print OUT_FILE "};\n";
    print OUT_FILE "static const int sizeof_$sname = sizeof($sname);\n\n";
}


print OUT_FILE "#endif /* USE_CERT_BUFFERS_2048 */\n\n";


# convert and print 3072-bit certs/keys
print OUT_FILE "#ifdef USE_CERT_BUFFERS_3072\n\n";
for (my $i = 0; $i < $num_3072; $i++) {

    my $fname = $fileList_3072[$i][0];
    my $sname = $fileList_3072[$i][1];

    print OUT_FILE "/* $fname, 3072-bit */\n";
    print OUT_FILE "static const unsigned char $sname\[] =\n";
    print OUT_FILE "{\n";
    file_to_hex($fname);
    print OUT_FILE "};\n";
    print OUT_FILE "static const int sizeof_$sname = sizeof($sname);\n\n";
}

print OUT_FILE "#endif /* USE_CERT_BUFFERS_3072 */\n\n";


# convert and print 4096-bit certs/keys
print OUT_FILE "#ifdef USE_CERT_BUFFERS_4096\n\n";
for (my $i = 0; $i < $num_4096; $i++) {

    my $fname = $fileList_4096[$i][0];
    my $sname = $fileList_4096[$i][1];

    print OUT_FILE "/* $fname, 4096-bit */\n";
    print OUT_FILE "static const unsigned char $sname\[] =\n";
    print OUT_FILE "{\n";
    file_to_hex($fname);
    print OUT_FILE "};\n";
    print OUT_FILE "static const int sizeof_$sname = sizeof($sname);\n\n";
}

print OUT_FILE "#endif /* USE_CERT_BUFFERS_4096 */\n\n";

# convert and print falcon keys
print OUT_FILE "#if defined(HAVE_FALCON)\n\n";
for (my $i = 0; $i < $num_falcon; $i++) {

    my $fname = $fileList_falcon[$i][0];
    my $sname = $fileList_falcon[$i][1];

    print OUT_FILE "/* $fname */\n";
    print OUT_FILE "static const unsigned char $sname\[] =\n";
    print OUT_FILE "{\n";
    file_to_hex($fname);
    print OUT_FILE "};\n";
    print OUT_FILE "static const int sizeof_$sname = sizeof($sname);\n\n";
}

print OUT_FILE "#endif /* HAVE_FALCON */\n\n";

# print dilithium raw keys
print OUT_FILE "#if defined(HAVE_DILITHIUM)
#ifndef WOLFSSL_DILITHIUM_NO_SIGN

/* raw private key without ASN1 syntax from
 * ./certs/dilithium/bench_dilithium_level2_key.der */
static const unsigned char bench_dilithium_level2_key[] = {
    0xea, 0x05, 0x24, 0x0d, 0x80, 0x72, 0x25, 0x55, 0xf4, 0x5b,
    0xc2, 0x13, 0x8b, 0x87, 0x5d, 0x31, 0x99, 0x2f, 0x1d, 0xa9,
    0x41, 0x09, 0x05, 0x76, 0xa7, 0xb7, 0x5e, 0x8c, 0x44, 0xe2,
    0x64, 0x79, 0xd8, 0x79, 0x4c, 0xee, 0x92, 0x2b, 0x37, 0xab,
    0xb1, 0x16, 0x65, 0x72, 0xc3, 0x49, 0xc2, 0xec, 0xfd, 0x9a,
    0xe6, 0x2d, 0x1e, 0x5b, 0xe3, 0x04, 0x96, 0x16, 0xad, 0x97,
    0x5d, 0xac, 0xf2, 0xcc, 0x62, 0x2e, 0x34, 0x5d, 0x67, 0x19,
    0x47, 0xee, 0x0f, 0x8b, 0x97, 0x60, 0xb4, 0x0b, 0xeb, 0x6a,
    0x7a, 0x75, 0x14, 0x27, 0x00, 0x39, 0xd6, 0x60, 0xce, 0x39,
    0x6e, 0x69, 0x46, 0xe1, 0x0d, 0xf9, 0xa6, 0xfa, 0x8c, 0xcf,
    0x65, 0x50, 0x59, 0x1d, 0xb0, 0x26, 0xc2, 0xe2, 0xf1, 0xb9,
    0xcd, 0x09, 0x60, 0xcc, 0xbb, 0x57, 0xd6, 0xac, 0xcc, 0xf9,
    0x58, 0x73, 0xa8, 0x81, 0x61, 0x2f, 0xd2, 0xa4, 0x5b, 0x98,
    0x0d, 0x12, 0x88, 0x51, 0x63, 0x38, 0x6e, 0xa2, 0x46, 0x64,
    0x52, 0xc0, 0x71, 0xc1, 0x42, 0x68, 0xd8, 0x42, 0x32, 0x5c,
    0xb4, 0x44, 0x08, 0x95, 0x48, 0xa2, 0x46, 0x6c, 0x0b, 0x10,
    0x09, 0xc8, 0x24, 0x4d, 0x18, 0x37, 0x4c, 0x4c, 0x82, 0x05,
    0x02, 0x22, 0x10, 0x4a, 0x86, 0x30, 0x03, 0x03, 0x11, 0x44,
    0x22, 0x62, 0x01, 0xa9, 0x51, 0x13, 0x02, 0x2c, 0x19, 0x85,
    0x65, 0x51, 0x14, 0x01, 0x9c, 0xb2, 0x81, 0x0a, 0x49, 0x52,
    0xa2, 0xb2, 0x4c, 0x98, 0x34, 0x01, 0x0a, 0x07, 0x06, 0x58,
    0xb2, 0x69, 0x51, 0x24, 0x2d, 0x59, 0x12, 0x52, 0xe0, 0xb4,
    0x04, 0x14, 0x40, 0x29, 0xa2, 0xb0, 0x31, 0x54, 0xc0, 0x40,
    0x63, 0x00, 0x69, 0x18, 0x47, 0x85, 0xc8, 0x30, 0x81, 0x0b,
    0x15, 0x0a, 0xd8, 0xa0, 0x0c, 0x5c, 0x20, 0x4a, 0x11, 0x38,
    0x64, 0x04, 0x94, 0x84, 0xd3, 0x24, 0x72, 0x58, 0x38, 0x28,
    0x18, 0x37, 0x6d, 0x94, 0xc0, 0x4d, 0xa0, 0xa6, 0x0c, 0x9a,
    0x82, 0x31, 0xc2, 0x40, 0x48, 0xda, 0x46, 0x85, 0x03, 0x00,
    0x05, 0xd8, 0x02, 0x4d, 0x0b, 0x85, 0x40, 0xe2, 0x32, 0x86,
    0x4c, 0xa0, 0x65, 0x8a, 0x36, 0x65, 0x42, 0x18, 0x6e, 0x60,
    0x36, 0x0d, 0x40, 0xc0, 0x01, 0x5a, 0x44, 0x42, 0xc4, 0xa4,
    0x0d, 0xd4, 0x88, 0x8d, 0x88, 0x22, 0x52, 0x00, 0xc0, 0x0c,
    0x5b, 0x36, 0x90, 0x09, 0x20, 0x22, 0x08, 0x03, 0x12, 0x90,
    0x12, 0x42, 0x04, 0x20, 0x29, 0x8c, 0x48, 0x6d, 0x20, 0x32,
    0x08, 0x94, 0x88, 0x6c, 0x10, 0x87, 0x21, 0xc1, 0x44, 0x02,
    0x52, 0x40, 0x12, 0xdb, 0xc8, 0x24, 0x14, 0x09, 0x2c, 0x93,
    0x40, 0x09, 0x64, 0xc8, 0x4c, 0x08, 0x48, 0x70, 0xa1, 0x10,
    0x81, 0x4a, 0x80, 0x8c, 0x20, 0x03, 0x31, 0x18, 0xb3, 0x80,
    0xd3, 0x82, 0x25, 0x4c, 0x94, 0x8c, 0x1c, 0x93, 0x89, 0x1a,
    0x91, 0x51, 0xd1, 0xb6, 0x68, 0x43, 0x14, 0x25, 0x84, 0x48,
    0x61, 0x82, 0x40, 0x24, 0xdb, 0x22, 0x4d, 0x63, 0x16, 0x66,
    0x62, 0x90, 0x50, 0xa1, 0x18, 0x86, 0x49, 0x28, 0x25, 0xa0,
    0x10, 0x68, 0x8c, 0x04, 0x00, 0x08, 0x32, 0x4e, 0x22, 0x43,
    0x31, 0x42, 0x96, 0x28, 0x11, 0x23, 0x89, 0xd2, 0xc4, 0x6d,
    0x11, 0x82, 0x8d, 0x8a, 0xa8, 0x90, 0xd2, 0x06, 0x29, 0x80,
    0x82, 0x89, 0x00, 0xa8, 0x41, 0x00, 0x13, 0x6a, 0x12, 0xa8,
    0x04, 0x83, 0xc2, 0x51, 0x13, 0x09, 0x08, 0x62, 0xb4, 0x8d,
    0x94, 0xc2, 0x44, 0x5a, 0xb4, 0x08, 0x0a, 0x10, 0x48, 0xa1,
    0x28, 0x20, 0x1b, 0xb7, 0x64, 0x60, 0x24, 0x25, 0x48, 0xc0,
    0x00, 0x0a, 0x10, 0x09, 0x64, 0xb8, 0x88, 0xcb, 0x44, 0x64,
    0x54, 0x90, 0x05, 0xd2, 0xb8, 0x21, 0x49, 0x28, 0x28, 0x49,
    0x42, 0x0d, 0x63, 0xa0, 0x65, 0xcb, 0x90, 0x30, 0x51, 0x82,
    0x8d, 0x5c, 0xc6, 0x0c, 0x51, 0x06, 0x6a, 0x1a, 0x27, 0x22,
    0x01, 0xa8, 0x24, 0x61, 0xb2, 0x84, 0x23, 0x40, 0x86, 0xa3,
    0xb4, 0x48, 0x19, 0x28, 0x0c, 0x14, 0x06, 0x2e, 0xe2, 0x02,
    0x0d, 0xc4, 0x90, 0x09, 0x08, 0x06, 0x66, 0x9b, 0xc8, 0x10,
    0x5c, 0x46, 0x21, 0xca, 0xa8, 0x30, 0x83, 0x20, 0x89, 0x03,
    0x83, 0x6c, 0xa1, 0x46, 0x8c, 0x90, 0x14, 0x4c, 0x99, 0x02,
    0x81, 0x53, 0x02, 0x10, 0x8b, 0x48, 0x91, 0xe4, 0x40, 0x4a,
    0x22, 0xb1, 0x88, 0xc1, 0x06, 0x0e, 0xc3, 0xa8, 0x08, 0xc8,
    0x46, 0x92, 0x03, 0xb5, 0x4c, 0x23, 0x03, 0x0c, 0xa4, 0x06,
    0x2e, 0xdc, 0x92, 0x81, 0x0c, 0x45, 0x22, 0x40, 0x34, 0x91,
    0x90, 0x96, 0x48, 0x81, 0x82, 0x31, 0xcb, 0x16, 0x72, 0x49,
    0xc8, 0x29, 0x44, 0x86, 0x90, 0x60, 0x22, 0x4e, 0x42, 0x42,
    0x09, 0x4b, 0x82, 0x20, 0x0a, 0xb2, 0x64, 0x20, 0x86, 0x70,
    0x1a, 0xc0, 0x00, 0x1c, 0x41, 0x49, 0x89, 0x84, 0x05, 0x0c,
    0x36, 0x49, 0x19, 0x99, 0x6d, 0x00, 0x08, 0x50, 0x23, 0x96,
    0x6c, 0xe0, 0x44, 0x08, 0x98, 0x24, 0x2c, 0x0a, 0x23, 0x20,
    0x12, 0x04, 0x31, 0xc9, 0x06, 0x32, 0x14, 0x01, 0x41, 0x08,
    0x37, 0x08, 0x58, 0x00, 0x0c, 0x19, 0x04, 0x29, 0x90, 0x18,
    0x05, 0xe1, 0x88, 0x44, 0xc2, 0x20, 0x6c, 0xd1, 0x46, 0x64,
    0xd9, 0x26, 0x62, 0x09, 0x88, 0x68, 0x02, 0x29, 0x29, 0xe1,
    0x18, 0x65, 0x98, 0x04, 0x24, 0xe4, 0x34, 0x0c, 0x12, 0x85,
    0x2d, 0x20, 0x14, 0x06, 0x24, 0x15, 0x82, 0x89, 0x08, 0x91,
    0x60, 0x84, 0x28, 0x24, 0x34, 0x41, 0x1b, 0x49, 0x22, 0xd3,
    0x96, 0x64, 0x1b, 0x86, 0x4c, 0x0c, 0xb9, 0x20, 0x20, 0x39,
    0x04, 0x04, 0x34, 0x6d, 0xc1, 0x28, 0x32, 0x08, 0x14, 0x44,
    0x81, 0x18, 0x2e, 0xda, 0x38, 0x41, 0x63, 0x18, 0x26, 0xd8,
    0x48, 0x26, 0x12, 0x20, 0x21, 0x09, 0xc5, 0x25, 0x92, 0x42,
    0x0c, 0x88, 0x04, 0x64, 0x11, 0x43, 0x8a, 0x19, 0x92, 0x60,
    0x5c, 0xc6, 0x31, 0xa1, 0x24, 0x6a, 0xd8, 0xb6, 0x49, 0x1b,
    0x81, 0x90, 0xe2, 0x32, 0x4e, 0x62, 0x44, 0x21, 0x80, 0xb8,
    0x10, 0x4b, 0x90, 0x49, 0x5c, 0x06, 0x09, 0x48, 0x20, 0x49,
    0xa2, 0x92, 0x71, 0x5c, 0x48, 0x02, 0xc8, 0x08, 0x81, 0xa4,
    0x32, 0x66, 0xc9, 0x30, 0x11, 0xca, 0x92, 0x91, 0xc0, 0x00,
    0x41, 0x44, 0x98, 0x4d, 0x98, 0x12, 0x4e, 0x92, 0x46, 0x8e,
    0x49, 0xb8, 0x64, 0xdc, 0x18, 0x50, 0x51, 0xb4, 0x48, 0x08,
    0x47, 0x24, 0x08, 0x46, 0x32, 0x1b, 0x23, 0x00, 0x09, 0xb8,
    0x04, 0x0a, 0x44, 0x0c, 0x0b, 0xc7, 0x8d, 0x19, 0xa4, 0x09,
    0x11, 0x30, 0x41, 0xe3, 0x24, 0x45, 0x89, 0x1f, 0x65, 0x54,
    0xf6, 0x38, 0x04, 0x37, 0xcc, 0x89, 0xc3, 0xc5, 0xdc, 0x43,
    0xd9, 0x13, 0x56, 0x06, 0x05, 0x50, 0x29, 0x4e, 0x0f, 0xa5,
    0x5c, 0x5d, 0xd7, 0x82, 0xa1, 0x63, 0x59, 0x0d, 0x3e, 0x5b,
    0x00, 0xe6, 0x0e, 0xd8, 0x1c, 0xc7, 0xaf, 0xc0, 0x48, 0xb6,
    0x07, 0x5c, 0x65, 0x00, 0x89, 0xb3, 0x09, 0xbc, 0x4a, 0xaa,
    0xa6, 0x72, 0xbe, 0x6b, 0x9a, 0xb3, 0x5b, 0x27, 0x82, 0x65,
    0x9b, 0xc9, 0x6f, 0x19, 0x88, 0x94, 0x0b, 0x37, 0x44, 0x2f,
    0xe3, 0x9a, 0x02, 0xda, 0xff, 0x11, 0xb0, 0x48, 0x89, 0x70,
    0x8c, 0x84, 0xc2, 0xc0, 0x31, 0x4a, 0xad, 0x70, 0xe1, 0xa7,
    0x15, 0xfd, 0xb2, 0x6d, 0x93, 0xda, 0x17, 0x68, 0xc4, 0xe3,
    0xfd, 0x2c, 0x08, 0x15, 0xb9, 0xa4, 0xc5, 0x1b, 0x97, 0xc9,
    0xa3, 0xaf, 0x0d, 0x21, 0x06, 0x3d, 0xf1, 0x05, 0xd4, 0x35,
    0x80, 0x2e, 0x23, 0x99, 0xbd, 0x3a, 0x1a, 0x6c, 0xad, 0xbf,
    0x56, 0xb5, 0xd3, 0x95, 0x1b, 0x30, 0x4d, 0x56, 0xc1, 0x77,
    0xe6, 0xd6, 0xab, 0x94, 0x46, 0x68, 0xd7, 0xb8, 0xe4, 0x9d,
    0xb2, 0x8d, 0xc4, 0xd1, 0xc8, 0x92, 0xbe, 0x5d, 0x1f, 0x58,
    0x55, 0x7f, 0x11, 0x55, 0xc5, 0x2e, 0xc3, 0x9e, 0x2a, 0x29,
    0x51, 0xe8, 0x75, 0x49, 0xa7, 0xa3, 0xda, 0x0b, 0xcf, 0xf8,
    0x3f, 0x78, 0xac, 0x4c, 0x4e, 0x78, 0x6f, 0x0e, 0x67, 0xad,
    0x94, 0x59, 0x20, 0x5e, 0x37, 0x18, 0xb9, 0x09, 0x87, 0xdb,
    0xdd, 0xf0, 0xc2, 0x4d, 0x03, 0xcc, 0x98, 0x22, 0x4b, 0xe5,
    0x7d, 0x8e, 0x74, 0x7e, 0xa9, 0x1b, 0xeb, 0x7a, 0xae, 0xaf,
    0x2e, 0x7c, 0x3c, 0xc0, 0x1a, 0x30, 0x40, 0x0d, 0x79, 0x86,
    0x53, 0xcc, 0x0b, 0x2b, 0xbe, 0xa5, 0x72, 0x3b, 0xbb, 0x53,
    0x9e, 0xd5, 0xc2, 0x23, 0x1d, 0x35, 0xcd, 0x22, 0x12, 0xed,
    0x9a, 0xee, 0xc8, 0xf9, 0x05, 0x27, 0xdb, 0x46, 0x56, 0xcc,
    0x24, 0x4d, 0xee, 0xaf, 0xab, 0xa9, 0x78, 0x75, 0x75, 0xb9,
    0xd1, 0xfd, 0x39, 0x3a, 0xb2, 0xa2, 0xeb, 0x87, 0x76, 0xb2,
    0x19, 0x47, 0x88, 0xab, 0x42, 0x85, 0x4b, 0xd9, 0x76, 0x22,
    0x68, 0x4b, 0xc9, 0x88, 0x38, 0x28, 0x0a, 0x34, 0x5d, 0x12,
    0x4f, 0xf5, 0x43, 0x64, 0x44, 0x8c, 0x3c, 0xc2, 0x99, 0x91,
    0x4e, 0xfd, 0xfd, 0x9c, 0x73, 0xbf, 0x85, 0xf9, 0x9f, 0xe1,
    0x53, 0x19, 0xc8, 0x19, 0xcb, 0x7c, 0xdb, 0x9a, 0x3a, 0x2c,
    0x34, 0x55, 0x8c, 0x64, 0x6f, 0xc5, 0xb7, 0x93, 0x53, 0xb4,
    0x97, 0x7e, 0xc2, 0xf8, 0x7e, 0x8d, 0x44, 0x10, 0xca, 0x49,
    0xf5, 0x5c, 0xe8, 0xce, 0xc4, 0xcc, 0x42, 0xf0, 0x85, 0xf1,
    0xf2, 0x10, 0xa7, 0x0b, 0x37, 0x6a, 0x8e, 0x50, 0x96, 0x96,
    0x9d, 0xd9, 0x8f, 0x54, 0x45, 0x56, 0xf8, 0x64, 0x88, 0xab,
    0x51, 0x4f, 0x9f, 0x61, 0xd9, 0x12, 0x87, 0xac, 0x1d, 0xc1,
    0x23, 0xea, 0xb3, 0x5d, 0xa4, 0x6d, 0xfa, 0x58, 0x92, 0x8f,
    0x77, 0x78, 0x61, 0xe5, 0xe4, 0x33, 0xdb, 0x10, 0x2d, 0xdd,
    0xb6, 0xd7, 0xb4, 0xd0, 0x8d, 0xd1, 0xa8, 0x0b, 0x94, 0xdf,
    0xcf, 0xd7, 0xac, 0xdf, 0x47, 0x0b, 0x38, 0xe0, 0xa5, 0xf8,
    0xc3, 0xd2, 0xc3, 0xfb, 0x0f, 0x98, 0x00, 0x2b, 0x17, 0x3c,
    0x44, 0x70, 0x36, 0x47, 0x27, 0x89, 0x41, 0xcb, 0x87, 0x5a,
    0xa4, 0x2c, 0x57, 0x6d, 0x8c, 0xcb, 0xc0, 0x7d, 0x6b, 0xf5,
    0xa1, 0x17, 0x39, 0x4a, 0xb5, 0xac, 0xc6, 0x41, 0x90, 0x66,
    0x85, 0xc4, 0x4b, 0x18, 0xc6, 0xe6, 0x09, 0x6d, 0x6e, 0xbb,
    0x7f, 0x72, 0x96, 0xd3, 0x21, 0x5a, 0x96, 0xaf, 0x9e, 0xb6,
    0x0b, 0x3f, 0xe8, 0x83, 0xe5, 0x53, 0x11, 0x81, 0xc6, 0xab,
    0x40, 0xa9, 0x09, 0xb6, 0x74, 0x5e, 0xe1, 0xc3, 0x82, 0x1e,
    0xda, 0x2f, 0x24, 0xe0, 0x94, 0x8f, 0x07, 0xb7, 0x9b, 0xc6,
    0x50, 0xef, 0x3a, 0x79, 0x89, 0x4d, 0x6f, 0x16, 0x33, 0x04,
    0x24, 0x7e, 0x4a, 0xab, 0x5d, 0x03, 0x29, 0xad, 0xba, 0xa3,
    0x6c, 0xe2, 0x05, 0xab, 0x4d, 0x69, 0xb6, 0x61, 0x39, 0x9d,
    0xc3, 0x53, 0x11, 0xc0, 0xe3, 0xaa, 0x2e, 0xdc, 0x74, 0x09,
    0xbd, 0x19, 0xb5, 0xbb, 0x51, 0x1e, 0x77, 0x3e, 0xce, 0x64,
    0x13, 0xeb, 0x74, 0x03, 0xb7, 0x49, 0x99, 0xb0, 0x71, 0x99,
    0xe6, 0x17, 0x3c, 0x80, 0xe6, 0xb5, 0x51, 0xe9, 0xb3, 0xe4,
    0x2b, 0xaa, 0x52, 0x15, 0x99, 0x4e, 0x46, 0x6d, 0x67, 0x8e,
    0x79, 0xc4, 0x3c, 0xa6, 0xdc, 0x8f, 0xed, 0x87, 0xb9, 0x68,
    0x6d, 0xdc, 0x19, 0xa1, 0x52, 0x37, 0x06, 0x76, 0xad, 0xe9,
    0x61, 0x5c, 0x82, 0x16, 0x81, 0xaf, 0x3a, 0x89, 0xbf, 0x72,
    0xb0, 0xc7, 0x88, 0x3c, 0x58, 0xfe, 0xe4, 0xa5, 0x41, 0x50,
    0xfc, 0x8a, 0x15, 0xb0, 0x78, 0xd4, 0x77, 0x06, 0x4b, 0xc4,
    0x21, 0x7f, 0xaa, 0x2b, 0x88, 0x7f, 0x8c, 0x3b, 0x9b, 0xbb,
    0x2e, 0x41, 0xcf, 0x9b, 0x06, 0xd3, 0x4d, 0xcf, 0xb2, 0x9c,
    0x91, 0x46, 0x35, 0x3a, 0x5a, 0x0b, 0xe4, 0xac, 0x96, 0x7c,
    0xe0, 0xd4, 0x34, 0xe5, 0xab, 0xae, 0xa7, 0x67, 0xbf, 0x4d,
    0xab, 0x48, 0xfd, 0xcb, 0x3f, 0x5c, 0xde, 0x3f, 0x83, 0xcc,
    0x52, 0x0f, 0xdd, 0x7f, 0x20, 0x25, 0xed, 0xee, 0xd0, 0x14,
    0x38, 0xf7, 0x33, 0x4c, 0x3c, 0x5e, 0x23, 0x80, 0xa3, 0x0a,
    0xe8, 0xb0, 0xef, 0x5b, 0xca, 0xc9, 0x97, 0x13, 0x98, 0xfe,
    0x91, 0x62, 0x14, 0xa8, 0x64, 0xf6, 0x20, 0xc9, 0xc9, 0x6f,
    0x8b, 0xc0, 0xec, 0x39, 0x15, 0xa7, 0x59, 0x62, 0x68, 0x21,
    0xe1, 0x5f, 0xf6, 0xa1, 0x76, 0xb0, 0xca, 0x1b, 0x2a, 0x71,
    0xe3, 0x1a, 0x24, 0x91, 0x1f, 0x3a, 0xbb, 0xf1, 0xc9, 0x09,
    0x42, 0x48, 0x7e, 0x19, 0x1b, 0xf1, 0xf0, 0x13, 0x33, 0xf1,
    0x62, 0x31, 0x00, 0x97, 0x73, 0x9b, 0x3c, 0x26, 0xf8, 0x42,
    0xd0, 0xd4, 0x41, 0x1b, 0x9f, 0x7e, 0x43, 0x4b, 0x0b, 0x08,
    0xd7, 0xa0, 0xa8, 0x32, 0x34, 0x0a, 0xc9, 0xef, 0xb8, 0xeb,
    0xe7, 0x64, 0x3b, 0x40, 0x88, 0xe0, 0x60, 0x59, 0x07, 0xef,
    0xb9, 0x5f, 0x71, 0x92, 0x90, 0xa4, 0x5f, 0x34, 0x38, 0x93,
    0x92, 0x43, 0x87, 0xaf, 0xdd, 0x87, 0x63, 0x8c, 0x1d, 0xe5,
    0x86, 0x9e, 0xe6, 0xde, 0x94, 0xdd, 0x33, 0x5d, 0x95, 0x64,
    0xd8, 0xc4, 0x8a, 0x3c, 0xe7, 0x4b, 0xd6, 0x3f, 0xc5, 0x69,
    0x6a, 0xa8, 0x7f, 0x0f, 0x93, 0x77, 0x02, 0x46, 0x66, 0xa5,
    0xa0, 0x60, 0x8b, 0xec, 0xb1, 0xa2, 0xfc, 0x2a, 0x09, 0xb8,
    0x08, 0x1c, 0x05, 0x6b, 0x78, 0xb7, 0x7a, 0xe5, 0x60, 0xa4,
    0xaf, 0x3a, 0x9d, 0xaa, 0xf5, 0x22, 0x9b, 0x5e, 0xef, 0xc3,
    0x46, 0xed, 0x67, 0xd0, 0x8b, 0xda, 0xb4, 0xa3, 0x34, 0x32,
    0x20, 0x9d, 0x88, 0x7e, 0x43, 0x42, 0x6f, 0x02, 0xf8, 0x48,
    0x9b, 0xc5, 0x02, 0xad, 0xaa, 0xa9, 0xee, 0x19, 0x1b, 0xde,
    0x02, 0x83, 0x81, 0x10, 0xa6, 0x79, 0x4e, 0xad, 0x15, 0xf7,
    0x3e, 0x4e, 0x1e, 0x72, 0xfe, 0x52, 0x49, 0x24, 0xce, 0x82,
    0x31, 0x59, 0x72, 0xae, 0xd5, 0x34, 0x50, 0x87, 0x8b, 0xe3,
    0x8e, 0xec, 0x61, 0x35, 0x13, 0x57, 0xb1, 0xe6, 0xac, 0xfb,
    0x16, 0xc3, 0x1a, 0x98, 0x92, 0xcb, 0xcd, 0xc9, 0xf7, 0x10,
    0x6a, 0x43, 0x96, 0x33, 0x2d, 0x6f, 0x6c, 0x76, 0xb0, 0xf6,
    0x48, 0x4c, 0xae, 0x13, 0x67, 0x5d, 0x42, 0x01, 0x8e, 0x54,
    0x51, 0xcc, 0x65, 0xf1, 0x95, 0x11, 0x3c, 0x96, 0x2a, 0x5a,
    0x42, 0x3d, 0x9b, 0xbb, 0xb7, 0x7b, 0x28, 0x96, 0x09, 0xbb,
    0xed, 0x2d, 0xbc, 0xb7, 0x90, 0x62, 0xd3, 0xbe, 0xbd, 0xae,
    0x50, 0x15, 0x96, 0xc1, 0x03, 0x91, 0x14, 0x34, 0x4f, 0x21,
    0xa5, 0x6e, 0x78, 0x4a, 0x5d, 0x8b, 0xcf, 0x5b, 0x1a, 0x8a,
    0x57, 0x43, 0xb8, 0x25, 0xd3, 0xa2, 0xcd, 0x78, 0xb4, 0x93,
    0x07, 0x7a, 0x14, 0xc1, 0x0c, 0x6f, 0x5f, 0x5e, 0xcb, 0x11,
    0x17, 0x81, 0x0d, 0x7d, 0x0f, 0xda, 0xd1, 0x92, 0x43, 0x56,
    0xaf, 0x75, 0x53, 0x44, 0x1f, 0xc7, 0x9c, 0xd3, 0xc5, 0x47,
    0xe0, 0xac, 0x4a, 0x11, 0xe4, 0xfe, 0x6c, 0x80, 0x79, 0xcc,
    0x60, 0x7a, 0xd9, 0x56, 0x65, 0x83, 0x5e, 0xcf, 0x37, 0x27,
    0x55, 0xe2, 0x4d, 0xf9, 0xd6, 0x09, 0x2d, 0xee, 0xda, 0x10,
    0x6b, 0xdc, 0xd2, 0x70, 0x46, 0x94, 0xaa, 0xf5, 0x21, 0xc5,
    0xf0, 0x79, 0xdb, 0x9b, 0x8e, 0x9a, 0xdb, 0x5a, 0x56, 0x41,
    0x43, 0xe7, 0x1f, 0x8d, 0xfd, 0xda, 0x12, 0x5f, 0xf7, 0x9e,
    0x47, 0x1a, 0xf7, 0x73, 0x40, 0x67, 0xc2, 0x61, 0x07, 0x33,
    0x16, 0x78, 0x60, 0x05, 0x85, 0x5c, 0x2f, 0x2b, 0xbf, 0x2c,
    0x7a, 0x39, 0xc6, 0xed, 0xcb, 0x43, 0x66, 0x27, 0x93, 0xcd,
    0x92, 0x8d, 0x62, 0x8c, 0xaa, 0x61, 0x1c, 0x9c, 0x4c, 0x90,
    0xba, 0xba, 0x4b, 0xc1, 0xf1, 0x22, 0xde, 0xe0, 0xf9, 0x3e,
    0x04, 0xb9, 0x56, 0xa3, 0x1c, 0xe8, 0xda, 0xd6, 0x09, 0x4a,
    0x7d, 0x89, 0xbc, 0xf4, 0xe8, 0x4d, 0xa1, 0xe8, 0x34, 0x90,
    0xa5, 0x31, 0x3a, 0xec, 0x56, 0xc5, 0xd2, 0x92, 0x0b, 0xe9,
    0x58, 0xbb, 0xb2, 0x84, 0x9b, 0xa9, 0x1d, 0x19, 0xdb, 0x7a,
    0x02, 0x75, 0x79, 0x16, 0x35, 0xee, 0x3a, 0x3f, 0x4e, 0x5e,
    0x11, 0x90, 0x04, 0x03, 0xce, 0x8b, 0xa0, 0xd8, 0xc1, 0xee,
    0x52, 0x33, 0x6e, 0xd2, 0x6e, 0x06, 0x5c, 0x99, 0x24, 0x6f,
    0x16, 0xd9, 0x90, 0x28, 0xe5, 0x2d, 0x91, 0x6f, 0x1a, 0x57,
    0xf0, 0x4c, 0x7c, 0x3f, 0x7b, 0xd7, 0x30, 0xed, 0x6d, 0x21,
    0xb7, 0xf8, 0xed, 0xf3, 0x34, 0x89, 0xfa, 0xf0, 0x51, 0x6f,
    0x99, 0xa0, 0x5e, 0xf8, 0x74, 0xc7, 0x4f, 0xb5, 0x59, 0x52,
    0xbe, 0x45, 0xac, 0x3f, 0x34, 0x51, 0x87, 0x6e, 0x84, 0xea,
    0xb0, 0x40, 0xe1, 0x84, 0x16, 0x66, 0x30, 0xf1, 0x5c, 0xb2,
    0x74, 0x25, 0x03, 0xe3, 0x2e, 0x82, 0xc5, 0x60, 0x9d, 0xe4,
    0xca, 0xec, 0x49, 0x6b, 0x4e, 0x5a, 0x09, 0xa8, 0xfe, 0xff,
    0x1d, 0xa1, 0xe8, 0xec, 0x9a, 0x22, 0x3b, 0xd6, 0x72, 0x93,
    0x6f, 0x6b, 0x5a, 0xfb, 0x2d, 0x5a, 0xde, 0x01, 0x3e, 0xf6,
    0xdc, 0x77, 0x55, 0x1e, 0x32, 0x19, 0xc8, 0xa1, 0xbb, 0xcf,
    0xcb, 0x41, 0x54, 0xa2, 0xcb, 0xe6, 0x61, 0xca, 0x43, 0x63,
    0xd2, 0x2c, 0xae, 0xf4, 0xd9, 0x49, 0xb1, 0x75, 0x1a, 0x06,
    0x92, 0x13, 0x90, 0x57, 0x89, 0x8e, 0x9f, 0x26, 0xc5, 0x14,
    0xd8, 0xc7, 0x93, 0xb2, 0xaa, 0x3a, 0x9c, 0x10, 0xd5, 0x68,
    0x52, 0x28, 0x39, 0xee, 0x30, 0xdc, 0x00, 0x4b, 0x65, 0x72,
    0x59, 0x98, 0xad, 0x2e, 0x8c, 0xaf, 0x4e, 0x79, 0x0a, 0x8c,
    0x0c, 0x9d, 0xb6, 0x43, 0x26, 0x83, 0x71, 0x7b, 0x1e, 0x86,
    0x4d, 0x33, 0xd7, 0x20, 0x29, 0x6a, 0xbf, 0x2f, 0x8e, 0x4b,
    0x13, 0x35, 0x65, 0xc8, 0xec, 0xe3, 0x2c, 0xde, 0xfb, 0x30,
    0x57, 0xa9, 0x92, 0x22, 0x5d, 0x79, 0x16, 0x07, 0x73, 0x9b,
    0xe2, 0x6e, 0xd4, 0x99, 0xb4, 0x35, 0xfd, 0xa2, 0xb5, 0xd9,
    0xe5, 0x74, 0xd1, 0xb2, 0xcf, 0x32, 0xf1, 0x19, 0x69, 0xcf,
    0x1e, 0x10, 0xcc, 0x3c, 0xaf, 0xbe, 0xa4, 0x33, 0x11, 0x83,
    0x64, 0xc0, 0x39, 0xe5, 0xb0, 0x8f, 0x32, 0xf4, 0x01, 0x6a,
    0x2a, 0x11, 0x8e, 0xdd, 0x03, 0x81, 0x39, 0xe7, 0x70, 0x16,
    0x2f, 0x0e, 0x24, 0xa9, 0x12, 0x0b, 0xdb, 0xa8, 0x6c, 0xb3,
    0xf3, 0x74, 0x95, 0xca, 0x64, 0x1d, 0xee, 0x25, 0xc5, 0x27,
    0xed, 0x0f, 0x82, 0xb5, 0x7a, 0x62, 0x27, 0xb2, 0x87, 0x53,
    0x11, 0x39, 0x5e, 0xb8, 0x11, 0xca, 0x25, 0xe8, 0x17, 0x46,
    0xd3, 0x0f, 0x5d, 0x70, 0x68, 0xe1, 0x5f, 0xd1, 0xab, 0x65,
    0xe5, 0x42, 0x87, 0x1e, 0x96, 0xaf, 0x13, 0x0c, 0x9b, 0x15,
    0x75, 0x14, 0x31, 0x75, 0xcc, 0x15, 0xbf, 0x2c, 0x74, 0xab,
    0xc9, 0x9c, 0xda, 0x62, 0x1d, 0xeb, 0x19, 0x81, 0x67, 0x5e,
    0xcd, 0x54, 0x87, 0x07, 0x67, 0xba, 0xe3, 0xf6, 0x03, 0xbe,
    0x6d, 0x64, 0x2d, 0xbc, 0xec, 0x54, 0x13, 0x12, 0x5b, 0x44,
    0x90, 0x95, 0x86, 0x77, 0x8c, 0x59, 0xbd, 0x8e, 0xba, 0xb1,
    0x12, 0xea, 0xc1, 0x94, 0x37, 0xa0, 0x11, 0xff, 0xb2, 0xa4,
    0xc3, 0x61, 0xf2, 0xa3, 0x49, 0xbe, 0xe7, 0xb6, 0x96, 0x2f,
};
static const int sizeof_bench_dilithium_level2_key = sizeof(bench_dilithium_level2_key);

#endif /* !WOLFSSL_DILITHIUM_NO_SIGN */

#ifndef WOLFSSL_DILITHIUM_NO_VERIFY

/* raw public key without ASN1 syntax from
 * ./certs/dilithium/bench_dilithium_level2_key.der */
static const unsigned char bench_dilithium_level2_pubkey[] = {
    0xea, 0x05, 0x24, 0x0d, 0x80, 0x72, 0x25, 0x55, 0xf4, 0x5b,
    0xc2, 0x13, 0x8b, 0x87, 0x5d, 0x31, 0x99, 0x2f, 0x1d, 0xa9,
    0x41, 0x09, 0x05, 0x76, 0xa7, 0xb7, 0x5e, 0x8c, 0x44, 0xe2,
    0x64, 0x79, 0xa0, 0xec, 0x1f, 0x24, 0xb6, 0xc8, 0x05, 0x5b,
    0xc1, 0x18, 0xb0, 0xb7, 0xcf, 0x8c, 0x60, 0x67, 0x6b, 0x81,
    0x44, 0x27, 0xb6, 0x0e, 0xfd, 0x9b, 0xc3, 0xcb, 0x52, 0x31,
    0xfa, 0xc9, 0x34, 0x8d, 0x22, 0x1e, 0x07, 0x9d, 0x96, 0x6a,
    0x63, 0x83, 0x5c, 0xd7, 0x83, 0x2d, 0x7f, 0x48, 0x64, 0x79,
    0xca, 0xb4, 0x9f, 0xa2, 0x02, 0xb7, 0x86, 0x1d, 0x0e, 0xc7,
    0xf9, 0x6c, 0x07, 0xc0, 0x35, 0x6a, 0x34, 0x79, 0x7c, 0xb8,
    0x0f, 0xed, 0x98, 0x50, 0xfb, 0x51, 0xe0, 0x36, 0x44, 0x4c,
    0xc6, 0x35, 0xa2, 0xbb, 0x55, 0xb0, 0x5c, 0x39, 0x08, 0x02,
    0x20, 0x35, 0x5c, 0x56, 0x6d, 0x2e, 0xb9, 0xef, 0x21, 0x26,
    0x87, 0x87, 0x85, 0x8a, 0x32, 0xb5, 0xa7, 0x68, 0x70, 0x3a,
    0xfd, 0x0d, 0x21, 0x48, 0x91, 0xa3, 0x29, 0xc1, 0x2a, 0x38,
    0xe5, 0x26, 0x31, 0x1f, 0x42, 0xde, 0x0b, 0x25, 0xff, 0x1d,
    0x6b, 0xb4, 0xe0, 0x5d, 0x2d, 0xcf, 0x44, 0xd5, 0x7d, 0xc4,
    0xf6, 0x95, 0xf2, 0x06, 0x4f, 0x83, 0x88, 0x9d, 0x1e, 0xeb,
    0x1c, 0x09, 0x45, 0x62, 0x67, 0x3d, 0xff, 0x51, 0x47, 0xe8,
    0xbc, 0x9b, 0x03, 0x1f, 0xc7, 0x72, 0x65, 0xce, 0xa8, 0x8c,
    0xc2, 0xa0, 0xc2, 0xbd, 0x5b, 0x7c, 0x17, 0x16, 0x8b, 0x72,
    0xfa, 0xb1, 0xbd, 0xdf, 0x49, 0xd6, 0xa1, 0x00, 0x65, 0xbe,
    0x82, 0xe7, 0x68, 0xc7, 0xe7, 0xbc, 0xc2, 0xa4, 0xdb, 0xaa,
    0xcc, 0xea, 0x41, 0x52, 0x7f, 0x56, 0xb4, 0x68, 0x1f, 0x92,
    0x96, 0x0f, 0xce, 0xd4, 0xd0, 0x87, 0x4c, 0x4a, 0x73, 0xb5,
    0x6c, 0xd4, 0x69, 0x55, 0x15, 0x47, 0xdc, 0x94, 0x7f, 0xd2,
    0x54, 0x5e, 0xb2, 0x90, 0xc2, 0x47, 0xe4, 0xf5, 0xde, 0x8b,
    0x9b, 0xc6, 0x5d, 0x50, 0x95, 0x60, 0xe0, 0xf0, 0xa7, 0x4e,
    0xe0, 0xcd, 0x41, 0x09, 0xef, 0xb3, 0x3d, 0x90, 0x5c, 0x77,
    0x54, 0xec, 0x9e, 0x5d, 0x8a, 0xe7, 0x09, 0x5c, 0xc9, 0x58,
    0x0c, 0xd0, 0x42, 0x35, 0xd2, 0x14, 0x59, 0x38, 0x69, 0xad,
    0xf9, 0xb5, 0xbf, 0x8a, 0x8e, 0x33, 0xd8, 0x5e, 0x7a, 0x55,
    0xd0, 0x53, 0x15, 0x40, 0x4e, 0xc5, 0x86, 0xd7, 0x8f, 0x5f,
    0x2f, 0x55, 0x82, 0xc2, 0x4f, 0x16, 0xe5, 0xea, 0x1c, 0xbc,
    0xff, 0x5e, 0x1f, 0x39, 0x46, 0x70, 0x54, 0x7a, 0x3a, 0x27,
    0x16, 0x1a, 0x2b, 0x6c, 0xd2, 0xb7, 0x80, 0xd3, 0xd1, 0x9d,
    0x25, 0x59, 0xed, 0xe6, 0x51, 0xb1, 0xf2, 0xad, 0x7e, 0x51,
    0x78, 0x14, 0x2b, 0x19, 0xae, 0x64, 0x72, 0x0f, 0xd8, 0x18,
    0x79, 0x8e, 0x66, 0x88, 0xd3, 0xa4, 0xa3, 0xc3, 0x76, 0x21,
    0xcb, 0xe4, 0x79, 0x5e, 0x95, 0x74, 0xe3, 0x31, 0x18, 0x79,
    0xed, 0xc7, 0xe7, 0xfb, 0x86, 0x48, 0x1b, 0x7b, 0x75, 0x5b,
    0x7f, 0x7c, 0x82, 0xc5, 0xab, 0x11, 0xb4, 0x5d, 0x59, 0x6f,
    0x78, 0xb2, 0xa5, 0x39, 0xc6, 0x63, 0x38, 0x6c, 0xeb, 0x50,
    0x06, 0x14, 0x76, 0xf0, 0xe8, 0xfb, 0x11, 0x95, 0x1f, 0x9d,
    0x9c, 0xa6, 0xe1, 0xe2, 0x0d, 0xa3, 0x66, 0xfc, 0x20, 0x83,
    0x50, 0x0e, 0x53, 0x75, 0xb5, 0x12, 0xf4, 0xdf, 0x31, 0x46,
    0x83, 0xac, 0x5b, 0xf3, 0x99, 0xa6, 0xd1, 0x7b, 0x2b, 0xc5,
    0xdc, 0x71, 0x07, 0x27, 0x33, 0x35, 0x34, 0xf5, 0x30, 0x19,
    0xc1, 0x3b, 0xba, 0x8a, 0xaf, 0x7e, 0x49, 0x93, 0x48, 0x5b,
    0x38, 0xc0, 0xbc, 0x2e, 0xc7, 0x59, 0x1b, 0xd9, 0xf5, 0xcc,
    0x86, 0xf5, 0x7b, 0x4d, 0xd7, 0x39, 0xa7, 0xa2, 0x56, 0x20,
    0x48, 0x98, 0x7d, 0x4f, 0x75, 0x56, 0x9b, 0xb8, 0x95, 0x45,
    0x17, 0xf3, 0x86, 0x3d, 0x97, 0x0a, 0x49, 0x1b, 0xca, 0xff,
    0x20, 0xc0, 0x24, 0x2c, 0x51, 0xc2, 0x0a, 0x3c, 0xbf, 0x07,
    0x60, 0x1c, 0x88, 0x85, 0x9b, 0x85, 0x2d, 0x4a, 0xfe, 0x5a,
    0x1c, 0x90, 0xf5, 0x90, 0x12, 0xd3, 0x03, 0x3c, 0x8c, 0x2e,
    0x95, 0x4a, 0x47, 0x76, 0x0f, 0x1f, 0x5d, 0x9e, 0xed, 0xc5,
    0x64, 0xc4, 0x9b, 0xbf, 0x86, 0xc5, 0x63, 0x84, 0x33, 0x00,
    0xf1, 0x26, 0x18, 0x21, 0xf3, 0x88, 0x1a, 0x08, 0x18, 0x6d,
    0x2f, 0xef, 0xd5, 0xeb, 0x2f, 0x69, 0xc8, 0x6e, 0x92, 0x34,
    0xfc, 0x72, 0x3d, 0x9a, 0xa7, 0x9e, 0x51, 0xfb, 0x56, 0xe3,
    0xdc, 0xf4, 0x8f, 0x9b, 0x6d, 0x0d, 0x2a, 0xec, 0x66, 0x12,
    0x26, 0x35, 0xbd, 0x61, 0xc2, 0x67, 0x19, 0xf5, 0x7e, 0xa1,
    0x67, 0xa2, 0x9c, 0x3b, 0x67, 0xb0, 0xc2, 0x51, 0x6a, 0x37,
    0x7c, 0x48, 0xe9, 0x4b, 0xb9, 0xa3, 0x38, 0x2f, 0xfc, 0xde,
    0xb4, 0x7c, 0xda, 0x52, 0x84, 0x0b, 0xb0, 0xd9, 0x08, 0xe9,
    0x7a, 0x4a, 0x6f, 0x79, 0x29, 0x3d, 0xc4, 0x5c, 0x78, 0xee,
    0x63, 0xb6, 0x96, 0x68, 0xd9, 0x82, 0x4e, 0xc1, 0x1b, 0x6f,
    0x52, 0xf5, 0xb3, 0xfb, 0xe8, 0xc4, 0x2a, 0x07, 0xc6, 0x3b,
    0x85, 0x0d, 0xf4, 0xbf, 0xb0, 0x6b, 0xfb, 0xce, 0x1d, 0xb4,
    0xbf, 0x63, 0x0b, 0x91, 0x67, 0xc4, 0xa3, 0x06, 0xa4, 0xaf,
    0x6c, 0xd3, 0xe5, 0x8b, 0x87, 0x4e, 0x64, 0x9c, 0xb1, 0xf3,
    0x70, 0x7c, 0x68, 0x43, 0x46, 0x13, 0x46, 0xee, 0x27, 0x75,
    0x12, 0x45, 0x42, 0xde, 0xa5, 0x8d, 0xcf, 0xf7, 0x09, 0x87,
    0xa8, 0x80, 0x3d, 0xb6, 0x45, 0xee, 0x41, 0x2d, 0x7c, 0x45,
    0x01, 0x9d, 0xaa, 0x78, 0xa8, 0x10, 0xa4, 0xfd, 0xb5, 0x5f,
    0xee, 0x0f, 0x77, 0xba, 0x73, 0xff, 0x49, 0xdc, 0xfa, 0x39,
    0xd6, 0xa3, 0x6f, 0x25, 0xb9, 0x63, 0x2c, 0x92, 0xc5, 0xdf,
    0xfb, 0xba, 0x89, 0xf9, 0xfa, 0x94, 0x5b, 0x6f, 0x5a, 0x4d,
    0x1c, 0xe4, 0xc9, 0x10, 0xf9, 0xa0, 0xe8, 0xc4, 0xcb, 0x55,
    0x1a, 0xdb, 0x56, 0x5f, 0x8e, 0x91, 0x03, 0x23, 0xca, 0xb0,
    0x1f, 0xef, 0xb8, 0x6c, 0x13, 0x5a, 0x99, 0x25, 0xf0, 0x49,
    0xa9, 0x5a, 0x45, 0xf7, 0xfd, 0x1a, 0xc2, 0x71, 0x06, 0xe3,
    0x2d, 0x25, 0x64, 0xb0, 0x52, 0x12, 0x03, 0x62, 0xc7, 0xb6,
    0xf9, 0xdc, 0x1f, 0x78, 0xff, 0x8b, 0xfa, 0xde, 0x7f, 0x71,
    0xa6, 0x35, 0x3e, 0xac, 0x20, 0x54, 0x94, 0xa7, 0x2e, 0x9d,
    0x47, 0x17, 0x4b, 0xad, 0x92, 0xb3, 0x14, 0x26, 0x8c, 0x5a,
    0xd0, 0x16, 0x4b, 0x22, 0xe9, 0x0c, 0x79, 0x6b, 0x8e, 0xac,
    0x0d, 0x12, 0xf5, 0x66, 0x8e, 0x82, 0x1a, 0x44, 0xf3, 0xe9,
    0x56, 0x5a, 0xcd, 0x1c, 0x1b, 0x81, 0x7b, 0x63, 0x59, 0xfe,
    0xc8, 0xc0, 0xe3, 0xda, 0x16, 0x6b, 0x6f, 0x0d, 0xba, 0x0e,
    0x47, 0x12, 0x86, 0x9e, 0xf0, 0x3b, 0x4d, 0x87, 0x3b, 0xf2,
    0x75, 0x73, 0x2d, 0xdf, 0xca, 0x76, 0x0b, 0xbd, 0xe7, 0xb7,
    0x74, 0x24, 0xf3, 0xc6, 0xe6, 0x75, 0x3f, 0x8b, 0x6a, 0xd9,
    0xad, 0xed, 0xc0, 0x70, 0x04, 0x1e, 0x0b, 0x8e, 0x8b, 0x7f,
    0xea, 0xbc, 0x39, 0x6b, 0x8a, 0x44, 0xa6, 0x9a, 0x2d, 0x0d,
    0x8c, 0x21, 0x60, 0x09, 0xd2, 0x4a, 0xe0, 0x62, 0xcf, 0xfa,
    0xe8, 0x9b, 0x35, 0x6f, 0x23, 0x2f, 0xb5, 0x65, 0x08, 0x60,
    0x92, 0x15, 0xd0, 0x5b, 0x63, 0xcc, 0x65, 0x05, 0xd1, 0xef,
    0x0f, 0x7e, 0x1b, 0xb3, 0x8e, 0xc6, 0x12, 0x85, 0xc9, 0x82,
    0x53, 0x79, 0x2e, 0x80, 0x5f, 0x0c, 0x7b, 0xc7, 0x1c, 0x83,
    0x41, 0x06, 0xd8, 0x41, 0xc9, 0xe7, 0xb9, 0x4b, 0xa1, 0x61,
    0xc6, 0x86, 0x67, 0xf5, 0x10, 0xf7, 0x34, 0x0d, 0x39, 0x9e,
    0x2b, 0x5f, 0x19, 0x06, 0x02, 0xa5, 0x02, 0x23, 0x71, 0xc2,
    0x12, 0x65, 0xcc, 0x81, 0x06, 0xfd, 0x8d, 0x09, 0x68, 0x37,
    0x06, 0x3b, 0xff, 0xc4, 0x24, 0xb3, 0x1f, 0xd6, 0xe6, 0x8f,
    0x9c, 0x74, 0x2c, 0x5e, 0xc5, 0xf4, 0xe9, 0xeb, 0xca, 0xd3,
    0x04, 0x5b, 0x92, 0x9e, 0x5c, 0x1a, 0x1d, 0xa1, 0xa7, 0x34,
    0xd2, 0x05, 0xae, 0xdb, 0x3d, 0x71, 0x10, 0x6e, 0x30, 0xd9,
    0xa3, 0x44, 0xa0, 0xbd, 0x9e, 0x7b, 0xb5, 0x12, 0x8a, 0x12,
    0x07, 0x60, 0xd7, 0x1f, 0x92, 0xe6, 0xfe, 0x04, 0xa9, 0x3e,
    0x62, 0x64, 0x00, 0x5f, 0x7c, 0x7b, 0x34, 0x09, 0xeb, 0x4a,
    0x18, 0x9e, 0x77, 0x72, 0x3a, 0x31, 0x1a, 0x62, 0x2a, 0xb5,
    0xcb, 0x4e, 0x53, 0xce, 0xad, 0x8b, 0x5a, 0x20, 0x4f, 0xd7,
    0x3e, 0x16, 0xf8, 0x10, 0xe2, 0xae, 0xbd, 0x3f, 0x02, 0xa9,
    0x18, 0xa0, 0x01, 0x18, 0x84, 0x95, 0x22, 0x2e, 0x93, 0x76,
    0x44, 0x4e, 0x11, 0x7b, 0x03, 0x51, 0x50, 0x19, 0x79, 0xe7,
    0xbb, 0x5c, 0x7b, 0xca, 0x74, 0xb4, 0x25, 0x26, 0xdb, 0x66,
    0xaa, 0x0b, 0x21, 0x07, 0xfb, 0x7a, 0x96, 0x10, 0x7d, 0x99,
    0xa9, 0x16, 0xcb, 0x0e, 0xba, 0x63, 0xab, 0x95, 0xfc, 0x5a,
    0xbe, 0xa6, 0x7f, 0xd8, 0xb4, 0xcd, 0x7c, 0xc5, 0xd0, 0xb1,
    0x1b, 0x48, 0x40, 0xfb, 0xe6, 0x2f, 0x2b, 0x94, 0xfe, 0x68,
    0xa2, 0xc4, 0x36, 0xd9, 0xcd, 0xc1, 0x93, 0x6d, 0xef, 0x39,
    0x5e, 0x43, 0x30, 0x5a, 0x2e, 0x66, 0xb6, 0xf2, 0xed, 0x9a,
    0x8d, 0x12, 0xdf, 0x5c, 0xae, 0xad, 0x16, 0x12, 0x7e, 0x81,
    0x82, 0x91, 0x7d, 0x2b, 0x12, 0xe9, 0x96, 0xb8, 0xb7, 0x42,
    0xcb, 0x1f, 0xf8, 0xd1, 0xfd, 0x83, 0x7a, 0xe4, 0x36, 0x1d,
    0x04, 0x27, 0x4c, 0xe5, 0xbd, 0x75, 0x24, 0xf7, 0xbd, 0xb6,
    0x6a, 0x68, 0x4e, 0x2c, 0x1b, 0x56, 0x3e, 0x60, 0xa4, 0x42,
    0xca, 0x7a, 0x54, 0xe5, 0x06, 0xe3, 0xda, 0x05, 0xf7, 0x77,
    0x36, 0x8b, 0x81, 0x26, 0x99, 0x92, 0x42, 0xda, 0x45, 0xb1,
    0xfe, 0x4b,
};
static const int sizeof_bench_dilithium_level2_pubkey =
    sizeof(bench_dilithium_level2_pubkey);

#endif /* !WOLFSSL_DILITHIUM_NO_VERIFY */

#ifndef WOLFSSL_DILITHIUM_NO_SIGN

/* raw private key without ASN1 syntax from
 * ./certs/dilithium/bench_dilithium_level3_key.der */
static const unsigned char bench_dilithium_level3_key[] = {
    0x15, 0xc9, 0xe5, 0x53, 0x2f, 0xd8, 0x1f, 0xb4, 0xa3, 0x9f,
    0xae, 0xad, 0xb3, 0x10, 0xd0, 0x72, 0x69, 0xd3, 0x02, 0xf3,
    0xdf, 0x67, 0x5a, 0x31, 0x52, 0x19, 0xca, 0x39, 0x27, 0x77,
    0x61, 0x6d, 0x72, 0xdd, 0x85, 0x06, 0xf6, 0x94, 0x0a, 0x57,
    0x52, 0xcd, 0xac, 0x83, 0x4a, 0xe5, 0xbe, 0xa4, 0x30, 0x79,
    0x9e, 0xc6, 0xd6, 0x04, 0xc8, 0x73, 0xdc, 0x5e, 0x41, 0x75,
    0x2f, 0xac, 0x76, 0x57, 0x03, 0x08, 0x46, 0xcb, 0xaf, 0x4c,
    0x6a, 0x4f, 0x20, 0x18, 0xb3, 0x2e, 0x11, 0x54, 0xb5, 0x94,
    0xe6, 0x6f, 0x76, 0xf6, 0xb9, 0x73, 0x9a, 0x07, 0x73, 0xe8,
    0x90, 0xd1, 0x04, 0xda, 0xc5, 0x97, 0xb9, 0x52, 0x51, 0xc8,
    0xc9, 0xcc, 0x87, 0x29, 0xa1, 0xde, 0x79, 0x9b, 0xf8, 0x7f,
    0x80, 0x3f, 0xfd, 0xb3, 0x24, 0xa5, 0xba, 0xf5, 0xd6, 0xd4,
    0x07, 0xbd, 0xa7, 0x1b, 0xd0, 0xe1, 0xd0, 0x43, 0x14, 0x52,
    0x27, 0x03, 0x33, 0x76, 0x00, 0x67, 0x30, 0x23, 0x76, 0x34,
    0x72, 0x02, 0x41, 0x62, 0x12, 0x43, 0x86, 0x30, 0x18, 0x28,
    0x46, 0x27, 0x45, 0x20, 0x88, 0x33, 0x54, 0x10, 0x03, 0x81,
    0x44, 0x50, 0x06, 0x44, 0x56, 0x30, 0x37, 0x38, 0x38, 0x46,
    0x03, 0x85, 0x01, 0x86, 0x43, 0x80, 0x78, 0x28, 0x83, 0x55,
    0x37, 0x44, 0x80, 0x12, 0x17, 0x51, 0x78, 0x46, 0x22, 0x01,
    0x53, 0x54, 0x63, 0x87, 0x77, 0x38, 0x11, 0x81, 0x43, 0x30,
    0x15, 0x47, 0x66, 0x11, 0x40, 0x65, 0x70, 0x56, 0x62, 0x28,
    0x21, 0x65, 0x30, 0x45, 0x63, 0x53, 0x31, 0x80, 0x81, 0x71,
    0x23, 0x62, 0x85, 0x03, 0x07, 0x56, 0x16, 0x28, 0x18, 0x35,
    0x07, 0x38, 0x60, 0x68, 0x17, 0x30, 0x15, 0x20, 0x04, 0x13,
    0x13, 0x61, 0x51, 0x58, 0x00, 0x37, 0x51, 0x58, 0x14, 0x06,
    0x12, 0x55, 0x13, 0x46, 0x76, 0x05, 0x51, 0x87, 0x32, 0x62,
    0x50, 0x41, 0x88, 0x24, 0x50, 0x31, 0x65, 0x36, 0x31, 0x02,
    0x75, 0x35, 0x78, 0x27, 0x36, 0x08, 0x01, 0x77, 0x22, 0x77,
    0x30, 0x80, 0x11, 0x21, 0x28, 0x26, 0x68, 0x27, 0x13, 0x70,
    0x50, 0x44, 0x88, 0x20, 0x50, 0x67, 0x65, 0x74, 0x17, 0x46,
    0x50, 0x16, 0x42, 0x75, 0x35, 0x12, 0x60, 0x12, 0x17, 0x13,
    0x36, 0x72, 0x04, 0x77, 0x07, 0x55, 0x20, 0x27, 0x15, 0x02,
    0x25, 0x12, 0x57, 0x71, 0x37, 0x45, 0x43, 0x34, 0x40, 0x31,
    0x78, 0x50, 0x31, 0x28, 0x17, 0x84, 0x87, 0x43, 0x25, 0x75,
    0x58, 0x05, 0x61, 0x56, 0x41, 0x44, 0x57, 0x67, 0x85, 0x54,
    0x00, 0x88, 0x88, 0x50, 0x68, 0x11, 0x14, 0x42, 0x08, 0x74,
    0x73, 0x00, 0x38, 0x08, 0x45, 0x28, 0x62, 0x43, 0x36, 0x20,
    0x30, 0x10, 0x87, 0x83, 0x67, 0x62, 0x02, 0x48, 0x46, 0x50,
    0x08, 0x08, 0x41, 0x43, 0x78, 0x22, 0x65, 0x87, 0x43, 0x84,
    0x25, 0x36, 0x58, 0x64, 0x30, 0x10, 0x20, 0x68, 0x82, 0x47,
    0x60, 0x31, 0x76, 0x68, 0x74, 0x68, 0x75, 0x61, 0x16, 0x26,
    0x82, 0x50, 0x32, 0x61, 0x41, 0x22, 0x38, 0x20, 0x86, 0x75,
    0x74, 0x00, 0x77, 0x12, 0x81, 0x35, 0x51, 0x78, 0x88, 0x64,
    0x82, 0x00, 0x41, 0x55, 0x62, 0x87, 0x51, 0x41, 0x74, 0x51,
    0x53, 0x27, 0x33, 0x84, 0x68, 0x86, 0x57, 0x60, 0x44, 0x30,
    0x22, 0x32, 0x10, 0x52, 0x22, 0x83, 0x48, 0x53, 0x66, 0x74,
    0x14, 0x52, 0x32, 0x71, 0x41, 0x08, 0x83, 0x67, 0x41, 0x38,
    0x46, 0x80, 0x88, 0x14, 0x84, 0x30, 0x85, 0x35, 0x46, 0x20,
    0x54, 0x84, 0x56, 0x84, 0x54, 0x82, 0x14, 0x11, 0x52, 0x07,
    0x86, 0x46, 0x05, 0x82, 0x26, 0x85, 0x75, 0x07, 0x88, 0x75,
    0x51, 0x17, 0x54, 0x32, 0x68, 0x66, 0x08, 0x23, 0x66, 0x06,
    0x42, 0x28, 0x00, 0x84, 0x27, 0x27, 0x43, 0x47, 0x12, 0x27,
    0x13, 0x15, 0x17, 0x74, 0x85, 0x14, 0x12, 0x62, 0x06, 0x47,
    0x17, 0x60, 0x00, 0x10, 0x85, 0x16, 0x55, 0x64, 0x46, 0x62,
    0x77, 0x05, 0x51, 0x23, 0x52, 0x37, 0x51, 0x78, 0x35, 0x66,
    0x14, 0x15, 0x78, 0x40, 0x16, 0x54, 0x67, 0x30, 0x61, 0x24,
    0x26, 0x86, 0x56, 0x83, 0x62, 0x78, 0x88, 0x83, 0x50, 0x06,
    0x13, 0x21, 0x33, 0x73, 0x16, 0x44, 0x86, 0x77, 0x65, 0x28,
    0x12, 0x40, 0x62, 0x54, 0x55, 0x84, 0x00, 0x11, 0x77, 0x38,
    0x71, 0x51, 0x38, 0x32, 0x33, 0x67, 0x15, 0x77, 0x24, 0x33,
    0x44, 0x11, 0x05, 0x65, 0x13, 0x03, 0x72, 0x63, 0x81, 0x58,
    0x08, 0x03, 0x34, 0x23, 0x61, 0x00, 0x02, 0x63, 0x86, 0x40,
    0x03, 0x71, 0x34, 0x27, 0x45, 0x10, 0x34, 0x26, 0x83, 0x28,
    0x31, 0x35, 0x26, 0x05, 0x58, 0x41, 0x11, 0x10, 0x65, 0x35,
    0x22, 0x42, 0x28, 0x88, 0x46, 0x06, 0x57, 0x33, 0x88, 0x46,
    0x04, 0x86, 0x88, 0x88, 0x51, 0x74, 0x82, 0x27, 0x58, 0x14,
    0x11, 0x08, 0x13, 0x16, 0x61, 0x16, 0x14, 0x44, 0x83, 0x85,
    0x71, 0x44, 0x55, 0x82, 0x16, 0x62, 0x85, 0x05, 0x43, 0x41,
    0x73, 0x53, 0x60, 0x01, 0x80, 0x68, 0x33, 0x13, 0x43, 0x44,
    0x73, 0x36, 0x65, 0x35, 0x22, 0x26, 0x13, 0x31, 0x36, 0x83,
    0x30, 0x27, 0x15, 0x11, 0x54, 0x53, 0x24, 0x84, 0x75, 0x24,
    0x72, 0x78, 0x34, 0x24, 0x35, 0x80, 0x06, 0x38, 0x88, 0x11,
    0x41, 0x01, 0x34, 0x87, 0x77, 0x20, 0x14, 0x50, 0x55, 0x12,
    0x17, 0x48, 0x87, 0x74, 0x58, 0x42, 0x31, 0x46, 0x36, 0x37,
    0x26, 0x50, 0x04, 0x75, 0x77, 0x15, 0x41, 0x53, 0x04, 0x04,
    0x26, 0x61, 0x65, 0x87, 0x55, 0x56, 0x07, 0x81, 0x28, 0x21,
    0x41, 0x61, 0x41, 0x50, 0x17, 0x47, 0x25, 0x50, 0x20, 0x83,
    0x46, 0x87, 0x18, 0x45, 0x40, 0x21, 0x06, 0x08, 0x12, 0x25,
    0x71, 0x13, 0x35, 0x55, 0x54, 0x61, 0x00, 0x52, 0x74, 0x78,
    0x13, 0x84, 0x55, 0x40, 0x14, 0x40, 0x78, 0x12, 0x88, 0x43,
    0x33, 0x24, 0x66, 0x88, 0x22, 0x44, 0x15, 0x37, 0x81, 0x27,
    0x84, 0x18, 0x28, 0x11, 0x58, 0x51, 0x71, 0x21, 0x02, 0x83,
    0x70, 0x48, 0x32, 0x46, 0x00, 0x70, 0x17, 0x30, 0x63, 0x21,
    0x46, 0x60, 0x50, 0x72, 0x77, 0x45, 0x83, 0x75, 0x26, 0x31,
    0x47, 0x34, 0x47, 0x84, 0x87, 0x63, 0x22, 0x83, 0x21, 0x10,
    0x21, 0x51, 0x47, 0x46, 0x31, 0x06, 0x57, 0x82, 0x65, 0x24,
    0x61, 0x66, 0x24, 0x68, 0x14, 0x03, 0x43, 0x41, 0x04, 0x14,
    0x47, 0x61, 0x57, 0x87, 0x43, 0x83, 0x43, 0x25, 0x87, 0x36,
    0x72, 0x51, 0x38, 0x51, 0x54, 0x54, 0x84, 0x40, 0x15, 0x30,
    0x35, 0x34, 0x43, 0x61, 0x63, 0x42, 0x77, 0x31, 0x42, 0x06,
    0x61, 0x03, 0x01, 0x41, 0x08, 0x84, 0x02, 0x65, 0x04, 0x72,
    0x32, 0x00, 0x21, 0x10, 0x54, 0x73, 0x04, 0x42, 0x48, 0x11,
    0x74, 0x18, 0x63, 0x73, 0x28, 0x61, 0x36, 0x80, 0x20, 0x86,
    0x24, 0x42, 0x16, 0x11, 0x71, 0x83, 0x78, 0x38, 0x82, 0x47,
    0x67, 0x18, 0x56, 0x86, 0x85, 0x66, 0x18, 0x24, 0x50, 0x74,
    0x72, 0x02, 0x66, 0x83, 0x63, 0x08, 0x25, 0x32, 0x15, 0x78,
    0x33, 0x08, 0x34, 0x44, 0x08, 0x28, 0x10, 0x25, 0x40, 0x11,
    0x04, 0x76, 0x60, 0x16, 0x65, 0x16, 0x13, 0x30, 0x53, 0x14,
    0x77, 0x06, 0x06, 0x88, 0x64, 0x47, 0x08, 0x23, 0x11, 0x56,
    0x46, 0x61, 0x48, 0x64, 0x73, 0x66, 0x07, 0x65, 0x41, 0x24,
    0x67, 0x45, 0x42, 0x18, 0x62, 0x01, 0x70, 0x88, 0x03, 0x77,
    0x22, 0x85, 0x77, 0x02, 0x85, 0x03, 0x65, 0x15, 0x57, 0x51,
    0x28, 0x72, 0x53, 0x32, 0x05, 0x58, 0x84, 0x54, 0x03, 0x81,
    0x63, 0x23, 0x38, 0x27, 0x01, 0x85, 0x61, 0x12, 0x28, 0x62,
    0x22, 0x67, 0x56, 0x66, 0x63, 0x08, 0x74, 0x63, 0x21, 0x01,
    0x46, 0x10, 0x08, 0x18, 0x07, 0x86, 0x47, 0x70, 0x50, 0x25,
    0x45, 0x06, 0x55, 0x88, 0x46, 0x11, 0x23, 0x84, 0x70, 0x02,
    0x24, 0x88, 0x52, 0x60, 0x12, 0x72, 0x63, 0x05, 0x81, 0x21,
    0x26, 0x07, 0x64, 0x03, 0x56, 0x48, 0x27, 0x04, 0x38, 0x86,
    0x25, 0x65, 0x21, 0x25, 0x77, 0x21, 0x62, 0x28, 0x82, 0x71,
    0x85, 0x73, 0x78, 0x24, 0x78, 0x51, 0x61, 0x02, 0x81, 0x14,
    0x67, 0x61, 0x08, 0x88, 0x31, 0x77, 0x06, 0x24, 0x45, 0x13,
    0x67, 0x67, 0x54, 0x67, 0x00, 0x12, 0x62, 0x54, 0x11, 0x27,
    0x51, 0x48, 0x07, 0x33, 0x01, 0x24, 0x04, 0x64, 0x11, 0x83,
    0x18, 0x52, 0x55, 0x23, 0x24, 0x58, 0x53, 0x78, 0x30, 0x43,
    0x31, 0x76, 0x62, 0x01, 0x08, 0x73, 0x21, 0x32, 0x12, 0x78,
    0x22, 0x68, 0x33, 0x45, 0x33, 0x73, 0x02, 0x74, 0x21, 0x81,
    0x02, 0x16, 0x54, 0x31, 0x55, 0x76, 0x25, 0x76, 0x41, 0x36,
    0x75, 0x22, 0x78, 0x16, 0x60, 0x48, 0x58, 0x28, 0x83, 0x50,
    0x88, 0x66, 0x72, 0x70, 0x21, 0x21, 0x24, 0x16, 0x62, 0x57,
    0x20, 0x13, 0x80, 0x61, 0x15, 0x45, 0x42, 0x86, 0x00, 0x25,
    0x77, 0x58, 0x84, 0x01, 0x66, 0x16, 0x46, 0x56, 0x68, 0x57,
    0x12, 0x20, 0x75, 0x60, 0x41, 0x85, 0x02, 0x88, 0x12, 0x68,
    0x20, 0x02, 0x41, 0x18, 0x87, 0x13, 0x17, 0x33, 0x74, 0x11,
    0x08, 0x37, 0x47, 0x08, 0x31, 0x67, 0x08, 0x50, 0x61, 0x54,
    0x56, 0x71, 0x63, 0x26, 0x85, 0x22, 0x07, 0x87, 0x71, 0x28,
    0x20, 0x47, 0x48, 0x66, 0x54, 0x38, 0x03, 0x41, 0x38, 0x21,
    0x70, 0x50, 0x66, 0x53, 0x56, 0x70, 0x74, 0x55, 0x70, 0x28,
    0x52, 0x01, 0x42, 0x65, 0x53, 0x73, 0x32, 0x33, 0x67, 0x42,
    0x67, 0x85, 0x18, 0x45, 0x12, 0x37, 0x58, 0x82, 0x13, 0x73,
    0x78, 0x77, 0x03, 0x42, 0x04, 0x65, 0x55, 0x66, 0x07, 0x25,
    0x07, 0x37, 0x40, 0x78, 0x66, 0x71, 0x11, 0x21, 0x43, 0x25,
    0x87, 0x40, 0x58, 0x63, 0x33, 0x43, 0x52, 0x10, 0x31, 0x53,
    0x56, 0x48, 0x05, 0x55, 0x77, 0x77, 0x26, 0x87, 0x28, 0x43,
    0x61, 0x46, 0x11, 0x76, 0x82, 0x50, 0x42, 0x04, 0x32, 0x88,
    0x18, 0x66, 0x16, 0x36, 0x64, 0x41, 0x38, 0x17, 0x55, 0x43,
    0x06, 0x25, 0x80, 0x27, 0x21, 0x16, 0x81, 0x22, 0x64, 0x60,
    0x38, 0x16, 0x82, 0x40, 0x72, 0x34, 0x73, 0x52, 0x61, 0x85,
    0x11, 0x16, 0x00, 0x25, 0x03, 0x30, 0x06, 0x80, 0x21, 0x56,
    0x64, 0x52, 0x23, 0x26, 0x37, 0x75, 0x73, 0x65, 0x53, 0x27,
    0x37, 0x47, 0x56, 0x76, 0x80, 0x38, 0x53, 0x62, 0x14, 0x24,
    0x64, 0x03, 0x66, 0x21, 0x72, 0x16, 0x36, 0x34, 0x11, 0x65,
    0x61, 0x62, 0x86, 0x02, 0x83, 0x27, 0x80, 0x82, 0x70, 0x72,
    0x52, 0x60, 0x20, 0x87, 0x58, 0x58, 0x14, 0x38, 0x47, 0x03,
    0x10, 0x72, 0x60, 0x48, 0x02, 0x01, 0x17, 0x21, 0x61, 0x62,
    0x38, 0x64, 0x27, 0x53, 0x57, 0x13, 0x68, 0x18, 0x26, 0x62,
    0x43, 0x42, 0x21, 0x85, 0x70, 0x23, 0x58, 0x13, 0x72, 0x04,
    0x04, 0x08, 0x05, 0x82, 0x26, 0x18, 0x82, 0x47, 0x87, 0x71,
    0x32, 0x28, 0x68, 0x25, 0x87, 0x24, 0x06, 0x74, 0x41, 0x44,
    0x08, 0x64, 0x68, 0x30, 0x24, 0x44, 0x21, 0x73, 0x03, 0x45,
    0x70, 0x41, 0x06, 0x78, 0x38, 0x33, 0x88, 0x13, 0x31, 0x14,
    0x18, 0x17, 0x45, 0x06, 0x26, 0x67, 0x66, 0x73, 0x82, 0x56,
    0x66, 0x88, 0x70, 0x22, 0x55, 0x47, 0x27, 0x50, 0x86, 0x55,
    0x53, 0x00, 0x28, 0x55, 0x40, 0x62, 0xe9, 0x37, 0x65, 0xe1,
    0x30, 0x48, 0x6b, 0x35, 0x76, 0x96, 0x05, 0x21, 0xce, 0xed,
    0x46, 0xae, 0x7e, 0x6d, 0xc9, 0xf1, 0xc9, 0xb3, 0x7a, 0xa7,
    0xde, 0xa7, 0x62, 0x18, 0x11, 0xc0, 0xd8, 0xd0, 0x17, 0x0f,
    0x38, 0xaf, 0x0e, 0x3d, 0xaf, 0xe6, 0x63, 0xb0, 0xc4, 0x68,
    0x4e, 0x29, 0xa4, 0xf4, 0x20, 0x22, 0xbc, 0x82, 0x15, 0x1d,
    0x08, 0x39, 0x18, 0xfe, 0x69, 0x55, 0x06, 0x3d, 0xf4, 0xa3,
    0xe7, 0x29, 0x23, 0xa4, 0xd9, 0xa4, 0x22, 0x06, 0x2d, 0x5f,
    0x22, 0xb3, 0x9b, 0x1c, 0xb6, 0x3e, 0xf3, 0xf4, 0x8a, 0xb3,
    0x35, 0x18, 0x4c, 0x1f, 0xaf, 0xd4, 0xcf, 0x5b, 0x9b, 0xa7,
    0xf8, 0xd2, 0x86, 0x71, 0x8e, 0x64, 0x96, 0xd1, 0x6e, 0xad,
    0xd2, 0x7e, 0x16, 0x5b, 0x38, 0x91, 0x0e, 0x40, 0xaa, 0x07,
    0x6a, 0x63, 0x2a, 0xc0, 0x5b, 0x14, 0x79, 0x52, 0xcb, 0x23,
    0x6e, 0x76, 0x95, 0xd0, 0x90, 0x6c, 0x18, 0xe7, 0x89, 0xee,
    0xb9, 0x7f, 0x33, 0x08, 0x35, 0x8f, 0xa3, 0xaa, 0xaa, 0x10,
    0x2f, 0x8b, 0xc9, 0x6c, 0x1d, 0x95, 0xb5, 0xb8, 0x54, 0x0d,
    0x67, 0x86, 0xd4, 0x5d, 0xae, 0x8f, 0x33, 0x20, 0xe2, 0x35,
    0xda, 0x71, 0x53, 0x24, 0xad, 0x16, 0x84, 0x2e, 0x98, 0xcd,
    0x00, 0xa2, 0x69, 0x6a, 0x12, 0x9a, 0x86, 0xf3, 0x9f, 0x18,
    0x6c, 0x9f, 0x24, 0xbe, 0xb3, 0xf4, 0x90, 0xb3, 0xc4, 0xa4,
    0x8b, 0xce, 0x88, 0x60, 0xa0, 0x91, 0xb8, 0x9a, 0x52, 0xe5,
    0xfe, 0x16, 0x6d, 0xff, 0xb3, 0xdc, 0x50, 0x79, 0xfe, 0x31,
    0x24, 0xd4, 0x59, 0x5f, 0xf9, 0xb4, 0x70, 0x0b, 0x15, 0x93,
    0xd9, 0xe9, 0x92, 0xb6, 0xf5, 0x80, 0x34, 0x63, 0x66, 0x78,
    0xcf, 0xa9, 0xce, 0x48, 0xbf, 0xbe, 0x9e, 0xfa, 0xdd, 0x7d,
    0xf4, 0x16, 0xe2, 0xd2, 0x98, 0x13, 0xe2, 0x76, 0xdd, 0x0a,
    0xc7, 0x2d, 0xe8, 0x88, 0x8e, 0x1a, 0xc0, 0xfc, 0xe8, 0x35,
    0xaf, 0x5d, 0xe2, 0x4c, 0x96, 0x82, 0x4c, 0xe5, 0x89, 0x14,
    0xb8, 0x27, 0x39, 0xb5, 0x55, 0xc5, 0xa5, 0x8a, 0x01, 0xcc,
    0xfd, 0xbd, 0xa9, 0xec, 0xae, 0xc0, 0xe7, 0xd7, 0xf8, 0x11,
    0x84, 0x35, 0x99, 0x26, 0xb6, 0xc6, 0xf7, 0x35, 0xe0, 0x93,
    0xd8, 0xd7, 0xbf, 0xc0, 0xc8, 0x44, 0xfd, 0x46, 0xf5, 0xb7,
    0xc5, 0x5a, 0x75, 0xd3, 0xc7, 0xfa, 0xf4, 0xe1, 0xc0, 0x84,
    0x5e, 0x31, 0xfe, 0x69, 0x80, 0x5a, 0xe5, 0x4b, 0x9b, 0x5b,
    0xa4, 0x5c, 0x23, 0xaa, 0x85, 0xc9, 0x9a, 0xbd, 0x71, 0x49,
    0x11, 0x30, 0x8b, 0x81, 0xa1, 0xdd, 0xf8, 0xb8, 0x74, 0x91,
    0xe7, 0xf7, 0x82, 0x42, 0x70, 0x22, 0x95, 0xf0, 0xcc, 0x9f,
    0x02, 0x33, 0x0f, 0x08, 0x3b, 0x04, 0x31, 0xd7, 0x4f, 0x86,
    0x78, 0x49, 0xb9, 0x90, 0xf5, 0x8f, 0xec, 0x12, 0x84, 0x52,
    0x03, 0x1f, 0x64, 0x5e, 0xf0, 0x2a, 0xeb, 0x87, 0xa5, 0xec,
    0x95, 0x25, 0x64, 0x25, 0x49, 0x3b, 0x3c, 0x30, 0xed, 0x3b,
    0xe9, 0x36, 0xfd, 0xae, 0xa6, 0x26, 0xd3, 0x45, 0xbc, 0x1b,
    0x78, 0x5f, 0xce, 0x27, 0x45, 0x1c, 0xd5, 0xf9, 0xa7, 0xda,
    0x62, 0xe6, 0x7e, 0xd3, 0xbb, 0xd8, 0x0a, 0xfd, 0xf5, 0xa5,
    0x31, 0x09, 0x6e, 0x40, 0xe8, 0xcf, 0xc1, 0x42, 0x8e, 0x2e,
    0x75, 0x65, 0xaa, 0x91, 0x6f, 0xc7, 0x75, 0x3a, 0x1e, 0x40,
    0x99, 0x71, 0x5e, 0x00, 0xae, 0x07, 0xad, 0x43, 0x49, 0xdd,
    0x6d, 0x36, 0xe3, 0xa8, 0xdf, 0x2c, 0x39, 0xa2, 0x57, 0xd7,
    0x93, 0xa1, 0x16, 0x80, 0x89, 0xa6, 0x56, 0x69, 0x75, 0xea,
    0xb8, 0xb2, 0x43, 0x0c, 0xdf, 0x46, 0x05, 0x9a, 0x39, 0x08,
    0x3b, 0xb6, 0x76, 0xe3, 0x5b, 0x98, 0x5b, 0x48, 0xc0, 0x11,
    0x14, 0x6f, 0xcd, 0xb7, 0xaa, 0x08, 0x1e, 0x53, 0x9b, 0x94,
    0x9d, 0xa2, 0xe6, 0x99, 0xcb, 0x1c, 0xb4, 0xbf, 0x55, 0x84,
    0x12, 0xc9, 0xf1, 0xf0, 0x94, 0xd9, 0x7d, 0x61, 0xa9, 0xe7,
    0xe6, 0xc1, 0xe2, 0xca, 0x6b, 0x36, 0x80, 0x72, 0x31, 0x79,
    0xbf, 0xe7, 0x3e, 0x99, 0x9e, 0xd5, 0x59, 0xd4, 0x97, 0x14,
    0xd5, 0xfa, 0x93, 0x37, 0x8a, 0x65, 0xa5, 0xb6, 0x4e, 0xba,
    0xb3, 0x84, 0xf2, 0xc1, 0x55, 0xb6, 0x94, 0x31, 0x30, 0xe7,
    0xb2, 0x71, 0x4e, 0xc6, 0x21, 0x50, 0xf3, 0xcf, 0x7c, 0xbc,
    0x26, 0xb7, 0x20, 0xcb, 0x2d, 0x9e, 0x55, 0x23, 0x7c, 0xf0,
    0x97, 0x16, 0x57, 0x5b, 0xcc, 0xc5, 0x48, 0xc9, 0xc8, 0xee,
    0x1e, 0x11, 0x6b, 0x72, 0x3b, 0x29, 0x71, 0xa4, 0xed, 0x08,
    0x6c, 0x38, 0xc6, 0x2e, 0x64, 0x3b, 0x16, 0xd8, 0x4d, 0x19,
    0xe8, 0x94, 0xd3, 0xd5, 0xb4, 0x18, 0xb4, 0x03, 0x24, 0x62,
    0xe7, 0x44, 0x5e, 0x09, 0x60, 0xc6, 0xa9, 0xa6, 0xca, 0xbe,
    0x83, 0xe5, 0xf1, 0xbd, 0x04, 0x22, 0x4b, 0x1b, 0x08, 0x0b,
    0xa6, 0x20, 0x95, 0xf2, 0x78, 0x8c, 0x3e, 0x73, 0x03, 0x7b,
    0x75, 0x2c, 0xe5, 0x72, 0xec, 0xc9, 0x25, 0x06, 0x6b, 0x3a,
    0x5e, 0x0e, 0x96, 0xd0, 0xe3, 0x85, 0xb0, 0xb5, 0x6a, 0x83,
    0x40, 0x41, 0x94, 0xce, 0xa1, 0x07, 0x79, 0x07, 0xe2, 0x50,
    0xa4, 0xde, 0x7d, 0x64, 0x2f, 0x7e, 0x43, 0xd5, 0x72, 0xd1,
    0xa7, 0xb9, 0x76, 0xa3, 0xfc, 0x25, 0x33, 0xd7, 0x95, 0xb5,
    0xd9, 0x94, 0x93, 0x55, 0xaf, 0x04, 0x86, 0x4a, 0xfc, 0x2f,
    0x5f, 0x3d, 0x34, 0x86, 0xf2, 0x9a, 0x31, 0x4c, 0xc9, 0xad,
    0x08, 0xa5, 0x03, 0x91, 0x8a, 0x7e, 0x46, 0xc9, 0x44, 0x61,
    0x11, 0x59, 0x4f, 0xbb, 0x70, 0xf9, 0x9d, 0x3e, 0x6d, 0x53,
    0xb4, 0x16, 0x28, 0xd3, 0x67, 0x52, 0x14, 0xad, 0xba, 0xb1,
    0x21, 0xaf, 0x84, 0x18, 0xc9, 0x37, 0x78, 0xb3, 0x78, 0x92,
    0x95, 0xad, 0x1b, 0xc0, 0x70, 0xe7, 0xe9, 0x06, 0x02, 0xed,
    0x6c, 0x99, 0x4e, 0x43, 0xc0, 0xa4, 0x6f, 0x23, 0xa8, 0x02,
    0xc4, 0xbd, 0xc0, 0x16, 0xc4, 0xed, 0xe0, 0xe1, 0x56, 0x06,
    0x3f, 0xf4, 0x77, 0x12, 0x72, 0x52, 0x04, 0xe8, 0xe4, 0x26,
    0xe5, 0x01, 0x47, 0x5b, 0x8a, 0xca, 0x07, 0x3b, 0xc9, 0xb1,
    0x42, 0x8f, 0x7d, 0x64, 0x7d, 0x5d, 0x6a, 0x95, 0xde, 0x4d,
    0x4b, 0xd3, 0xfa, 0xcf, 0xf0, 0x25, 0x27, 0x96, 0x48, 0xb6,
    0xcc, 0x68, 0x29, 0x37, 0x95, 0xcd, 0x36, 0xb7, 0xb0, 0xd6,
    0xf1, 0xfc, 0x4f, 0xe9, 0xa8, 0x6b, 0x9d, 0x75, 0xc7, 0x9b,
    0x19, 0xaf, 0xbb, 0x8a, 0xaf, 0x4b, 0xb8, 0xe2, 0xeb, 0x8d,
    0xd9, 0xf5, 0x75, 0xc5, 0xc8, 0x0b, 0xf2, 0x1c, 0xf9, 0x9e,
    0xc7, 0x4d, 0x7c, 0x71, 0x47, 0xbd, 0x57, 0x7e, 0xe6, 0x59,
    0xca, 0x8c, 0xf2, 0x0c, 0x47, 0x4a, 0x90, 0xa7, 0xf5, 0xb8,
    0xb2, 0x43, 0x97, 0xdb, 0xbe, 0x76, 0x37, 0x29, 0x36, 0x40,
    0xaa, 0x7a, 0x81, 0xf0, 0xa0, 0xd0, 0x81, 0x39, 0x88, 0xf0,
    0x23, 0xb0, 0xa4, 0xbe, 0x5e, 0xd8, 0x33, 0x98, 0x5d, 0x9d,
    0xb5, 0xd4, 0x1c, 0x00, 0xe2, 0x30, 0xb8, 0x68, 0x58, 0x65,
    0x30, 0x94, 0x3d, 0xf2, 0x75, 0x0c, 0x8e, 0x3b, 0xee, 0x9b,
    0xce, 0x6c, 0x67, 0x68, 0x54, 0x86, 0x7d, 0x27, 0x2a, 0x2f,
    0xf7, 0x25, 0xff, 0x22, 0x1e, 0x74, 0xbd, 0x72, 0x11, 0xf4,
    0x47, 0x8e, 0x2f, 0x0d, 0xb9, 0x31, 0xac, 0x5c, 0x1d, 0xa0,
    0x11, 0xea, 0x16, 0x24, 0x86, 0x76, 0xbd, 0xa3, 0x41, 0x7f,
    0x00, 0xe6, 0xe2, 0x86, 0x93, 0xff, 0x02, 0x07, 0xce, 0x49,
    0xe4, 0xaf, 0x00, 0x9b, 0x15, 0xa6, 0x05, 0xf7, 0x54, 0xd1,
    0xbb, 0xa7, 0x09, 0x67, 0xe6, 0x99, 0xf9, 0x23, 0xe6, 0xaa,
    0x6f, 0xcb, 0xe1, 0xc1, 0xac, 0x7b, 0x98, 0xa9, 0x14, 0x43,
    0x55, 0x22, 0x2c, 0x7a, 0x4a, 0x4a, 0x63, 0xc1, 0xfe, 0x5c,
    0xca, 0xf4, 0x91, 0x3b, 0x6f, 0xf8, 0x7e, 0x2a, 0xa1, 0x4a,
    0xc3, 0x16, 0x1c, 0x1d, 0x53, 0x7d, 0x0e, 0x77, 0x0d, 0x72,
    0x07, 0x78, 0xea, 0xce, 0xe4, 0x0c, 0xf7, 0xce, 0xa0, 0xef,
    0xa1, 0xdb, 0x6b, 0x5f, 0xfd, 0xeb, 0x68, 0xc7, 0x76, 0xfd,
    0x35, 0xd2, 0xcb, 0xa4, 0xf6, 0xe6, 0x6b, 0xdb, 0xe9, 0xd5,
    0x1e, 0x05, 0x8a, 0xba, 0xed, 0x77, 0x94, 0x36, 0x6c, 0x3c,
    0xe2, 0x23, 0xf8, 0x84, 0xa1, 0xe3, 0xcd, 0xfa, 0x1d, 0x31,
    0x52, 0x4d, 0xbc, 0x16, 0x31, 0x92, 0xd7, 0xbe, 0x2e, 0xd6,
    0x6d, 0x1d, 0x58, 0x4e, 0xd8, 0x06, 0x8f, 0xb3, 0xe6, 0x79,
    0x60, 0x92, 0x71, 0x1f, 0x72, 0x84, 0x55, 0x7b, 0xfa, 0xc8,
    0xcf, 0x20, 0x16, 0x2f, 0xc7, 0x13, 0x17, 0xd1, 0x2d, 0xd1,
    0x0d, 0x84, 0x48, 0x08, 0x69, 0xd1, 0x55, 0xb1, 0x08, 0xb6,
    0x17, 0x8c, 0x38, 0x31, 0xa4, 0x77, 0x73, 0xc0, 0xe9, 0xfc,
    0x5f, 0x8e, 0xb3, 0x74, 0x1f, 0xab, 0xcf, 0xf5, 0x26, 0x26,
    0x20, 0x80, 0xd8, 0x13, 0x42, 0xcf, 0xc7, 0x9d, 0xd6, 0x5b,
    0x1a, 0xfd, 0x46, 0x83, 0xba, 0xc1, 0xe5, 0x92, 0xe9, 0x27,
    0xa8, 0xa0, 0x36, 0xd5, 0x31, 0x75, 0x7b, 0x8f, 0x53, 0xf6,
    0xbd, 0x08, 0x1a, 0x86, 0x81, 0x83, 0x85, 0x07, 0x44, 0x3e,
    0xf9, 0x72, 0x47, 0xe0, 0xf1, 0xbe, 0x43, 0x6a, 0xc3, 0x00,
    0x94, 0xd3, 0x19, 0x81, 0xde, 0xf3, 0xfd, 0x57, 0x98, 0xdc,
    0x57, 0xfe, 0x9f, 0x4b, 0x38, 0x23, 0xad, 0xa8, 0xd4, 0x07,
    0x07, 0x5c, 0xca, 0x25, 0xb8, 0x77, 0x7e, 0x45, 0x01, 0x9b,
    0xd4, 0x45, 0x5b, 0x94, 0x47, 0x18, 0x35, 0x66, 0xad, 0x0a,
    0x97, 0x06, 0xc6, 0xa7, 0xaa, 0x50, 0xbf, 0x07, 0x90, 0xfe,
    0x50, 0x8d, 0xd9, 0x1f, 0xdd, 0x33, 0xa4, 0xa7, 0x23, 0x48,
    0xa3, 0xd6, 0x5d, 0xb8, 0x9e, 0x97, 0x22, 0x32, 0xd3, 0x8a,
    0xb0, 0x5e, 0xb3, 0xc9, 0x0b, 0x24, 0x09, 0x66, 0x2e, 0xea,
    0x94, 0x9c, 0x90, 0x4f, 0x3e, 0x93, 0xcf, 0x30, 0x3f, 0xb4,
    0xbe, 0x5e, 0x6c, 0xaf, 0x1a, 0xff, 0x00, 0xc7, 0x74, 0x2e,
    0x8b, 0x08, 0xe9, 0x22, 0x61, 0xc5, 0xd1, 0x21, 0x15, 0xa1,
    0xba, 0x37, 0xd2, 0x24, 0xfd, 0xa5, 0x63, 0x9a, 0x97, 0xfa,
    0xfe, 0xb2, 0xa5, 0x1b, 0x3b, 0xbd, 0xb7, 0xb3, 0x2f, 0x3d,
    0xf1, 0x5a, 0xf2, 0xf6, 0xe4, 0x12, 0xe4, 0x3a, 0x26, 0x3c,
    0x21, 0x5c, 0xd6, 0x83, 0x65, 0x26, 0x86, 0xcc, 0x47, 0x84,
    0xd7, 0x26, 0x31, 0x31, 0xcf, 0x1d, 0xd6, 0xc4, 0xa4, 0xf2,
    0xd4, 0x25, 0x54, 0x2b, 0x81, 0x00, 0x1d, 0xd8, 0xdf, 0x04,
    0xb8, 0x4b, 0xcf, 0xe5, 0x16, 0xf4, 0x4a, 0x17, 0xc5, 0xd8,
    0xd3, 0xdf, 0xe4, 0xb7, 0xd3, 0x98, 0xb6, 0x73, 0xa0, 0x37,
    0x67, 0xbb, 0x8b, 0xc3, 0xfc, 0xac, 0x6e, 0x6c, 0x0e, 0x5d,
    0x44, 0xb0, 0x9d, 0xf8, 0xae, 0x17, 0x9b, 0xf9, 0xcb, 0xe8,
    0xfe, 0xc1, 0x7b, 0x78, 0x16, 0xf6, 0x74, 0x04, 0x7d, 0x38,
    0x17, 0x36, 0x09, 0xe3, 0x73, 0xa1, 0x76, 0x78, 0x7c, 0x14,
    0xb3, 0x83, 0x91, 0x59, 0x27, 0xea, 0x8c, 0x69, 0xe6, 0xa5,
    0x21, 0xcd, 0x78, 0xc7, 0x26, 0xa2, 0xfb, 0xd4, 0xf3, 0xaf,
    0x3f, 0xcf, 0x51, 0x10, 0xcc, 0x4b, 0xdd, 0x14, 0xf4, 0xf3,
    0xb8, 0xea, 0x07, 0xa7, 0x76, 0xe7, 0xbe, 0xec, 0x01, 0xb5,
    0x1e, 0xdc, 0xc3, 0x55, 0x19, 0xb1, 0x16, 0x3f, 0xfe, 0xd4,
    0x15, 0x49, 0xaf, 0x04, 0x9d, 0x38, 0xdd, 0x86, 0x53, 0x2a,
    0x80, 0x62, 0x42, 0xb7, 0x98, 0x42, 0x38, 0xaf, 0x9d, 0x87,
    0xe2, 0x3f, 0xea, 0x7e, 0x0a, 0x35, 0xb8, 0xee, 0xa5, 0x48,
    0x09, 0x08, 0xc5, 0x0d, 0xae, 0x01, 0xd5, 0xec, 0x43, 0x29,
    0x3b, 0xfb, 0x78, 0xc4, 0x96, 0x01, 0x1c, 0x21, 0xf2, 0xc9,
    0x44, 0x68, 0x24, 0x66, 0x86, 0x96, 0xb8, 0xc8, 0xe9, 0xd0,
    0x38, 0x0e, 0x96, 0x4d, 0xcc, 0x45, 0xab, 0xe1, 0xca, 0x50,
    0x10, 0x20, 0x01, 0xbe, 0x89, 0xc0, 0x43, 0x84, 0xd8, 0x38,
    0x52, 0xc0, 0xaf, 0x4d, 0x6b, 0x99, 0x0b, 0xc0, 0xc2, 0x99,
    0x07, 0xc6, 0x78, 0xa8, 0xf7, 0x32, 0x84, 0x86, 0xc5, 0x1a,
    0x95, 0x81, 0xa6, 0x6a, 0x05, 0xa7, 0x9d, 0x81, 0x0e, 0x32,
    0x18, 0x11, 0x4a, 0x0f, 0xfc, 0x17, 0x9e, 0xf7, 0xbf, 0x54,
    0x82, 0xed, 0xba, 0x6f, 0xbd, 0x41, 0xc1, 0xca, 0x55, 0x6c,
    0xff, 0x32, 0x6b, 0xa2, 0x59, 0xae, 0xae, 0x92, 0xc1, 0xb5,
    0xa6, 0xfc, 0xaf, 0x09, 0x48, 0x57, 0xd6, 0xee, 0x38, 0x99,
    0xb4, 0xe3, 0x8f, 0xb7, 0xfc, 0x6a, 0x0a, 0x3b, 0x08, 0xe1,
    0x81, 0x46, 0x11, 0xeb, 0x4a, 0x98, 0x43, 0x16, 0x16, 0x1f,
    0x68, 0xdb, 0xb9, 0x71, 0x19, 0xfe, 0x8b, 0xe6, 0xb7, 0x8b,
    0xc1, 0x3b, 0x90, 0xc5, 0x89, 0x1d, 0xca, 0xd9, 0x19, 0x6c,
    0xe8, 0x01, 0xf4, 0x19, 0x50, 0x3e, 0x93, 0x84, 0xbf, 0xaa,
    0x9a, 0x3d, 0x20, 0x4c, 0x4e, 0x79, 0x83, 0xec, 0x46, 0x83,
    0x09, 0x00, 0xc3, 0x8a, 0xad, 0xd5, 0x2b, 0x08, 0xd1, 0x47,
    0xac, 0x96, 0x0e, 0x34, 0xf0, 0x89, 0x1a, 0x0f, 0xf2, 0x51,
    0x8d, 0x2c, 0xb5, 0xf2, 0xfe, 0x8c, 0xdc, 0xed, 0x41, 0x51,
    0x8c, 0x71, 0x12, 0x05, 0xec, 0x68, 0x21, 0x86, 0x94, 0xf4,
    0xfb, 0xfc, 0xaa, 0xc7, 0xc7, 0xbb, 0x74, 0xa2, 0x8b, 0x76,
    0x62, 0x1c, 0x64, 0x11, 0xa0, 0xd0, 0x5f, 0x46, 0x64, 0xd4,
    0x47, 0xbc, 0x8a, 0x5b, 0x2b, 0xc2, 0xc1, 0x88, 0xb2, 0x30,
    0xbd, 0x02, 0x17, 0x18, 0x0a, 0xd7, 0x9b, 0x3d, 0x91, 0xb9,
    0x2c, 0x83, 0x24, 0xb4, 0x8b, 0x9d, 0x02, 0xaf, 0xb2, 0x4e,
    0x57, 0xe1, 0xb0, 0xa2, 0xf3, 0x7c, 0xde, 0x15, 0xba, 0x60,
    0xbd, 0x80, 0xbe, 0x6d, 0x6f, 0x16, 0xb3, 0xb9, 0xb8, 0x6a,
    0x55, 0xb4, 0xad, 0xf1, 0x01, 0x63, 0x40, 0x01, 0xba, 0x5b,
    0x5d, 0x9a, 0xbc, 0xf0, 0x58, 0xa8, 0xf7, 0xbb, 0x8e, 0x91,
    0xa0, 0xfd, 0x8c, 0x49, 0x8f, 0x1a, 0xbb, 0x2a, 0x28, 0x0d,
    0x7a, 0xa6, 0xc2, 0xd7, 0x41, 0x16, 0xed, 0x61, 0x5d, 0xc4,
    0xe7, 0xcf, 0x2b, 0xb4, 0xb9, 0x10, 0x6f, 0x38, 0x42, 0x88,
    0x94, 0x6e, 0x75, 0x2c, 0x89, 0xac, 0xa0, 0xe9, 0x81, 0xec,
    0x2d, 0x62, 0xa3, 0xba, 0x3c, 0x40, 0xdb, 0x65, 0x56, 0x8e,
    0xc7, 0xd8, 0xb0, 0xd4, 0xf9, 0x04, 0x2b, 0x4c, 0x83, 0x20,
    0xbe, 0xad, 0xb8, 0x66, 0x1c, 0x20, 0x32, 0xb3, 0xf6, 0xf1,
    0xac, 0xa5, 0x8a, 0x72, 0x9a, 0x41, 0x1d, 0x6e, 0xa0, 0x16,
    0xe0, 0x0c, 0x39, 0xb6, 0x06, 0x96, 0x55, 0xb7, 0xda, 0x1c,
    0x54, 0x08, 0xf6, 0x30, 0x1b, 0xb6, 0x57, 0xca, 0x7d, 0xb0,
    0xdc, 0x9e, 0xfa, 0x5c, 0x38, 0x7f, 0xac, 0x37, 0x80, 0x26,
    0xba, 0xdc, 0x7a, 0x95, 0xe5, 0x7b, 0x90, 0xf3, 0x1a, 0xc7,
    0x31, 0x8e, 0x97, 0x07, 0x9a, 0xb8, 0xbe, 0xae, 0x16, 0x11,
    0x44, 0xb0, 0x01, 0xf5, 0xe8, 0x37, 0x1a, 0x67, 0xfe, 0x00,
    0x8f, 0xa1, 0xf5, 0x03, 0x7c, 0xed, 0xbf, 0x42, 0xf4, 0x78,
    0x2b, 0xfb, 0x9f, 0x8c, 0xb3, 0x63, 0x0b, 0x42, 0xbf, 0xae,
    0x8e, 0xf7, 0x6f, 0xb4, 0xb1, 0xe8, 0x75, 0x8c, 0xdf, 0x69,
    0xc6, 0xe1, 0x3a, 0x26, 0x05, 0x47, 0x03, 0x61, 0xfc, 0xc5,
    0xa9, 0xc1, 0x4f, 0x70, 0xce, 0x18, 0xbb, 0x01, 0xe6, 0x11,
    0xc9, 0xa7, 0x7e, 0x65, 0xb8, 0xdc, 0x61, 0x3d, 0x9b, 0x47,
    0x2e, 0x34, 0x16, 0xa1, 0x73, 0x61, 0x91, 0xed, 0x45, 0xe3,
    0x01, 0x26, 0xee, 0x16, 0x76, 0x0e, 0xb7, 0xa1, 0xc0, 0xb3,
    0xac, 0xf0, 0xa5, 0x3b, 0xf6, 0x64, 0x1b, 0x93, 0x94, 0x5c,
    0x8f, 0x4c, 0x25, 0x89, 0xa1, 0x92, 0x32, 0x50, 0x28, 0x03,
    0x8b, 0xff, 0xc4, 0xf6, 0x2a, 0xe8, 0xda, 0x8d, 0xfe, 0x49,
    0xb5, 0x33, 0x01, 0xca, 0x2d, 0x2d, 0x60, 0x33, 0xd6, 0x30,
    0x38, 0x8a, 0x1e, 0x38, 0x3d, 0x78, 0x11, 0xff, 0xef, 0x1c,
    0x82, 0x33, 0xbb, 0xfc, 0x95, 0xef, 0x79, 0xb0, 0x59, 0xbd,
    0x2c, 0xfd, 0x1c, 0x3f, 0x42, 0xda, 0xdf, 0xbd, 0x56, 0xf2,
    0xd6, 0xae, 0x2d, 0x23, 0x36, 0xed, 0xb1, 0x8d, 0x62, 0x58,
    0x71, 0x66, 0x21, 0xe0, 0x4d, 0xee, 0xf4, 0x16, 0x48, 0xa6,
    0xcf, 0x1a, 0x8a, 0xf0, 0x8a, 0xd1, 0x53, 0xf6, 0xe5, 0x4e,
    0x98, 0x9d, 0x7d, 0x6c, 0xd2, 0xdf, 0xb8, 0x2d, 0xa6, 0xe5,
    0x8a, 0xd6, 0xb5, 0xae, 0x61, 0x96, 0xfa, 0x6b, 0xca, 0x7f,
    0x08, 0xc2, 0x2b, 0x67, 0x30, 0x5e, 0x21, 0x3b, 0xa4, 0x84,
    0x95, 0xc6, 0x2f, 0x2c, 0x1f, 0xe2, 0x0e, 0x1a, 0xc3, 0x89,
    0x6a, 0x6a, 0xe7, 0x08, 0xf9, 0x74, 0xee, 0x4f, 0xcd, 0x5e,
    0xe8, 0xce, 0x55, 0x4d, 0x38, 0xed, 0x62, 0x35, 0xee, 0xfc,
    0x14, 0x56, 0xb9, 0xf0, 0xce, 0x29, 0x1c, 0x21, 0x40, 0x51,
    0xe4, 0x76, 0xe3, 0xa6, 0xd8, 0x3d, 0x54, 0x58, 0x51, 0xe5,
    0xf0, 0xdc, 0x50, 0x39, 0x43, 0x67, 0x44, 0x14, 0xcc, 0x6e,
    0x5a, 0xb1, 0x15, 0xec, 0xb4, 0x3e, 0x0e, 0xef, 0x8e, 0x72,
    0x6a, 0xdf, 0xba, 0x37, 0x27, 0x15, 0x62, 0xc3, 0xbd, 0xee,
    0x1d, 0xb1, 0x24, 0x2f, 0x57, 0x51, 0xf1, 0x8f, 0xfb, 0xd1,
    0x10, 0x6f, 0x11, 0xb9, 0x94, 0x5c, 0x9c, 0x12, 0x26, 0x46,
    0x46, 0x7b, 0x31, 0x0e, 0xad, 0x93, 0xe4, 0x4f, 0x09, 0xe3,
    0xbf, 0xc5, 0xe3, 0x11, 0xa4, 0x25, 0x8d, 0x9b, 0x8e, 0x26,
    0x02, 0xaa, 0x72, 0x18, 0xce, 0x89, 0x67, 0xfc, 0x1c, 0x28,
    0xab, 0x11, 0x5a, 0x84, 0x23, 0x7c, 0x91, 0xac, 0x6b, 0x48,
    0x9c, 0x39, 0x14, 0xa3, 0xac, 0xc6, 0x30, 0xbc, 0x1e, 0x0c,
    0xd3, 0x34, 0x19, 0xa9, 0x2b, 0xe7, 0xa4, 0xf8, 0xc1, 0xf0,
    0x3c, 0x60, 0xa2, 0xf7, 0x51, 0x86, 0xcf, 0x42, 0xad, 0x34,
    0x81, 0xa6, 0x93, 0x0b, 0x88, 0x4c, 0xbf, 0xd2, 0x4f, 0xe0,
    0xdb, 0xb2, 0x1d, 0x6d, 0xb2, 0x5c, 0xac, 0xd8, 0x64, 0x85,
    0xc3, 0x35, 0x6e, 0x5d, 0xaf, 0x63, 0x3e, 0x47, 0xb7, 0x5d,
    0x39, 0x21, 0x36, 0xa6, 0xd4, 0xef, 0x9e, 0x1c, 0x1f, 0xd6,
    0xa4, 0xe0, 0xe4, 0x22, 0x75, 0x1e, 0xeb, 0x15, 0xb4, 0xee,
    0x43, 0x37, 0x06, 0xf9, 0x77, 0xbf, 0x68, 0x9b, 0x9a, 0x7f,
    0x38, 0x30, 0x87, 0xde, 0x0c, 0x6a, 0x39, 0x41, 0xe1, 0xed,
    0xf4, 0x18, 0x6e, 0x29, 0x44, 0xf0, 0xfc, 0xb6, 0x09, 0x5b,
    0xb3, 0x30, 0xc9, 0x0a, 0x8c, 0x41, 0x6f, 0x1e, 0x95, 0xbe,
    0x93, 0x3c, 0x11, 0x9b, 0x24, 0xf7, 0x57, 0xb8, 0xc5, 0x9b,
    0x08, 0xaa, 0xcd, 0x24, 0x86, 0x98, 0x59, 0x0f, 0xc6, 0x0e,
    0xd2, 0x71, 0xb2, 0x5e, 0xae, 0x72, 0xc9, 0x69, 0x3b, 0x80,
    0xc2, 0x27,
};
static const int sizeof_bench_dilithium_level3_key = sizeof(bench_dilithium_level3_key);

#endif /* !WOLFSSL_DILITHIUM_NO_SIGN */

#ifndef WOLFSSL_DILITHIUM_NO_VERIFY

/* raw public key without ASN1 syntax from
 * ./certs/dilithium/bench_dilithium_level3_key.der */
static const unsigned char bench_dilithium_level3_pubkey[] = {
    0x15, 0xc9, 0xe5, 0x53, 0x2f, 0xd8, 0x1f, 0xb4, 0xa3, 0x9f,
    0xae, 0xad, 0xb3, 0x10, 0xd0, 0x72, 0x69, 0xd3, 0x02, 0xf3,
    0xdf, 0x67, 0x5a, 0x31, 0x52, 0x19, 0xca, 0x39, 0x27, 0x77,
    0x61, 0x6d, 0x0f, 0xc1, 0x33, 0x26, 0x09, 0xf0, 0xf9, 0x4d,
    0x12, 0x7a, 0xef, 0xf7, 0x21, 0x26, 0x2c, 0xe0, 0xe2, 0x92,
    0x1f, 0x9d, 0xd1, 0xaa, 0xaf, 0x08, 0x14, 0xf2, 0xaa, 0x24,
    0x99, 0x0f, 0x20, 0x57, 0x35, 0x04, 0x32, 0x96, 0x8e, 0x6e,
    0x10, 0x64, 0xe3, 0xe3, 0x57, 0x26, 0x33, 0x32, 0x7b, 0xe4,
    0x18, 0x41, 0x77, 0xd3, 0x24, 0x63, 0x3d, 0x11, 0xea, 0xdc,
    0xbe, 0x59, 0xff, 0x8d, 0xc2, 0xe4, 0xc7, 0x04, 0xf3, 0xd4,
    0xe0, 0x1d, 0x5e, 0x09, 0x46, 0xbf, 0x02, 0x05, 0xc7, 0xa6,
    0xb7, 0x82, 0x40, 0x1f, 0x55, 0xe9, 0x77, 0x82, 0xc0, 0xcc,
    0x86, 0x99, 0x19, 0x99, 0xa2, 0xc9, 0x1b, 0x4f, 0xdd, 0x49,
    0x4c, 0x78, 0x0a, 0x58, 0xb8, 0xf0, 0x23, 0xac, 0x1a, 0x71,
    0x57, 0x6d, 0xd6, 0x3a, 0x3a, 0x6f, 0x93, 0xb3, 0x2b, 0x09,
    0xbe, 0xec, 0x7b, 0x5b, 0xf7, 0x3a, 0xed, 0xf9, 0xd0, 0xb1,
    0xfe, 0x9f, 0x9b, 0xec, 0x11, 0xb6, 0x6b, 0xd1, 0xb6, 0x00,
    0x72, 0x7f, 0x68, 0x9a, 0x61, 0xa5, 0xf5, 0x6e, 0xe9, 0x46,
    0xa4, 0x82, 0x08, 0x9f, 0x50, 0x4c, 0x75, 0xc3, 0x48, 0x85,
    0x76, 0x39, 0xea, 0x0c, 0xf2, 0xe8, 0x7e, 0x48, 0x69, 0xd9,
    0x6f, 0x9a, 0x89, 0x7d, 0x98, 0xc1, 0x16, 0xdc, 0x2f, 0xc7,
    0x0a, 0x11, 0xa8, 0xbb, 0xe7, 0x91, 0xb1, 0x0f, 0x0e, 0xf0,
    0xb4, 0xc8, 0x41, 0x7e, 0x62, 0x9e, 0x3c, 0x30, 0x4c, 0xbc,
    0x4c, 0xeb, 0x37, 0xaf, 0x48, 0x72, 0x59, 0x64, 0x8e, 0xfb,
    0x77, 0x11, 0x28, 0xdd, 0x30, 0x52, 0x8e, 0x69, 0x8c, 0x9f,
    0x3d, 0xec, 0xdf, 0xa7, 0x5f, 0x42, 0x18, 0xda, 0xba, 0x1a,
    0x96, 0x91, 0x7d, 0x62, 0xd5, 0x52, 0xff, 0x44, 0xc9, 0x1d,
    0x29, 0xa6, 0xb9, 0x03, 0x9a, 0x26, 0x26, 0xcf, 0x57, 0x40,
    0x70, 0x7e, 0x2b, 0xbd, 0xf0, 0x81, 0x71, 0x0f, 0x0b, 0x2e,
    0x9b, 0x03, 0xba, 0x31, 0x41, 0x68, 0x37, 0xc8, 0xff, 0xea,
    0xc4, 0x73, 0xa5, 0xf9, 0xc2, 0x92, 0x78, 0x0c, 0xe7, 0xfd,
    0x5d, 0xb2, 0x01, 0xb5, 0x8d, 0xeb, 0x64, 0xd4, 0x14, 0xea,
    0x7a, 0xd1, 0x42, 0xc8, 0x99, 0xe4, 0x7d, 0x5b, 0x7e, 0x3b,
    0x8f, 0xab, 0x82, 0x12, 0xdf, 0xbb, 0xa1, 0x45, 0x30, 0xc9,
    0x0f, 0xb9, 0xe5, 0xba, 0xe6, 0x8a, 0xf3, 0x78, 0x61, 0xcc,
    0x9f, 0xe1, 0x46, 0x2a, 0x9a, 0x18, 0x0e, 0x2a, 0x57, 0xf3,
    0xe5, 0x56, 0xd1, 0x42, 0x48, 0xe1, 0x5a, 0x8e, 0x33, 0xce,
    0x19, 0xe5, 0x3e, 0x7f, 0x00, 0x70, 0x9c, 0x4c, 0xd3, 0xe1,
    0x0c, 0xa1, 0x7e, 0xd4, 0xa9, 0x9e, 0x8b, 0xe2, 0xf0, 0xac,
    0xdb, 0xa6, 0x72, 0x75, 0x67, 0xa6, 0x57, 0xed, 0x79, 0x2e,
    0xca, 0x8d, 0xeb, 0x9b, 0x9e, 0xb7, 0xbf, 0x30, 0x02, 0x2b,
    0xb3, 0x43, 0x89, 0x9b, 0xa8, 0x88, 0xa5, 0xbb, 0x33, 0xd9,
    0x99, 0x30, 0x7c, 0xc7, 0xd4, 0x28, 0x5e, 0x5e, 0x3f, 0x9d,
    0x6d, 0x35, 0x75, 0x33, 0x8e, 0xff, 0x84, 0x2e, 0x2d, 0xda,
    0xf0, 0xff, 0x70, 0xe5, 0xb5, 0x62, 0x96, 0x33, 0x3a, 0xd9,
    0xb5, 0x82, 0x25, 0x81, 0x81, 0x40, 0x5d, 0x4f, 0x11, 0x86,
    0x63, 0x1a, 0x06, 0xc1, 0x67, 0xc7, 0x49, 0x03, 0xc7, 0xe4,
    0x6f, 0xb4, 0x13, 0x3e, 0x57, 0x62, 0xfd, 0x8a, 0xc6, 0x2b,
    0x65, 0x5b, 0xa4, 0x29, 0x57, 0x8d, 0xde, 0xa5, 0xee, 0x32,
    0xc2, 0x76, 0x03, 0xca, 0xce, 0xc1, 0x48, 0xec, 0x45, 0xcf,
    0x30, 0x21, 0x28, 0x7f, 0x10, 0x47, 0xd2, 0xdb, 0xee, 0xca,
    0x5b, 0x0f, 0xd5, 0x39, 0x3a, 0xc3, 0xa6, 0x78, 0xb2, 0x15,
    0xaf, 0x82, 0x3c, 0x2f, 0xc4, 0x51, 0x5c, 0x52, 0xad, 0xf2,
    0x89, 0x92, 0x8e, 0xf3, 0x50, 0x38, 0xed, 0xf8, 0xc9, 0x14,
    0x4c, 0xe4, 0xa3, 0x9a, 0xaf, 0xc4, 0x5c, 0xf3, 0x9f, 0xc3,
    0xa3, 0xc0, 0xbe, 0x45, 0x1b, 0x21, 0x63, 0xfa, 0xe0, 0xe0,
    0x91, 0x2b, 0x42, 0xca, 0x91, 0xfb, 0x5e, 0x97, 0x9a, 0x0a,
    0xd4, 0x88, 0xba, 0xb8, 0x22, 0xc6, 0xbf, 0x56, 0x58, 0x1e,
    0x92, 0xa9, 0x9d, 0xa7, 0xed, 0xc9, 0xab, 0x54, 0x4f, 0x75,
    0x8d, 0x42, 0xc1, 0xe1, 0x61, 0xd0, 0x91, 0x9a, 0x3a, 0x40,
    0x9a, 0xa3, 0xfb, 0x7b, 0x4e, 0xf0, 0x85, 0xf0, 0xdc, 0x40,
    0x72, 0x9f, 0x05, 0xa8, 0xbe, 0x95, 0x5a, 0x7f, 0xba, 0x75,
    0x00, 0x6e, 0x95, 0x76, 0xbd, 0xb2, 0x40, 0xf5, 0xb0, 0x64,
    0x0a, 0x2f, 0x06, 0x3d, 0x9f, 0xac, 0x6a, 0xa5, 0x46, 0x5a,
    0x85, 0xa4, 0x6f, 0xee, 0x27, 0xa0, 0xeb, 0x5f, 0x1f, 0x91,
    0xbd, 0x2b, 0x02, 0x16, 0xdf, 0x74, 0x97, 0x2c, 0xd0, 0xa8,
    0x9f, 0x3a, 0x7b, 0xdf, 0x3e, 0x98, 0x4a, 0x91, 0xdc, 0x19,
    0x96, 0x88, 0x75, 0x21, 0x1a, 0x6a, 0xa8, 0x4b, 0x1f, 0x35,
    0xd1, 0x92, 0xf5, 0x76, 0xf4, 0x72, 0x55, 0x13, 0xdb, 0x5d,
    0x07, 0x8d, 0xd9, 0x72, 0xe4, 0x75, 0xde, 0x80, 0xbc, 0xe9,
    0x9c, 0xf0, 0x5c, 0x6a, 0x8a, 0x0e, 0x34, 0xf6, 0x3f, 0x5c,
    0xef, 0x0e, 0xcc, 0x52, 0x38, 0x2d, 0x7b, 0xc2, 0x1b, 0x69,
    0x9f, 0xe5, 0xed, 0x14, 0xb0, 0x91, 0x0b, 0xe9, 0x4d, 0x34,
    0xd5, 0xaa, 0xd4, 0xd2, 0x46, 0x39, 0x45, 0x7e, 0x85, 0x2f,
    0xdb, 0x89, 0xf4, 0xff, 0x05, 0x74, 0x51, 0xba, 0xdd, 0xee,
    0xf6, 0xc2, 0xc1, 0x0a, 0x8f, 0xd9, 0xeb, 0xc7, 0x61, 0x30,
    0x8f, 0x86, 0x8b, 0x1f, 0x82, 0xc1, 0x22, 0xfd, 0x83, 0xf4,
    0x5d, 0xc5, 0x94, 0xf5, 0xd7, 0x17, 0xc7, 0x7b, 0x71, 0xf5,
    0x5e, 0x15, 0x49, 0x70, 0xb2, 0x57, 0xa0, 0xc0, 0x57, 0x63,
    0x53, 0x35, 0xb6, 0x52, 0x20, 0x7b, 0x83, 0xd4, 0x57, 0x63,
    0x25, 0x8e, 0x83, 0xb3, 0x8e, 0x26, 0x1f, 0x09, 0xde, 0x14,
    0xd6, 0xa6, 0xfc, 0xe5, 0x93, 0x3c, 0x88, 0x8e, 0xf5, 0x10,
    0x57, 0xb9, 0xc9, 0x9b, 0xff, 0x72, 0x9d, 0x3d, 0x3f, 0x97,
    0xd9, 0x3c, 0x20, 0xe2, 0x57, 0xfd, 0x2a, 0x5c, 0x17, 0x12,
    0xe6, 0x08, 0xaf, 0xe4, 0x26, 0x96, 0xb9, 0x6d, 0xc3, 0xac,
    0x22, 0xf3, 0x8b, 0x89, 0xde, 0xc7, 0x8a, 0x93, 0x06, 0xf7,
    0x1d, 0x08, 0x21, 0x36, 0x16, 0x74, 0x2b, 0x97, 0x23, 0xe4,
    0x79, 0x31, 0x08, 0x23, 0x62, 0x30, 0x67, 0xe2, 0xed, 0x30,
    0x9b, 0x0c, 0xf9, 0x08, 0x7a, 0x29, 0x73, 0xc6, 0x77, 0x8a,
    0xbb, 0x2a, 0x1c, 0x66, 0xd0, 0xdd, 0x9e, 0xa3, 0xe9, 0x62,
    0xcc, 0xb7, 0x88, 0x25, 0x4a, 0x5f, 0xbc, 0xaa, 0xe3, 0xe4,
    0x4f, 0xec, 0xa6, 0x8e, 0xa6, 0xa4, 0x1b, 0x22, 0x2b, 0x2c,
    0x8f, 0x57, 0x7f, 0xb7, 0x33, 0xfe, 0x16, 0x43, 0x85, 0xc5,
    0xd2, 0x95, 0xe6, 0xb9, 0x21, 0x68, 0x88, 0x98, 0x33, 0x8c,
    0x1d, 0x15, 0x9c, 0x4d, 0x62, 0x1f, 0x6b, 0xe8, 0x7a, 0x2d,
    0x6b, 0x0e, 0xc3, 0xde, 0x1a, 0xa8, 0xed, 0x67, 0xb3, 0xb3,
    0x36, 0x5b, 0x4b, 0xcb, 0xe8, 0xa8, 0x5c, 0x0b, 0x2f, 0xca,
    0xd7, 0x71, 0xe8, 0x85, 0xe7, 0x4d, 0xe5, 0x7b, 0x45, 0xed,
    0xb2, 0x4c, 0x69, 0x04, 0x7e, 0x4f, 0xc0, 0xef, 0x1a, 0xca,
    0x0d, 0xa6, 0xc4, 0x79, 0x15, 0x78, 0x9c, 0xd2, 0x91, 0x3c,
    0x32, 0x55, 0x40, 0xe7, 0xcb, 0x7e, 0xde, 0x07, 0xa6, 0x97,
    0x00, 0x2d, 0x70, 0xf6, 0x3d, 0x15, 0xdf, 0x29, 0x8e, 0xa3,
    0x96, 0x6d, 0xf2, 0xbb, 0xa5, 0x1b, 0x7b, 0x58, 0x30, 0xf6,
    0x17, 0xbd, 0xda, 0x13, 0xf7, 0x33, 0xc2, 0x62, 0x32, 0xd4,
    0x1c, 0x2e, 0x31, 0x74, 0x92, 0xad, 0x99, 0x8c, 0x0e, 0x7c,
    0x50, 0x21, 0xcd, 0xff, 0x41, 0xeb, 0xd1, 0xca, 0x14, 0xb7,
    0xb2, 0x31, 0x2f, 0xbe, 0x16, 0xce, 0x4f, 0x26, 0x16, 0x04,
    0xc2, 0xaf, 0xbe, 0x0d, 0x24, 0xab, 0x9a, 0x21, 0x37, 0x06,
    0xac, 0x50, 0x23, 0xf1, 0xbe, 0x5c, 0xbb, 0x64, 0xf3, 0xd3,
    0x66, 0xa3, 0xb8, 0xbe, 0x8b, 0x49, 0x8d, 0xf6, 0xc7, 0xb9,
    0x8f, 0x4e, 0x31, 0x06, 0x51, 0xe5, 0xf3, 0x0e, 0x56, 0xc4,
    0x24, 0x30, 0xf5, 0xe9, 0x36, 0x71, 0xbc, 0xc9, 0x70, 0x2c,
    0x6c, 0x4c, 0x15, 0x43, 0x44, 0xa4, 0xfc, 0xf1, 0xd2, 0x71,
    0x6c, 0x4c, 0xce, 0x30, 0x6c, 0x05, 0x7d, 0x2e, 0xb7, 0xbc,
    0xe4, 0x65, 0x76, 0x24, 0x75, 0x36, 0xdf, 0x28, 0xfc, 0xcd,
    0x9a, 0xba, 0xc2, 0xcd, 0xb0, 0x30, 0xdb, 0xe7, 0x2e, 0x3c,
    0x92, 0x63, 0x1d, 0x30, 0x23, 0x74, 0xb1, 0xb8, 0xcc, 0xd7,
    0xb6, 0x90, 0x65, 0x73, 0xa2, 0x2a, 0x6e, 0x49, 0x95, 0x0d,
    0xab, 0x24, 0xdf, 0x2d, 0xbf, 0x76, 0x46, 0x01, 0x44, 0xe4,
    0x18, 0x8e, 0xd5, 0x9a, 0x76, 0xc9, 0xc6, 0xbc, 0xdb, 0x7f,
    0x80, 0x52, 0xc6, 0x40, 0x41, 0x12, 0x36, 0x7c, 0x80, 0x69,
    0xce, 0x7b, 0xe1, 0xa0, 0x53, 0xa2, 0xd6, 0x8f, 0x3f, 0xf7,
    0xd7, 0x61, 0x09, 0x70, 0xa2, 0xa0, 0xc6, 0xaf, 0xa0, 0xd0,
    0xfa, 0x13, 0xbf, 0xc0, 0x69, 0x15, 0xce, 0x15, 0xec, 0x24,
    0x4b, 0x6b, 0xdc, 0x93, 0x51, 0xc6, 0x82, 0x19, 0x92, 0x84,
    0x5d, 0x99, 0xb0, 0x90, 0x2c, 0xcc, 0x2a, 0x81, 0x6b, 0x22,
    0x64, 0x0a, 0xcb, 0x51, 0x25, 0x82, 0x50, 0x02, 0x2d, 0x3e,
    0xd4, 0x72, 0xb3, 0x0c, 0x15, 0x77, 0xd2, 0xca, 0x98, 0x2f,
    0x41, 0x93, 0x14, 0xb2, 0x7f, 0xa1, 0x97, 0xa3, 0xb8, 0x8a,
    0x56, 0x24, 0x38, 0xa7, 0x36, 0xc5, 0x01, 0xc0, 0x9f, 0x3f,
    0x3e, 0x9a, 0xf6, 0xe9, 0x16, 0x82, 0x01, 0x58, 0x70, 0x0e,
    0x0d, 0xbc, 0xfa, 0x03, 0x57, 0x65, 0xa8, 0x5a, 0x3d, 0x57,
    0x81, 0x23, 0xbe, 0x6e, 0xa9, 0xe8, 0x22, 0xdf, 0x2f, 0x70,
    0xeb, 0x0a, 0x03, 0x96, 0x6b, 0xef, 0x20, 0x9f, 0xf2, 0x62,
    0xe7, 0xb2, 0x6e, 0x3a, 0x1e, 0x40, 0x1f, 0xd2, 0x97, 0x48,
    0xd1, 0x18, 0xf0, 0xeb, 0x52, 0x58, 0x02, 0x26, 0xce, 0x75,
    0xb1, 0x3a, 0x9d, 0x5b, 0x52, 0x94, 0xb2, 0x6e, 0x0e, 0x3f,
    0x39, 0xb6, 0xd9, 0x8a, 0x9d, 0xe8, 0x7c, 0x83, 0x32, 0xcc,
    0x43, 0x35, 0x9b, 0x7a, 0xed, 0xb2, 0x1e, 0x51, 0x37, 0x6c,
    0x14, 0xd8, 0xb8, 0x55, 0xb3, 0x91, 0xef, 0x0c, 0x3a, 0xe5,
    0x77, 0xd0, 0xbd, 0xb0, 0x7d, 0x38, 0x84, 0x2a, 0x47, 0xb2,
    0xb6, 0xda, 0xd7, 0x75, 0xd6, 0x2e, 0x60, 0xc7, 0x10, 0x52,
    0xf7, 0xdd, 0x09, 0x15, 0x6f, 0x04, 0x31, 0xc3, 0x5a, 0x6b,
    0x0c, 0x60, 0x10, 0xa8, 0x6e, 0x20, 0xa9, 0xdd, 0xb7, 0x72,
    0xc3, 0x9e, 0x85, 0xd2, 0x8f, 0x16, 0x7e, 0x3d, 0xe0, 0x63,
    0x81, 0x32, 0xfd, 0xca, 0xbc, 0x0f, 0xef, 0x3e, 0x74, 0x6a,
    0xb1, 0x60, 0xc1, 0x10, 0x50, 0x7c, 0x67, 0xa4, 0x19, 0xa7,
    0xb8, 0xed, 0xe6, 0xf5, 0x4e, 0x41, 0x53, 0xa6, 0x72, 0x1b,
    0x2c, 0x33, 0x6a, 0x37, 0xf1, 0xb5, 0x1c, 0x01, 0x7d, 0xa2,
    0x1f, 0x2c, 0x4e, 0x0a, 0xbf, 0xd4, 0x2c, 0x24, 0x91, 0x58,
    0x62, 0xfb, 0xf8, 0x63, 0xd9, 0xf8, 0x78, 0xf5, 0xc7, 0x78,
    0x32, 0xda, 0x99, 0xeb, 0x58, 0x20, 0x25, 0x19, 0xb1, 0x06,
    0x7f, 0x6a, 0x29, 0x20, 0xdb, 0xc8, 0x22, 0x48, 0xa9, 0x7f,
    0x24, 0x54, 0x8d, 0x7d, 0x8d, 0xb1, 0x69, 0xb2, 0xa3, 0x98,
    0x14, 0x0f, 0xba, 0xfa, 0xb6, 0x15, 0xe8, 0x28, 0x99, 0x3f,
    0x30, 0x04, 0x50, 0xab, 0x5a, 0x3c, 0xf1, 0x97, 0xe1, 0xc8,
    0x0f, 0x0e, 0xb4, 0x11, 0x63, 0x5a, 0x79, 0x08, 0x48, 0x75,
    0xaf, 0x9b, 0xca, 0xd9, 0x13, 0x18, 0xcc, 0xb1, 0xb3, 0xee,
    0xdd, 0x63, 0xdd, 0xf4, 0x21, 0x98, 0x76, 0xe2, 0x3e, 0xd5,
    0x86, 0x23, 0x33, 0x7e, 0xc7, 0xb4, 0x35, 0x4b, 0xc2, 0x2d,
    0xe1, 0xe2, 0xb0, 0x6c, 0x8b, 0x9b, 0x20, 0x3d, 0x48, 0x24,
    0x7c, 0xea, 0xa1, 0x75, 0x27, 0xe5, 0xf4, 0x70, 0xeb, 0x3b,
    0xc7, 0x26, 0x37, 0x04, 0xff, 0x8a, 0x7a, 0xd0, 0xc2, 0xb7,
    0x84, 0xb7, 0x29, 0xfb, 0x0e, 0xa3, 0xa8, 0x71, 0xcd, 0x58,
    0x06, 0x36, 0xe2, 0xf2, 0x77, 0xcc, 0x0f, 0x78, 0x08, 0x2b,
    0xbb, 0xe3, 0x53, 0x05, 0x71, 0xdc, 0x6c, 0x37, 0x32, 0x91,
    0x46, 0x42, 0x4f, 0x21, 0xe0, 0x34, 0xad, 0x3f, 0x30, 0x5a,
    0xc7, 0x0d, 0x17, 0x19, 0x39, 0x31, 0x58, 0x69, 0x3c, 0x8c,
    0xbe, 0xe7, 0xa6, 0x3b, 0xad, 0xfb, 0x46, 0x89, 0x06, 0xc1,
    0x8c, 0x16, 0x9a, 0x06, 0x3a, 0xd0, 0x7e, 0xd6, 0xb0, 0x7b,
    0x7d, 0xf8, 0x91, 0x7c, 0xfa, 0xd9, 0x66, 0x39, 0xfa, 0xbc,
    0x57, 0xa7, 0x78, 0x8b, 0x36, 0x78, 0xc0, 0x1c, 0x0e, 0x23,
    0x05, 0x0e, 0x04, 0x61, 0x16, 0x34, 0xf9, 0xc6, 0x63, 0x58,
    0xdf, 0xf4, 0x52, 0xce, 0xd0, 0x0f, 0x0c, 0xec, 0xb1, 0x82,
    0xf4, 0x72, 0x73, 0x72, 0x3f, 0x02, 0xbe, 0xe3, 0x9c, 0x63,
    0x73, 0xc8, 0x21, 0x65, 0xba, 0x57, 0x52, 0xa9, 0x19, 0xac,
    0x68, 0x50, 0xbd, 0x2d, 0x72, 0x5b, 0x93, 0x0f, 0x1c, 0x81,
    0x77, 0xd7, 0x2e, 0xc3, 0x93, 0x52, 0x6e, 0xdc, 0x79, 0x52,
    0x9f, 0xe3, 0xde, 0xe1, 0xba, 0x58, 0x55, 0xab, 0x8a, 0xf2,
    0x35, 0x6a, 0xcf, 0x94, 0x1f, 0x17, 0xa4, 0x23, 0x2e, 0x8e,
    0x18, 0x21, 0xbe, 0x14, 0xfa, 0xe7, 0x59, 0xc5, 0x44, 0x34,
    0xce, 0x03, 0xf4, 0xb7, 0x75, 0xd3, 0x51, 0x55, 0xdf, 0xff,
    0xcf, 0x4f, 0x44, 0xee, 0x13, 0x9b, 0xcb, 0x12, 0xae, 0xe5,
    0x5b, 0x44, 0x65, 0x28, 0xcb, 0x6a, 0x9c, 0x24, 0x1d, 0xea,
    0x2d, 0x5e, 0xa5, 0xc3, 0x78, 0xad, 0xed, 0x0c, 0x05, 0xa6,
    0xaf, 0x95, 0x04, 0xd2, 0xb5, 0x91, 0x0e, 0xa0, 0x06, 0x77,
    0xc5, 0x82, 0xf6, 0xdd, 0x72, 0x83, 0x04, 0xcc, 0xb0, 0xab,
    0x7a, 0xf0, 0xb4, 0x4d, 0x36, 0x71, 0x72, 0x1a, 0x9a, 0x0d,
    0xcd, 0xa3, 0x11, 0xa8, 0x0d, 0x7d, 0x49, 0xce, 0x9c, 0x09,
    0x1d, 0x08, 0xa4, 0x39, 0x2e, 0x03, 0xdf, 0x3a, 0xc8, 0xfe,
    0x6a, 0x2b, 0x0b, 0x07, 0x80, 0x55, 0x8a, 0xa8, 0xe6, 0x0e,
    0xc9, 0x7e, 0x83, 0xce, 0x3a, 0x98, 0x98, 0x4e, 0x3e, 0x08,
    0x20, 0x8f, 0x10, 0xfc, 0xc1, 0xc4, 0xcf, 0x37, 0x8d, 0x69,
    0xd8, 0x57, 0x9d, 0x48, 0x80, 0x6a, 0xef, 0x0c, 0xdd, 0x27,
    0x99, 0xf9, 0xe7, 0xd0, 0xd2, 0x36, 0xd8, 0xed, 0x41, 0x14,
    0x1b, 0x10,
};
static const int sizeof_bench_dilithium_level3_pubkey =
    sizeof(bench_dilithium_level3_pubkey);

#endif /* !WOLFSSL_DILITHIUM_NO_VERIFY */

#ifndef WOLFSSL_DILITHIUM_NO_SIGN

/* raw private key without ASN1 syntax from
 * ./certs/dilithium/bench_dilithium_level5_key.der */
static const unsigned char bench_dilithium_level5_key[] = {
    0xef, 0x49, 0x79, 0x47, 0x15, 0xc4, 0x8a, 0xa9, 0x74, 0x2a,
    0xf0, 0x36, 0x94, 0x5c, 0x91, 0x1c, 0x5d, 0xff, 0x2c, 0x83,
    0xf2, 0x8b, 0x04, 0xfc, 0x5d, 0x64, 0xbd, 0x49, 0x73, 0xcd,
    0xcc, 0x99, 0xfd, 0x0f, 0x8f, 0x6e, 0xad, 0x75, 0x9b, 0xc9,
    0xb1, 0xb9, 0x90, 0x93, 0xbf, 0xce, 0x02, 0x2d, 0x12, 0x0c,
    0x54, 0x2e, 0xe2, 0x3e, 0x52, 0xff, 0xe0, 0x7a, 0xca, 0x2d,
    0x81, 0x84, 0xea, 0x16, 0x1f, 0x10, 0xc4, 0xc9, 0xde, 0xcd,
    0xf6, 0xbd, 0x60, 0xc9, 0xb3, 0xd0, 0x0f, 0x57, 0xeb, 0x71,
    0x78, 0x9b, 0xb5, 0x72, 0x2a, 0x65, 0x11, 0x14, 0xff, 0x63,
    0x8d, 0x38, 0xcf, 0xa4, 0xf4, 0xad, 0xd0, 0x68, 0x84, 0x97,
    0xfe, 0xd3, 0x91, 0xa0, 0xe4, 0xc3, 0x74, 0xcf, 0x20, 0x87,
    0x89, 0x84, 0x1f, 0x75, 0x91, 0xe3, 0xb3, 0x47, 0x8b, 0xfe,
    0x76, 0xb7, 0x2d, 0x30, 0x89, 0x02, 0x04, 0xc9, 0x93, 0xa8,
    0x31, 0xd3, 0x84, 0x2d, 0xe4, 0x26, 0x12, 0xdb, 0x94, 0x08,
    0x12, 0x45, 0x45, 0xca, 0x44, 0x89, 0x52, 0xc4, 0x28, 0x41,
    0x46, 0x01, 0x1c, 0x93, 0x20, 0x8b, 0x40, 0x6d, 0x09, 0x36,
    0x65, 0x4c, 0xa2, 0x40, 0x62, 0xb8, 0x2c, 0x1b, 0x00, 0x20,
    0x61, 0x42, 0x8c, 0x24, 0xa7, 0x10, 0x19, 0x27, 0x25, 0x22,
    0x14, 0x31, 0x13, 0x33, 0x46, 0x0c, 0x22, 0x22, 0x18, 0xa7,
    0x91, 0x0c, 0x24, 0x61, 0xd9, 0x32, 0x46, 0xc8, 0x96, 0x49,
    0x5c, 0x90, 0x89, 0x9b, 0x84, 0x01, 0x5c, 0x08, 0x42, 0x64,
    0x84, 0x85, 0x0c, 0x42, 0x21, 0x20, 0x48, 0x21, 0x92, 0x00,
    0x28, 0x83, 0x20, 0x4c, 0x08, 0xc7, 0x51, 0x99, 0x06, 0x66,
    0x01, 0x18, 0x51, 0x13, 0x48, 0x0a, 0x0b, 0x42, 0x90, 0x4c,
    0x14, 0x08, 0x83, 0x14, 0x6d, 0x10, 0x10, 0x91, 0xe2, 0xc4,
    0x8d, 0xe1, 0x12, 0x11, 0x10, 0x40, 0x29, 0x99, 0x92, 0x30,
    0x12, 0x39, 0x6c, 0x91, 0x86, 0x68, 0x08, 0x83, 0x0c, 0x54,
    0x80, 0x80, 0xa2, 0x08, 0x52, 0x09, 0x30, 0x71, 0x0c, 0x10,
    0x04, 0x53, 0x00, 0x65, 0x91, 0x12, 0x2d, 0x0c, 0xa2, 0x8c,
    0x18, 0x14, 0x45, 0xd8, 0x14, 0x06, 0xe4, 0x36, 0x72, 0x93,
    0x10, 0x68, 0x09, 0xc2, 0x08, 0x51, 0x14, 0x8c, 0x13, 0x39,
    0x11, 0xd8, 0x44, 0x02, 0x18, 0x39, 0x29, 0x98, 0x16, 0x71,
    0x82, 0x40, 0x70, 0x01, 0x10, 0x8c, 0x1a, 0x30, 0x08, 0x02,
    0x03, 0x41, 0x5a, 0x00, 0x40, 0xa4, 0x16, 0x90, 0x20, 0x26,
    0x32, 0x00, 0x49, 0x61, 0x20, 0x20, 0x0c, 0x1a, 0xb0, 0x10,
    0x63, 0x10, 0x11, 0x58, 0x30, 0x0d, 0x59, 0x80, 0x68, 0x90,
    0x46, 0x2a, 0x91, 0xa8, 0x71, 0x98, 0x20, 0x40, 0x21, 0x83,
    0x6c, 0xc0, 0x48, 0x0d, 0x8b, 0x90, 0x11, 0x08, 0x09, 0x31,
    0x8c, 0x00, 0x12, 0x10, 0x14, 0x6e, 0xc2, 0x06, 0x32, 0x1a,
    0x26, 0x10, 0x0a, 0x91, 0x44, 0x08, 0x99, 0x8d, 0x60, 0x86,
    0x28, 0x11, 0x20, 0x6d, 0xa3, 0x12, 0x81, 0x8b, 0xc6, 0x51,
    0xcb, 0xa0, 0x61, 0x09, 0x97, 0x61, 0x48, 0xb6, 0x0d, 0x21,
    0x49, 0x51, 0x08, 0x13, 0x0c, 0x0a, 0x34, 0x86, 0x49, 0x80,
    0x65, 0x14, 0x39, 0x04, 0x21, 0x01, 0x81, 0x9a, 0xb8, 0x4d,
    0x04, 0x41, 0x48, 0x03, 0x92, 0x81, 0x62, 0x14, 0x6c, 0x10,
    0x16, 0x11, 0xe2, 0xa2, 0x49, 0xe3, 0x30, 0x65, 0x04, 0x93,
    0x8d, 0x1c, 0x33, 0x70, 0x1b, 0x15, 0x50, 0xe4, 0x38, 0x80,
    0x21, 0x37, 0x06, 0x20, 0xc6, 0x24, 0xc8, 0x22, 0x88, 0x4a,
    0x44, 0x80, 0x14, 0x43, 0x88, 0x54, 0x44, 0x42, 0x11, 0x49,
    0x41, 0x19, 0xb9, 0x2d, 0xcc, 0x04, 0x0d, 0x19, 0xc1, 0x65,
    0x5b, 0xa0, 0x11, 0x94, 0x00, 0x84, 0xe4, 0xb6, 0x41, 0xc2,
    0x18, 0x72, 0x5c, 0x02, 0x69, 0x11, 0x85, 0x24, 0x13, 0x35,
    0x00, 0x62, 0x34, 0x04, 0x58, 0x40, 0x21, 0x00, 0xc4, 0x28,
    0x0c, 0x17, 0x30, 0x10, 0x47, 0x60, 0x4b, 0xc2, 0x61, 0x9c,
    0x80, 0x2c, 0x20, 0x94, 0x31, 0x58, 0x92, 0x09, 0xcc, 0x00,
    0x02, 0x42, 0x94, 0x69, 0x99, 0x28, 0x06, 0x98, 0x02, 0x52,
    0x90, 0x32, 0x6e, 0x8a, 0x18, 0x2e, 0x54, 0x94, 0x81, 0x03,
    0xc6, 0x89, 0x03, 0xa1, 0x84, 0x48, 0x82, 0x48, 0x52, 0xc4,
    0x00, 0x91, 0x30, 0x24, 0x20, 0x12, 0x0d, 0x83, 0x80, 0x05,
    0x92, 0x48, 0x61, 0x98, 0x46, 0x92, 0xe1, 0xa6, 0x25, 0x20,
    0x93, 0x4d, 0x1c, 0x37, 0x2c, 0x9b, 0x94, 0x8d, 0xc8, 0x88,
    0x80, 0xa2, 0x18, 0x72, 0x0c, 0x09, 0x70, 0x81, 0x36, 0x90,
    0x24, 0x45, 0x69, 0x53, 0x36, 0x6c, 0xd2, 0x20, 0x51, 0x23,
    0xc1, 0x8c, 0x62, 0xb0, 0x70, 0x11, 0xb2, 0x70, 0xcb, 0x84,
    0x69, 0x4b, 0x32, 0x89, 0x01, 0x21, 0x81, 0x02, 0x38, 0x66,
    0xa3, 0x26, 0x12, 0x24, 0xa3, 0x30, 0x22, 0x24, 0x84, 0x18,
    0xb9, 0x84, 0x40, 0x16, 0x50, 0x22, 0x44, 0x31, 0x1b, 0x13,
    0x8d, 0x53, 0x02, 0x89, 0x4a, 0x22, 0x10, 0x53, 0x18, 0x01,
    0x58, 0x30, 0x2d, 0x00, 0x05, 0x08, 0x13, 0x80, 0x84, 0xc2,
    0x22, 0x0e, 0x88, 0x26, 0x2a, 0x04, 0xc4, 0x4c, 0x19, 0x43,
    0x01, 0xc8, 0x38, 0x4c, 0xd1, 0xb2, 0x90, 0x13, 0x29, 0x10,
    0x12, 0x48, 0x22, 0x01, 0xa8, 0x51, 0xd1, 0x92, 0x40, 0x11,
    0x27, 0x62, 0x10, 0x01, 0x0c, 0x0c, 0xc6, 0x28, 0xe3, 0x46,
    0x60, 0x24, 0x01, 0x8d, 0x14, 0xb6, 0x10, 0x50, 0xb6, 0x25,
    0x44, 0x38, 0x40, 0x44, 0xc2, 0x0c, 0x19, 0xc0, 0x64, 0x9c,
    0x44, 0x02, 0x21, 0x25, 0x65, 0x02, 0x23, 0x86, 0x1a, 0x12,
    0x70, 0x51, 0x24, 0x91, 0x09, 0x08, 0x44, 0x09, 0x35, 0x66,
    0x91, 0x04, 0x12, 0x43, 0x42, 0x8d, 0x22, 0xa0, 0x70, 0x14,
    0x91, 0x25, 0xa0, 0x00, 0x80, 0xe4, 0x00, 0x90, 0x44, 0xb2,
    0x61, 0x14, 0x20, 0x6e, 0xca, 0x14, 0x0d, 0x23, 0x85, 0x68,
    0xda, 0x40, 0x92, 0x0b, 0xb1, 0x20, 0x92, 0x04, 0x46, 0xc0,
    0x08, 0x8a, 0x40, 0xc4, 0x4d, 0x0c, 0x17, 0x45, 0xd3, 0x18,
    0x52, 0x1b, 0x46, 0x24, 0xc2, 0x24, 0x71, 0x83, 0x10, 0x80,
    0xc8, 0x82, 0x68, 0xc2, 0x96, 0x81, 0x0a, 0x01, 0x92, 0x60,
    0xb4, 0x84, 0x09, 0xc6, 0x00, 0x04, 0x37, 0x90, 0x0b, 0xa0,
    0x28, 0x12, 0x27, 0x09, 0x94, 0x80, 0x50, 0xd8, 0x04, 0x86,
    0x08, 0x13, 0x8a, 0x4a, 0x06, 0x89, 0x9b, 0xc4, 0x60, 0xe3,
    0xa2, 0x20, 0xe0, 0x38, 0x21, 0x22, 0xb4, 0x68, 0x0a, 0xa1,
    0x0c, 0x01, 0x24, 0x32, 0x4c, 0x48, 0x30, 0xa2, 0x80, 0x8d,
    0x58, 0x44, 0x10, 0xc8, 0x94, 0x6d, 0x21, 0xc3, 0x61, 0xcb,
    0x98, 0x24, 0xdc, 0x38, 0x11, 0xc9, 0x18, 0x11, 0x20, 0x01,
    0x50, 0x1c, 0x34, 0x8d, 0x02, 0x03, 0x09, 0x0a, 0x40, 0x61,
    0xd4, 0xb8, 0x84, 0x9c, 0xc2, 0x09, 0x04, 0xb1, 0x89, 0x83,
    0x86, 0x84, 0x19, 0x83, 0x0c, 0x5a, 0x86, 0x89, 0x10, 0x21,
    0x0d, 0xd1, 0xc2, 0x80, 0x18, 0x29, 0x2a, 0x0c, 0x01, 0x50,
    0x89, 0x88, 0x48, 0x03, 0xa7, 0x85, 0x21, 0x92, 0x64, 0xc4,
    0x16, 0x81, 0x94, 0x06, 0x6c, 0x53, 0x26, 0x12, 0x90, 0xb6,
    0x21, 0x0b, 0xa8, 0x64, 0x43, 0x96, 0x84, 0x41, 0x88, 0x70,
    0xe3, 0xa6, 0x44, 0x12, 0xc0, 0x09, 0x01, 0xc7, 0x60, 0xc3,
    0x20, 0x42, 0xc3, 0x40, 0x68, 0x10, 0xa6, 0x51, 0xa4, 0xa0,
    0x71, 0x54, 0x98, 0x04, 0x88, 0xb2, 0x00, 0x54, 0x18, 0x6a,
    0x48, 0x98, 0x20, 0x21, 0xb2, 0x8d, 0x82, 0x20, 0x81, 0x99,
    0x16, 0x81, 0x0a, 0xc5, 0x88, 0x0a, 0x23, 0x11, 0x8a, 0x16,
    0x44, 0x24, 0xc9, 0x29, 0x59, 0x08, 0x91, 0x1c, 0x29, 0x05,
    0x14, 0xc9, 0x44, 0xe3, 0x20, 0x10, 0x1b, 0xa1, 0x64, 0x82,
    0xa2, 0x90, 0x00, 0x00, 0x82, 0x98, 0xb2, 0x85, 0xc8, 0x04,
    0x28, 0xc8, 0xb2, 0x65, 0xc9, 0xc6, 0x88, 0xcc, 0x08, 0x91,
    0x84, 0x08, 0x30, 0x94, 0x94, 0x8d, 0xc0, 0x18, 0x46, 0x82,
    0x36, 0x4c, 0x83, 0x10, 0x72, 0x23, 0xb1, 0x88, 0x81, 0x20,
    0x8e, 0x19, 0x03, 0x8a, 0x94, 0x46, 0x22, 0x21, 0x35, 0x8e,
    0x04, 0xc0, 0x88, 0x5b, 0xb6, 0x09, 0x0a, 0x18, 0x44, 0x21,
    0x90, 0x65, 0x03, 0xb2, 0x21, 0xc4, 0x10, 0x50, 0xc1, 0x80,
    0x0c, 0x09, 0x40, 0x49, 0xe4, 0xa8, 0x8c, 0xa4, 0x36, 0x61,
    0x59, 0x12, 0x86, 0x20, 0x08, 0x2d, 0x10, 0x19, 0x85, 0xe4,
    0x34, 0x60, 0xc4, 0xb6, 0x60, 0x00, 0x18, 0x06, 0x8c, 0xb8,
    0x45, 0x19, 0x13, 0x4a, 0x53, 0xc4, 0x40, 0xc9, 0x38, 0x71,
    0xd9, 0x48, 0x10, 0x59, 0x08, 0x02, 0x02, 0x10, 0x69, 0x53,
    0x28, 0x80, 0x22, 0x81, 0x4c, 0xc9, 0x16, 0x26, 0xa1, 0x48,
    0x64, 0x19, 0x21, 0x11, 0x1c, 0x37, 0x88, 0x4b, 0x94, 0x2c,
    0x48, 0xc8, 0x6c, 0x63, 0x88, 0x65, 0x81, 0x40, 0x61, 0xa1,
    0x44, 0x31, 0x82, 0x18, 0x08, 0x80, 0x00, 0x26, 0x50, 0x14,
    0x49, 0xa1, 0x32, 0x50, 0x02, 0xc8, 0x45, 0x0c, 0x07, 0x24,
    0x13, 0x01, 0x6d, 0x0a, 0xb3, 0x90, 0x64, 0x30, 0x85, 0x21,
    0x09, 0x61, 0x44, 0x44, 0x72, 0x08, 0x32, 0x06, 0xe1, 0xa2,
    0x21, 0xdb, 0xa4, 0x09, 0x5a, 0xb4, 0x71, 0x43, 0xb2, 0x09,
    0x82, 0xc4, 0x64, 0x88, 0xa0, 0x91, 0xca, 0x14, 0x90, 0xa4,
    0xa8, 0x41, 0xc1, 0x38, 0x85, 0x12, 0x32, 0x60, 0x1a, 0x11,
    0x72, 0x53, 0x32, 0x2c, 0xe3, 0x08, 0x4d, 0x24, 0xc6, 0x28,
    0x0a, 0x03, 0x8c, 0x88, 0x06, 0x05, 0xa0, 0xa8, 0x05, 0x84,
    0xa2, 0x4c, 0x80, 0x40, 0x62, 0xda, 0x24, 0x81, 0x9a, 0x16,
    0x91, 0x24, 0x81, 0x04, 0xa4, 0x46, 0x51, 0xc2, 0xa8, 0x25,
    0x20, 0x28, 0x42, 0x13, 0x46, 0x2c, 0x63, 0x42, 0x72, 0x03,
    0x88, 0x28, 0xa3, 0x22, 0x24, 0x1a, 0x02, 0x26, 0x42, 0xa2,
    0x11, 0x11, 0xb0, 0x51, 0x92, 0xb4, 0x6c, 0xe2, 0x32, 0x85,
    0x10, 0xc2, 0x41, 0xc1, 0x40, 0x46, 0x4c, 0x26, 0x01, 0x1c,
    0x35, 0x02, 0x0c, 0x14, 0x0c, 0x18, 0x81, 0x00, 0x10, 0x26,
    0x02, 0xc8, 0x32, 0x8c, 0xe4, 0x02, 0x68, 0xcc, 0x14, 0x2e,
    0x89, 0x38, 0x60, 0x10, 0x12, 0x24, 0x93, 0x42, 0x65, 0xe3,
    0x24, 0x29, 0x08, 0x80, 0x41, 0x09, 0x29, 0x46, 0x5b, 0x26,
    0x49, 0x5b, 0x30, 0x80, 0x03, 0xc1, 0x2c, 0x04, 0x09, 0x82,
    0x4c, 0x48, 0x2d, 0x1c, 0x36, 0x4d, 0xdb, 0x02, 0x86, 0x21,
    0xb5, 0x51, 0x81, 0x80, 0x2d, 0xcb, 0x20, 0x81, 0x5b, 0x34,
    0x41, 0x89, 0x36, 0x48, 0x44, 0xa0, 0x05, 0x59, 0xb6, 0x64,
    0x12, 0x45, 0x21, 0x20, 0x31, 0x51, 0x0a, 0xc3, 0x8c, 0x14,
    0x48, 0x71, 0x18, 0x35, 0x24, 0x20, 0x45, 0x05, 0x88, 0x20,
    0x09, 0x08, 0xb1, 0x29, 0x18, 0xa0, 0x09, 0x4a, 0x00, 0x8a,
    0xe2, 0xb8, 0x45, 0x02, 0x27, 0x89, 0xd8, 0x10, 0x25, 0x51,
    0x82, 0x8c, 0x13, 0x92, 0x30, 0x1c, 0x24, 0x8e, 0x1c, 0x93,
    0x4d, 0xa3, 0x48, 0x51, 0x93, 0xa8, 0x69, 0xe2, 0x04, 0x89,
    0x13, 0x13, 0x61, 0xcb, 0x98, 0x8c, 0x09, 0x21, 0x62, 0x4b,
    0x14, 0x4e, 0x11, 0xa3, 0x09, 0x98, 0x40, 0x42, 0x91, 0x12,
    0x08, 0x80, 0x84, 0x2d, 0xc0, 0x12, 0x60, 0x03, 0xa4, 0x29,
    0x18, 0x80, 0x01, 0x94, 0x44, 0x8a, 0x12, 0x11, 0x72, 0xc4,
    0x22, 0x32, 0x9a, 0x46, 0x88, 0x1b, 0x16, 0x4d, 0x4b, 0x08,
    0x11, 0x02, 0x48, 0x45, 0x81, 0xa4, 0x64, 0xe1, 0x88, 0x0c,
    0x63, 0x10, 0x70, 0x48, 0x98, 0x05, 0x9b, 0xb8, 0x84, 0x03,
    0x14, 0x05, 0x44, 0x86, 0x0c, 0x20, 0x11, 0x68, 0xbe, 0x71,
    0x83, 0xc2, 0x69, 0xde, 0x49, 0xad, 0xb4, 0xdb, 0x93, 0xcb,
    0x20, 0x2b, 0xbd, 0x95, 0x97, 0x57, 0x7e, 0xcb, 0xbc, 0x73,
    0xb6, 0x3d, 0x16, 0x4a, 0x0e, 0xe4, 0x9c, 0x81, 0xb1, 0x5d,
    0x27, 0x64, 0xa2, 0x14, 0x12, 0x1b, 0x8e, 0xd0, 0xd8, 0x38,
    0xf6, 0xc7, 0xbb, 0x9f, 0x77, 0x3c, 0x62, 0x04, 0x92, 0xe1,
    0x97, 0xaf, 0x24, 0xa7, 0xf9, 0xf0, 0x8d, 0x3a, 0xbf, 0x5d,
    0xab, 0x5c, 0x97, 0x0f, 0xfc, 0x35, 0xbc, 0x62, 0xd8, 0x42,
    0xfd, 0xc7, 0x8b, 0xf7, 0x80, 0xd1, 0x38, 0x68, 0x14, 0x5e,
    0x4f, 0x99, 0x31, 0xc7, 0xaf, 0xbd, 0x27, 0xce, 0x1c, 0x5b,
    0x09, 0x1b, 0xcf, 0xbb, 0xfb, 0xf9, 0xf4, 0x90, 0x4c, 0xc1,
    0xa2, 0x12, 0xf9, 0xd0, 0xa5, 0x2c, 0xfd, 0x7b, 0x55, 0xb0,
    0xb1, 0xc6, 0x42, 0xe6, 0xeb, 0x10, 0x5e, 0xe9, 0x00, 0xe8,
    0x46, 0xe4, 0xe0, 0x8b, 0x21, 0xbc, 0xb1, 0xa9, 0x9e, 0x75,
    0x66, 0xf0, 0xb8, 0x87, 0xb9, 0x11, 0x7e, 0x28, 0x6c, 0x4d,
    0x58, 0xcd, 0x54, 0x71, 0x0c, 0x6a, 0xcc, 0xfb, 0x52, 0xc2,
    0x5b, 0xcc, 0x19, 0x67, 0x4f, 0xc2, 0x2f, 0x09, 0x62, 0x51,
    0x82, 0xeb, 0x9b, 0x94, 0x11, 0xb4, 0x5a, 0x67, 0x7f, 0x58,
    0x18, 0xb2, 0x3f, 0x37, 0x1f, 0x94, 0x44, 0x73, 0x6a, 0x02,
    0xf5, 0xfb, 0x5b, 0x03, 0xac, 0x5d, 0xc6, 0xa9, 0x79, 0x8f,
    0x0f, 0x50, 0xa0, 0x57, 0x46, 0x05, 0x6d, 0x58, 0xde, 0x6e,
    0x8d, 0x9c, 0x0e, 0x6a, 0xb5, 0x9b, 0x1b, 0x22, 0x74, 0xad,
    0x00, 0x55, 0x27, 0x46, 0xce, 0xbb, 0x82, 0x77, 0x4e, 0x6e,
    0x59, 0x38, 0x26, 0xb3, 0xc7, 0xbc, 0x97, 0x54, 0x83, 0x69,
    0x1f, 0x3e, 0xbd, 0x0f, 0xff, 0x2f, 0xca, 0xb9, 0xea, 0x91,
    0x26, 0x8e, 0x0a, 0x78, 0x25, 0xf6, 0x6b, 0x11, 0x30, 0xd7,
    0xe2, 0xf4, 0x2b, 0xda, 0xcf, 0xe1, 0x4a, 0x47, 0xab, 0x5f,
    0x54, 0x34, 0x38, 0xac, 0xd1, 0xbf, 0x45, 0xad, 0x4b, 0x52,
    0x0f, 0x4c, 0xa2, 0xac, 0x22, 0x7c, 0xb6, 0xed, 0x7f, 0xd5,
    0x63, 0x3b, 0x1a, 0x3b, 0xf2, 0x3d, 0x9b, 0x96, 0x92, 0x08,
    0xb9, 0x95, 0x13, 0xaf, 0x20, 0x26, 0x8b, 0x15, 0x97, 0x89,
    0xa5, 0x88, 0x8f, 0x78, 0xb4, 0x57, 0x9d, 0x51, 0x96, 0x9c,
    0x98, 0x93, 0xd5, 0x83, 0xf9, 0xff, 0x94, 0x29, 0x1e, 0xa5,
    0x28, 0xa4, 0x0c, 0x22, 0xab, 0xbc, 0x70, 0x48, 0xa2, 0x16,
    0x1c, 0xa4, 0xba, 0x8b, 0xfe, 0xb2, 0xa9, 0x03, 0x96, 0x5f,
    0xb4, 0x84, 0x8e, 0xb4, 0xbb, 0x7b, 0x11, 0xc5, 0xc2, 0xdb,
    0xe3, 0x88, 0xb5, 0xd3, 0xac, 0x07, 0x33, 0x53, 0xe8, 0x10,
    0x9e, 0xc5, 0x81, 0xb0, 0x77, 0x2f, 0x4f, 0x6d, 0x0d, 0x89,
    0xb4, 0x04, 0x98, 0x05, 0xe6, 0xd3, 0x36, 0x97, 0xcd, 0x3e,
    0x4d, 0xc6, 0x21, 0xe4, 0x0b, 0xcf, 0xed, 0xa7, 0x4d, 0xd9,
    0xd3, 0x25, 0xec, 0xec, 0x47, 0xfd, 0x06, 0x92, 0x77, 0x25,
    0x3c, 0x44, 0xe6, 0x5d, 0xb4, 0x35, 0x2b, 0x5d, 0x05, 0x65,
    0x63, 0x0b, 0xd9, 0xb8, 0x28, 0xdf, 0xdd, 0xfd, 0x64, 0x18,
    0x42, 0x19, 0x7f, 0x12, 0x78, 0xdd, 0xf0, 0x64, 0xd6, 0x99,
    0xb8, 0x74, 0x81, 0xe2, 0xb9, 0xc8, 0x67, 0x6d, 0x31, 0x22,
    0xa5, 0x68, 0xa1, 0x8d, 0x3e, 0x49, 0xbe, 0x10, 0x68, 0xa8,
    0x74, 0x1d, 0x18, 0xcf, 0x00, 0xe1, 0x4f, 0x77, 0xd8, 0xc6,
    0xe3, 0x08, 0xbb, 0x4c, 0xed, 0xff, 0xd9, 0x9b, 0xb0, 0xd1,
    0x50, 0xbb, 0x8b, 0x91, 0xcd, 0x5f, 0x2a, 0xfb, 0x8f, 0x4d,
    0x3c, 0x98, 0xba, 0xd7, 0x98, 0x99, 0xa7, 0x22, 0x14, 0xd7,
    0x94, 0xb5, 0xb8, 0xa4, 0x52, 0x31, 0xa7, 0xa1, 0xa4, 0x28,
    0xee, 0x31, 0xb5, 0xd0, 0xc1, 0x07, 0x05, 0x16, 0x1d, 0x53,
    0x45, 0x62, 0x23, 0x05, 0x44, 0xb6, 0x4f, 0x92, 0x03, 0x53,
    0x9a, 0x71, 0x56, 0xae, 0x16, 0x81, 0xb4, 0xc9, 0x98, 0xf4,
    0x7f, 0x11, 0x37, 0xc2, 0xc8, 0xf2, 0xe4, 0x48, 0xe3, 0xcc,
    0xf1, 0xe3, 0x3d, 0x8e, 0x13, 0x5b, 0x25, 0xad, 0xce, 0x6f,
    0xed, 0x60, 0x4f, 0x7d, 0x51, 0xe1, 0xd0, 0x74, 0xf4, 0xed,
    0xf3, 0x84, 0xa6, 0x0e, 0xba, 0xb4, 0x8e, 0x5a, 0xb9, 0x12,
    0x70, 0x43, 0x4c, 0xb5, 0xa5, 0x1e, 0x86, 0xa5, 0xe3, 0x4d,
    0x76, 0x95, 0xce, 0x2c, 0x53, 0x3a, 0x4e, 0x3f, 0x47, 0x73,
    0x85, 0x88, 0xd9, 0x39, 0x21, 0x83, 0x24, 0x68, 0x6a, 0x1e,
    0x77, 0xdf, 0x59, 0xc5, 0x1b, 0xe2, 0xb1, 0x47, 0x9d, 0xee,
    0x45, 0x1e, 0xc6, 0xd4, 0x43, 0xe2, 0xc7, 0x1c, 0x98, 0x84,
    0xe0, 0x39, 0xe9, 0x9f, 0xa0, 0xa2, 0x24, 0x4a, 0x88, 0x46,
    0xf3, 0x50, 0x52, 0xb5, 0xae, 0x37, 0x5c, 0xa1, 0x7d, 0xad,
    0x7c, 0x30, 0x3e, 0xcd, 0x80, 0x1c, 0xac, 0xf4, 0xe6, 0xb5,
    0x9f, 0x22, 0xb6, 0xfb, 0x0e, 0x6d, 0x80, 0x10, 0xf7, 0x3f,
    0xdd, 0x5b, 0xd9, 0xd4, 0x03, 0x14, 0x41, 0x90, 0x88, 0xa8,
    0xcf, 0x50, 0xa2, 0xf2, 0x7e, 0xf0, 0x0a, 0x7f, 0xed, 0x77,
    0x09, 0x48, 0x32, 0x55, 0xe9, 0x93, 0xe7, 0x27, 0x18, 0x46,
    0x17, 0x03, 0x25, 0x8e, 0x17, 0x5d, 0xe8, 0x9e, 0xb1, 0xb4,
    0x9d, 0x1a, 0x5e, 0xbe, 0xa8, 0xb8, 0x45, 0x30, 0xc6, 0xa5,
    0xb4, 0xaf, 0xf3, 0x0d, 0x91, 0x9c, 0xa9, 0x5b, 0x4c, 0xbb,
    0x19, 0x19, 0x39, 0x51, 0x36, 0x80, 0xf7, 0x10, 0xf7, 0x73,
    0x49, 0x17, 0xec, 0xbc, 0x92, 0x08, 0x21, 0xb1, 0x0c, 0x23,
    0xc4, 0xd6, 0xd2, 0xb3, 0xfd, 0xae, 0xe7, 0x71, 0xf3, 0x50,
    0x11, 0x27, 0x1a, 0x85, 0xf0, 0xab, 0xd8, 0x16, 0x64, 0xcb,
    0xad, 0xbb, 0xae, 0x54, 0x37, 0xa3, 0xa8, 0xf4, 0x09, 0x67,
    0x54, 0x61, 0x86, 0x0f, 0x0e, 0x25, 0x0d, 0xda, 0x4a, 0xc7,
    0xe7, 0x02, 0x80, 0x6b, 0x59, 0xd2, 0xc8, 0x88, 0x4d, 0x7d,
    0xfd, 0x3d, 0x48, 0x04, 0x6d, 0x95, 0xdf, 0xc2, 0x8b, 0x23,
    0x70, 0x4a, 0xf5, 0xdc, 0xc9, 0x24, 0x8d, 0x7e, 0x52, 0x22,
    0x7e, 0x9c, 0x5c, 0x32, 0xa5, 0xd5, 0xf2, 0x11, 0x08, 0xa0,
    0xd4, 0xa2, 0xd8, 0xdb, 0x1d, 0x9f, 0x1b, 0x54, 0x8f, 0xb5,
    0xf6, 0x71, 0x71, 0x49, 0xbc, 0x38, 0x09, 0xb6, 0x24, 0x94,
    0x80, 0x1f, 0x2d, 0x0c, 0xc7, 0xe4, 0xd6, 0xcd, 0xab, 0x53,
    0x79, 0x28, 0xed, 0x48, 0x23, 0x14, 0x2f, 0x0b, 0x3a, 0xd0,
    0xa7, 0x08, 0xe1, 0xfd, 0x1e, 0xb6, 0xdd, 0x12, 0x93, 0x2d,
    0x95, 0x06, 0xba, 0x95, 0xcb, 0x1a, 0xed, 0xfb, 0x60, 0xe7,
    0xf1, 0x1c, 0xad, 0xc3, 0xea, 0x8d, 0x3c, 0x53, 0x32, 0xb5,
    0x38, 0x26, 0xdd, 0x39, 0xf0, 0x39, 0x4e, 0x6f, 0x3e, 0xa9,
    0xea, 0x25, 0x29, 0xb8, 0x6c, 0x7d, 0x0a, 0x91, 0xd4, 0xb9,
    0x7b, 0x67, 0xe4, 0xe5, 0x63, 0xd7, 0x6b, 0x03, 0xa5, 0xd7,
    0xe8, 0xd2, 0xc0, 0x34, 0x53, 0xa6, 0x16, 0x21, 0x2a, 0x2a,
    0x09, 0xd3, 0xad, 0xa1, 0x2c, 0x6a, 0x88, 0x2d, 0x90, 0x06,
    0xba, 0x0b, 0xaa, 0xd1, 0xdb, 0xa4, 0xd0, 0x49, 0x0f, 0x42,
    0xe1, 0xca, 0xf0, 0x69, 0x15, 0x63, 0xcb, 0x0b, 0x4c, 0x2e,
    0x99, 0x20, 0x44, 0xe3, 0x6e, 0x32, 0x8a, 0xa1, 0x5c, 0x5b,
    0x03, 0xeb, 0xb5, 0x05, 0xff, 0x1a, 0x76, 0x38, 0x1c, 0xb0,
    0x74, 0xf1, 0x5a, 0x0d, 0x8a, 0xd2, 0x4e, 0x38, 0x11, 0x86,
    0xb0, 0x2d, 0xd3, 0x88, 0xe2, 0x0f, 0x51, 0x68, 0xb9, 0x79,
    0x96, 0x50, 0x95, 0xdc, 0x69, 0xcb, 0xa6, 0x25, 0x4a, 0xdf,
    0xa1, 0x39, 0x13, 0x47, 0x0a, 0xf0, 0xeb, 0xcb, 0x14, 0x01,
    0x28, 0x9c, 0x0f, 0xe2, 0x62, 0xca, 0xb5, 0x40, 0x51, 0x45,
    0x8e, 0x18, 0x88, 0xc9, 0x58, 0xaf, 0xb3, 0x48, 0xd5, 0x20,
    0xe8, 0xd8, 0x5b, 0xa2, 0x98, 0x74, 0x25, 0xfa, 0x25, 0x19,
    0x82, 0x22, 0xfa, 0x82, 0x7c, 0x38, 0x8d, 0x62, 0x86, 0x01,
    0x63, 0x20, 0x36, 0x8e, 0xaf, 0x15, 0x8a, 0x74, 0x1e, 0xfd,
    0x7f, 0xbe, 0x60, 0xc3, 0x65, 0x31, 0xce, 0xdb, 0x92, 0xb9,
    0x13, 0x2a, 0x78, 0xa9, 0xfc, 0x6a, 0x7b, 0x18, 0xec, 0x0c,
    0x7b, 0x4c, 0x86, 0xaf, 0xea, 0x6d, 0x52, 0x09, 0x76, 0x52,
    0x87, 0x8a, 0x0b, 0x2a, 0xf3, 0x93, 0x35, 0x92, 0x8b, 0x60,
    0x42, 0x2e, 0x12, 0xa9, 0xf7, 0x7c, 0x61, 0x5c, 0x8f, 0xc0,
    0xaa, 0x6e, 0x6a, 0xf6, 0x48, 0x48, 0xc6, 0x3e, 0xe0, 0x1d,
    0xb4, 0xfb, 0xc4, 0xd8, 0x01, 0xb8, 0xf2, 0xf4, 0xdf, 0xc1,
    0xba, 0xb5, 0xf2, 0x27, 0x3f, 0xdb, 0x78, 0x62, 0x1c, 0x0a,
    0xbe, 0xdb, 0xdd, 0x3c, 0x0c, 0x29, 0x85, 0xf1, 0x44, 0x5f,
    0x2b, 0x43, 0x80, 0x57, 0xa7, 0x5a, 0x4d, 0x1b, 0xbe, 0x03,
    0xe7, 0x55, 0x7b, 0x91, 0x9d, 0x4c, 0x8b, 0xd7, 0xfd, 0xde,
    0x65, 0x7e, 0xa8, 0x48, 0xbb, 0xa9, 0x96, 0x06, 0x7f, 0xc0,
    0x6c, 0xed, 0x87, 0x53, 0x77, 0xb4, 0x5a, 0x7c, 0xbb, 0xce,
    0xcf, 0x01, 0x08, 0x45, 0x61, 0xc1, 0x28, 0xb6, 0xf2, 0xb4,
    0x5b, 0x6b, 0x84, 0xfe, 0x18, 0x09, 0x39, 0xc1, 0xc8, 0x96,
    0x36, 0x6e, 0xba, 0x7e, 0x48, 0x12, 0xe6, 0xdc, 0x22, 0x48,
    0x17, 0x0b, 0xbd, 0x92, 0x64, 0xfa, 0xc9, 0x9b, 0x07, 0xda,
    0xed, 0x04, 0x68, 0x42, 0x15, 0x8c, 0xf9, 0xd8, 0xc3, 0x0d,
    0x21, 0x9d, 0x96, 0xbc, 0xc3, 0x07, 0x1a, 0x2c, 0x59, 0x3f,
    0x1a, 0x83, 0x43, 0xf0, 0xe0, 0xde, 0xe3, 0x40, 0x8e, 0x04,
    0x66, 0x3c, 0x87, 0x1e, 0xfa, 0x7b, 0x8a, 0x7b, 0xd2, 0x9e,
    0x15, 0xf5, 0xec, 0x3c, 0x72, 0x7e, 0x2d, 0x19, 0xf8, 0xfd,
    0xf0, 0x28, 0x71, 0x8a, 0xf5, 0xcb, 0x4c, 0x61, 0x5f, 0x85,
    0xe0, 0x6f, 0xb8, 0xf3, 0x17, 0x10, 0xcb, 0x44, 0x45, 0x8c,
    0x96, 0x08, 0xa1, 0xf1, 0x48, 0xa4, 0x1d, 0xea, 0x35, 0x2f,
    0x82, 0x2b, 0xc2, 0x0b, 0xef, 0x73, 0xe1, 0xc2, 0x35, 0xdb,
    0xe7, 0x68, 0xfd, 0xb0, 0xe8, 0x7b, 0x2d, 0x0f, 0xfd, 0x53,
    0x1b, 0xb8, 0x36, 0x54, 0xd6, 0x43, 0x30, 0xcf, 0x83, 0xb0,
    0x18, 0xda, 0x9b, 0x86, 0x82, 0xfa, 0xe6, 0x37, 0x5b, 0x9e,
    0xa4, 0xdb, 0x7c, 0x59, 0x25, 0x59, 0xc6, 0x46, 0x36, 0x72,
    0xc5, 0x72, 0xd8, 0x2f, 0x26, 0xe2, 0xee, 0xe3, 0xcb, 0xe5,
    0x33, 0x1f, 0x18, 0x2e, 0x16, 0xce, 0xd2, 0x9c, 0x89, 0x6e,
    0xd5, 0x21, 0xfa, 0x58, 0x83, 0xa9, 0x4c, 0x69, 0x97, 0x7d,
    0xae, 0x1f, 0x65, 0xd5, 0xdb, 0xf0, 0xfe, 0xd5, 0x32, 0xb1,
    0x50, 0x72, 0xdf, 0x2b, 0xe2, 0xc1, 0xe6, 0x2e, 0x8b, 0x87,
    0xa8, 0x4e, 0x84, 0xbe, 0xc9, 0x27, 0xb5, 0x74, 0x7e, 0x13,
    0x17, 0x57, 0x9c, 0xc6, 0xd3, 0x9f, 0xcd, 0x86, 0x50, 0x4b,
    0x6c, 0x50, 0xa2, 0xba, 0xfe, 0xf6, 0xd5, 0x85, 0x68, 0x31,
    0x89, 0xfb, 0xeb, 0xfe, 0x92, 0xb0, 0xd0, 0x4c, 0xbc, 0x65,
    0x4b, 0x62, 0xe2, 0xdf, 0x88, 0x7e, 0x90, 0xe0, 0xb3, 0xec,
    0x13, 0x69, 0x33, 0xea, 0x53, 0x69, 0x9a, 0x0b, 0x27, 0xfb,
    0xca, 0x9f, 0x9e, 0x1f, 0xcf, 0xb1, 0xeb, 0xf4, 0x8f, 0xe2,
    0x53, 0xc8, 0xe6, 0x51, 0x75, 0xee, 0xb1, 0x34, 0x3e, 0x37,
    0xdd, 0x2d, 0x3a, 0x72, 0x76, 0x33, 0xc1, 0x27, 0xe7, 0xbd,
    0xc1, 0x7f, 0xcb, 0x53, 0x5d, 0xdf, 0xc4, 0x1f, 0x36, 0xdb,
    0x6a, 0x91, 0x1f, 0x6a, 0xa5, 0xc6, 0xe2, 0x37, 0x68, 0x1a,
    0x7d, 0xf7, 0xed, 0x2a, 0xc7, 0x99, 0x5e, 0xbd, 0x59, 0x57,
    0x09, 0x22, 0x7e, 0x9c, 0xbd, 0x8e, 0xad, 0xbe, 0xee, 0xa5,
    0x2a, 0xe3, 0x9f, 0xff, 0x14, 0xda, 0xba, 0x90, 0x37, 0xba,
    0x3a, 0x42, 0xcd, 0x4a, 0x28, 0x47, 0x27, 0x58, 0x7a, 0x33,
    0x93, 0x77, 0x83, 0x29, 0xab, 0x47, 0x19, 0x43, 0x00, 0x6f,
    0xe7, 0x77, 0xc1, 0xaa, 0xd6, 0xbc, 0xc0, 0x1b, 0xd0, 0xdf,
    0xf9, 0x40, 0x4d, 0xb2, 0x60, 0xce, 0x59, 0x17, 0x0a, 0xa9,
    0x14, 0x4e, 0x6a, 0x30, 0x1b, 0x26, 0x68, 0x55, 0x12, 0x19,
    0x62, 0x85, 0x5d, 0xa6, 0xb4, 0x48, 0x4a, 0xe9, 0xe1, 0x57,
    0xb1, 0x48, 0xf3, 0x86, 0xd1, 0x50, 0x2e, 0x1d, 0x57, 0xbe,
    0x09, 0xf8, 0x53, 0x40, 0xd9, 0x55, 0xd9, 0x71, 0x4c, 0xa7,
    0xdb, 0x61, 0x82, 0x4e, 0x00, 0x58, 0xe4, 0x89, 0xae, 0xa6,
    0x1a, 0x4b, 0xe3, 0x9d, 0xec, 0x65, 0xee, 0xe1, 0x7b, 0xdb,
    0x4f, 0x8d, 0xf3, 0xd9, 0x89, 0xaa, 0xd1, 0x31, 0x30, 0xde,
    0xc3, 0x5c, 0xbc, 0xb9, 0x60, 0x0a, 0xe0, 0x13, 0x14, 0x85,
    0x08, 0x60, 0xc5, 0x1c, 0xc2, 0x9d, 0x8b, 0x6e, 0xb8, 0x94,
    0x11, 0x6f, 0xd3, 0xee, 0xfb, 0xf8, 0x15, 0xd8, 0xa4, 0x0b,
    0x92, 0xdf, 0x7c, 0x9a, 0xa2, 0xec, 0xa3, 0x3d, 0xbc, 0xcd,
    0xe8, 0xb5, 0xb3, 0xf5, 0xe8, 0xee, 0x2a, 0x57, 0xf7, 0x58,
    0xc4, 0xaa, 0xeb, 0x33, 0x44, 0x5f, 0x62, 0xbe, 0x90, 0x48,
    0xe5, 0xcb, 0x6a, 0xcb, 0x55, 0x94, 0x6d, 0xe6, 0x22, 0x03,
    0xeb, 0xcb, 0x05, 0xb8, 0xb4, 0xa5, 0xbe, 0xec, 0x79, 0x21,
    0x0d, 0xb3, 0x5c, 0x74, 0x11, 0xcb, 0xb3, 0xa6, 0x06, 0x2f,
    0x73, 0xd1, 0x14, 0xd9, 0x70, 0x4e, 0xc5, 0xf5, 0xff, 0xfd,
    0x49, 0x3b, 0xa9, 0x22, 0x80, 0x2a, 0x5e, 0xf9, 0xae, 0xa5,
    0xd4, 0x3c, 0x74, 0xd7, 0x5a, 0x5d, 0x88, 0x6f, 0x99, 0xe2,
    0x4c, 0xa3, 0x9b, 0x15, 0xb8, 0xfd, 0x0b, 0x0d, 0x57, 0x03,
    0xe8, 0xda, 0x78, 0xc4, 0x63, 0x49, 0x48, 0x7a, 0x39, 0xcd,
    0xfa, 0xad, 0x92, 0x55, 0x4a, 0x0e, 0x68, 0x08, 0xb9, 0x34,
    0xe0, 0x14, 0x6e, 0x19, 0xed, 0x69, 0x14, 0x7f, 0xc1, 0x7d,
    0x12, 0xac, 0x5d, 0xf7, 0x62, 0x6f, 0x77, 0x65, 0xa3, 0xc2,
    0xf9, 0xda, 0x43, 0x9e, 0x6b, 0x82, 0xd9, 0x14, 0x57, 0x02,
    0x09, 0x9f, 0xa7, 0x15, 0x27, 0xe8, 0xad, 0xa1, 0x73, 0xc7,
    0xb6, 0x11, 0x4c, 0x5e, 0xf4, 0x1a, 0x0a, 0x97, 0x98, 0x5e,
    0x29, 0x8a, 0x8b, 0xa5, 0xbd, 0x86, 0x7f, 0x6d, 0x31, 0x72,
    0x6d, 0xe5, 0xcf, 0x13, 0xff, 0xb9, 0x4e, 0x69, 0x66, 0x37,
    0x1b, 0xfb, 0xe8, 0xb7, 0x60, 0xfe, 0xbf, 0xaa, 0x06, 0x88,
    0xa4, 0xa2, 0x0b, 0x33, 0x55, 0xac, 0x61, 0x77, 0x0a, 0x6f,
    0x1f, 0xaf, 0xd8, 0x9b, 0xc7, 0x26, 0x13, 0xf6, 0xc4, 0xef,
    0xce, 0x0f, 0x16, 0x86, 0x64, 0x1b, 0xc0, 0x71, 0x35, 0xf9,
    0x1f, 0xaf, 0xc4, 0x7a, 0xa3, 0x3b, 0x89, 0x40, 0xcb, 0x09,
    0x11, 0x7b, 0x01, 0x54, 0xd5, 0xd2, 0x2a, 0xc8, 0xfe, 0x0e,
    0xef, 0x8c, 0xfb, 0x2b, 0x08, 0x12, 0x6d, 0xbb, 0xa8, 0x2e,
    0x7a, 0x2b, 0xc2, 0x91, 0x2a, 0x76, 0x0b, 0x31, 0x30, 0x4a,
    0x5b, 0xca, 0x96, 0xc9, 0x89, 0xa0, 0x12, 0x40, 0x76, 0xbe,
    0xcd, 0x59, 0x5f, 0xc2, 0x7b, 0xaf, 0xf6, 0x29, 0xde, 0xe9,
    0x24, 0x61, 0x3f, 0x46, 0x78, 0xa7, 0xda, 0x65, 0xb0, 0xb3,
    0xae, 0xf3, 0x72, 0x6e, 0x37, 0x6e, 0xae, 0xb1, 0x3b, 0xf6,
    0x60, 0xa1, 0x92, 0x86, 0x9e, 0x97, 0x4f, 0x5e, 0x86, 0x88,
    0x32, 0x06, 0x7c, 0xe3, 0x37, 0x7e, 0xb1, 0x83, 0xf5, 0x83,
    0x05, 0x43, 0xb3, 0xe3, 0xa1, 0x68, 0xe5, 0x4c, 0x92, 0x9c,
    0x61, 0xa3, 0x5d, 0xcf, 0x23, 0xe7, 0xce, 0xf5, 0x7f, 0xbb,
    0xf7, 0x89, 0x5e, 0xa8, 0xf0, 0xa1, 0xff, 0x1a, 0xaf, 0x15,
    0xc8, 0x3d, 0x8b, 0xce, 0x06, 0xa4, 0x60, 0xd6, 0x40, 0x19,
    0x48, 0x33, 0x53, 0x34, 0x9e, 0xd8, 0x75, 0xfc, 0x45, 0x73,
    0x35, 0x8f, 0x70, 0x04, 0x80, 0xa1, 0xe5, 0xfc, 0x98, 0xb0,
    0x52, 0x63, 0x41, 0x84, 0x57, 0xa2, 0x85, 0x4e, 0x68, 0x13,
    0x2d, 0x3e, 0x4b, 0x68, 0x7f, 0x43, 0x04, 0x05, 0x02, 0x5a,
    0x16, 0x67, 0x5a, 0xc5, 0xea, 0xac, 0x25, 0x61, 0xd4, 0xa4,
    0xe7, 0xbe, 0x13, 0x95, 0xbd, 0x03, 0xb4, 0x26, 0xe3, 0xbf,
    0x7e, 0xe5, 0x0b, 0x34, 0xeb, 0x59, 0x5d, 0xd7, 0xdb, 0x1e,
    0x07, 0xfc, 0x63, 0xab, 0xbb, 0xc6, 0x7a, 0x51, 0x50, 0x59,
    0x13, 0x4b, 0x27, 0x88, 0x98, 0xdc, 0x01, 0x37, 0xeb, 0x58,
    0x75, 0xde, 0x5a, 0xa4, 0x6b, 0xdd, 0xba, 0x01, 0x40, 0xf7,
    0x1c, 0x0a, 0xf3, 0x02, 0x3d, 0x54, 0x64, 0xf2, 0x85, 0x43,
    0x90, 0xc0, 0x69, 0x18, 0x94, 0x95, 0x6e, 0x57, 0x14, 0xda,
    0x27, 0x0a, 0x42, 0xb2, 0x5a, 0x78, 0xe4, 0xf1, 0x45, 0x85,
    0x54, 0xec, 0x44, 0xa0, 0xcb, 0xf4, 0xd1, 0x3a, 0x85, 0x74,
    0x0f, 0x04, 0x67, 0xf4, 0x42, 0x01, 0xc4, 0x04, 0x66, 0x48,
    0x6c, 0xbe, 0x84, 0x38, 0x6e, 0xda, 0x23, 0xd0, 0xd1, 0x26,
    0x94, 0x11, 0x65, 0x2e, 0xc6, 0xd8, 0x6e, 0x25, 0x17, 0x43,
    0x9f, 0x55, 0x2d, 0x1d, 0x55, 0xa9, 0xdd, 0x3b, 0xc7, 0x09,
    0xde, 0x26, 0x64, 0xd4, 0x85, 0x21, 0x15, 0x0d, 0x4a, 0x45,
    0x4d, 0xba, 0x13, 0x9e, 0x3b, 0x5e, 0xc2, 0xf7, 0xc1, 0x34,
    0xc5, 0x74, 0xd4, 0x95, 0x19, 0x3d, 0x69, 0x9c, 0xae, 0xef,
    0x13, 0x95, 0x2c, 0x77, 0xdd, 0x64, 0x2c, 0x12, 0x31, 0x7d,
    0xb5, 0x55, 0xde, 0x69, 0x35, 0x3f, 0x77, 0x72, 0xc6, 0x21,
    0x22, 0x23, 0x7a, 0x05, 0xbf, 0x92, 0xae, 0x49, 0x7f, 0x74,
    0x17, 0x97, 0x5f, 0x5b, 0x4d, 0x7d, 0x86, 0x23, 0x04, 0xe0,
    0xff, 0x10, 0x06, 0xc3, 0xd3, 0x05, 0xde, 0xc4, 0xae, 0xaf,
    0x3d, 0x2d, 0xaf, 0x3c, 0xaf, 0xd3, 0xd5, 0xfd, 0x84, 0xd8,
    0x3b, 0x6c, 0x8e, 0x8b, 0x23, 0x8b, 0x16, 0xaa, 0x67, 0xf1,
    0xde, 0xa4, 0x4b, 0x5a, 0x39, 0x60, 0x73, 0xd2, 0x9f, 0x1f,
    0x8c, 0xcf, 0xbc, 0xaa, 0x74, 0x9e, 0x8d, 0xfd, 0xc3, 0xb7,
    0x86, 0xe5, 0xbb, 0x5a, 0x4d, 0x3d, 0xe2, 0xc3, 0x28, 0x78,
    0x26, 0xd4, 0xb3, 0x45, 0x94, 0xd3, 0x2d, 0xbf, 0x8c, 0x92,
    0x56, 0x3c, 0x6e, 0xea, 0x53, 0x38, 0x7f, 0x22, 0x67, 0xc9,
    0xa7, 0x14, 0x20, 0xb9, 0x13, 0xc4, 0xa0, 0x44, 0x83, 0xc4,
    0x19, 0xca, 0x98, 0x71, 0xc7, 0x13, 0x70, 0x3a, 0xa7, 0xfb,
    0x9e, 0xc4, 0x94, 0x8c, 0xfd, 0x21, 0x36, 0x88, 0xea, 0x23,
    0xc7, 0x43, 0x52, 0x9f, 0xf4, 0x9e, 0xb1, 0xb4, 0xd3, 0x20,
    0x65, 0xd8, 0x18, 0x25, 0x80, 0xb7, 0xe4, 0x5c, 0x96, 0x3a,
    0xa3, 0xb5, 0x40, 0x63, 0xac, 0x02, 0x34, 0x51, 0xf7, 0x12,
    0xea, 0x97, 0x9d, 0x3e, 0xe7, 0xcb, 0x88, 0x15, 0xaa, 0xe3,
    0xfe, 0xe5, 0x42, 0xe5, 0x48, 0xcf, 0xc6, 0x8e, 0x0e, 0xc6,
    0x48, 0xdb, 0xe5, 0x1e, 0x79, 0x99, 0xed, 0x78, 0xa6, 0x37,
    0xdd, 0xe3, 0x7b, 0x01, 0xdd, 0x20, 0x63, 0x45, 0x57, 0xd1,
    0x0f, 0x05, 0x5d, 0x29, 0xad, 0x99, 0x6c, 0x27, 0xa3, 0x0c,
    0x72, 0x81, 0xb1, 0x26, 0x16, 0xaf, 0x11, 0x65, 0xba, 0x79,
    0xbc, 0xb8, 0xfe, 0xe7, 0xc5, 0xe6, 0x4c, 0xfa, 0x37, 0xc5,
    0xe0, 0x2e, 0x4e, 0xef, 0x75, 0xe4, 0x04, 0xaf, 0xfa, 0x41,
    0x7f, 0x58, 0x2e, 0x8f, 0x95, 0x5f, 0x15, 0x5c, 0x15, 0x23,
    0x81, 0xb7, 0x2c, 0x81, 0x70, 0xf5, 0xcc, 0x60, 0x09, 0x7e,
    0xf1, 0x0d, 0x9c, 0x9d, 0xcc, 0xa0, 0x30, 0xa8, 0x82, 0x23,
    0x5f, 0x94, 0xcb, 0x18, 0xc4, 0x32, 0xe6, 0xab, 0xcd, 0x96,
    0x9e, 0xab, 0xcd, 0x68, 0x6f, 0x88, 0xb7, 0x72, 0x65, 0xbc,
    0x1e, 0x05, 0x60, 0xfe, 0x6b, 0x77, 0x2a, 0x11, 0x63, 0x59,
    0x29, 0xdb, 0xba, 0xe0, 0x50, 0xd5, 0x51, 0x77, 0x16, 0xb8,
    0xb7, 0xf4, 0xa9, 0xbe, 0xf0, 0xa5, 0xaa, 0x20, 0x50, 0x2e,
    0x73, 0x21, 0xee, 0x77, 0xa3, 0xc8, 0xbc, 0x0c, 0x16, 0x0f,
    0x83, 0x7b, 0xaf, 0xbb, 0x91, 0x95, 0xd3, 0x6e, 0xe7, 0x28,
    0x77, 0x00, 0xbc, 0x83, 0x46, 0xa5, 0x0a, 0x19, 0xe8, 0x10,
    0xfb, 0x24, 0xeb, 0x27, 0xc2, 0xa3, 0xdd, 0xb8, 0x5b, 0x27,
    0xb9, 0xbb, 0x49, 0xd9, 0xd0, 0x32, 0x94, 0x48, 0x1b, 0xb8,
    0xf8, 0xb2, 0x30, 0xf4, 0x1f, 0x3d, 0xbf, 0xe6, 0xf3, 0x34,
    0xd3, 0x32, 0x85, 0x67, 0x85, 0x13, 0x3e, 0x20, 0xb7, 0xfa,
    0x74, 0x27, 0x74, 0x8f, 0x55, 0x47, 0x15, 0x91, 0x0b, 0x3f,
    0xb1, 0x18, 0xe7, 0x11, 0x1e, 0x52, 0xd8, 0xd1, 0x3f, 0xb9,
    0x5d, 0x4f, 0x88, 0xb9, 0x1e, 0x5a, 0xb6, 0x90, 0x64, 0xad,
    0x6f, 0x8d, 0x33, 0xb3, 0x57, 0xde, 0x3e, 0x13, 0xb3, 0x9f,
    0x2d, 0x00, 0xb1, 0x79, 0x84, 0x60, 0x6d, 0x3c, 0x5f, 0xc0,
    0x34, 0x08, 0x4b, 0x58, 0x33, 0x59, 0xfe, 0xe5, 0xed, 0xd3,
    0x10, 0xd8, 0xd8, 0x85, 0xc3, 0xc9, 0x71, 0xcf, 0x40, 0x96,
    0xc0, 0xd5, 0x5e, 0x62, 0xe7, 0xcb, 0x33, 0xee, 0x72, 0xb5,
    0xb8, 0x6e, 0xea, 0x13, 0xde, 0xeb, 0x82, 0x03, 0x8e, 0x6c,
    0xb3, 0x67, 0xb1, 0x5f, 0xd4, 0xe1, 0xd9, 0xc2, 0x7a, 0x97,
    0xbb, 0xd4, 0x5e, 0x0b, 0xfe, 0xc1, 0xb3, 0x1f, 0x2b, 0x1a,
    0x37, 0x98, 0x26, 0x27, 0xb1, 0xaf, 0x4c, 0x55, 0xe1, 0xae,
    0x4c, 0x86, 0x80, 0x4b, 0xc5, 0xf2, 0x35, 0x48, 0x81, 0xf7,
    0x83, 0x75, 0x63, 0x08, 0x0d, 0x77, 0x41, 0x14, 0xbc, 0xf3,
    0x6e, 0x46, 0xbd, 0x9c, 0x5a, 0x4f, 0x5c, 0x89, 0x26, 0xb6,
    0x6c, 0xde, 0x0d, 0x15, 0x31, 0xec, 0x7e, 0x13, 0xf2, 0x99,
    0x74, 0x40, 0x3c, 0xe1, 0xea, 0xa0, 0xc9, 0x99, 0x0a, 0x4b,
    0x17, 0x74, 0xff, 0x47, 0x15, 0x76, 0x5e, 0x44, 0xa2, 0x1c,
    0x93, 0xd3, 0xe6, 0xa2, 0x82, 0x0f, 0x7f, 0x55, 0xa8, 0xf3,
    0x79, 0xc3, 0xa8, 0x9f, 0x37, 0x2b, 0x97, 0x7e, 0x90, 0x71,
    0xfc, 0xa7, 0xff, 0xc6, 0xc7, 0x93, 0x5c, 0xc9, 0xed, 0x20,
    0x60, 0xbd, 0x5c, 0x36, 0x05, 0x55, 0x51, 0x55, 0x51, 0x15,
    0x36, 0x01, 0x17, 0xa9, 0x56, 0x27, 0x44, 0x66, 0xc9, 0x3a,
    0xb9, 0xbb, 0xee, 0x04, 0xb6, 0x2a, 0xfd, 0x10, 0x9a, 0x46,
    0xdd, 0x5d, 0x6d, 0xad, 0x21, 0x86, 0x6d, 0x62, 0x8a, 0x4a,
    0xbc, 0x73, 0xf0, 0x9d, 0x93, 0x0d, 0xf1, 0x62, 0xfa, 0x58,
    0x64, 0x37, 0x4f, 0x0b, 0xa3, 0xa1, 0x52, 0xce, 0x03, 0xce,
    0x0f, 0x77, 0x29, 0xad, 0x47, 0x38, 0xca, 0xbc, 0x61, 0xe6,
    0xad, 0xe4, 0x8b, 0xf1, 0x82, 0xa8, 0xd5, 0xe3, 0x8c, 0xd3,
    0xa0, 0xc4, 0xc0, 0x5e, 0x3b, 0xa1, 0x66, 0x2a, 0x6e, 0x88,
    0x24, 0x56, 0xe4, 0x84, 0x0a, 0x36, 0x72, 0xf3, 0x5c, 0x11,
    0xd9, 0x66, 0xd8, 0x45, 0x5c, 0x83, 0x9e, 0x1c, 0x8c, 0xc6,
    0xf6, 0x6e, 0x6a, 0xb1, 0x52, 0xed, 0x6c, 0x6a, 0x6d, 0x23,
    0xb9, 0x0b, 0x66, 0x26, 0x5a, 0x16, 0x16, 0x90, 0x43, 0xb9,
    0xc3, 0x02, 0xc1, 0x43, 0x93, 0x13, 0x94, 0xfe, 0xc3, 0x59,
    0x49, 0xbe, 0x1e, 0x26, 0x1b, 0x9d, 0x8e, 0xba, 0xc4, 0x29,
    0x51, 0x05, 0x28, 0x1f, 0x55, 0x59, 0x1c, 0x3e, 0x25, 0x86,
    0xcc, 0xc7, 0xd9, 0xd3, 0xa8, 0xe7, 0x10, 0xa0, 0xb6, 0x23,
    0xb9, 0xaf, 0x00, 0x8b, 0x7d, 0xf1, 0x5b, 0xd6, 0xb7, 0x56,
    0x44, 0x9b, 0x0a, 0xec, 0xa6, 0x2b, 0xb4, 0x4e, 0x1d, 0x4f,
    0xc5, 0x0b, 0x45, 0xd2, 0x3a, 0xc5, 0xc0, 0xbf, 0xb9, 0xdd,
    0x59, 0x21, 0xf2, 0x67, 0x25, 0x88, 0x9b, 0xb6, 0x66, 0x83,
    0xbf, 0x62, 0xfe, 0x7c, 0xfa, 0x9e, 0x50, 0xed, 0x15, 0x93,
    0xb6, 0x7a, 0xb0, 0xc4, 0xbe, 0xcf, 0x2a, 0x70, 0x4e, 0x52,
    0x20, 0xc1, 0x24, 0x08, 0x49, 0xd9, 0x05, 0x04, 0x53, 0x73,
    0xf3, 0xcf, 0x14, 0x70, 0xac, 0x3c, 0x45, 0x0f, 0x08, 0xa3,
    0xae, 0x43, 0xe7, 0x7f, 0x1f, 0xe2, 0x14, 0xf1, 0xbb, 0x25,
    0x20, 0xfd, 0xe4, 0xaf, 0x44, 0x9e, 0x77, 0x88, 0x4d, 0x26,
    0x09, 0xb1, 0xb0, 0x12, 0xf5, 0xdf, 0x3c, 0x53, 0x48, 0x78,
    0xb9, 0x60, 0x41, 0xd3, 0x8f, 0x8d, 0x11, 0x63, 0x60, 0x28,
    0x30, 0x07, 0xa2, 0x14, 0x3b, 0x8c, 0x50, 0xe2, 0xee, 0x73,
    0x39, 0x66, 0xd1, 0x51, 0x87, 0xac, 0x90, 0x9b, 0x2c, 0x6d,
    0x8d, 0xd5, 0x75, 0x3f, 0xc6, 0xf1, 0x8f, 0xdf, 0xdb, 0x45,
    0x38, 0xf8, 0xd6, 0x7e, 0xc7, 0x7c, 0x44, 0x08, 0x4a, 0x14,
    0xa0, 0x84, 0x7c, 0x8b, 0x88, 0x40, 0x93, 0x89, 0xae, 0x2c,
    0x20, 0x07, 0x80, 0xec, 0xce, 0x4c, 0x2c, 0x4e, 0x49, 0x79,
    0x53, 0xe7, 0xde, 0xa2, 0x9e, 0x67, 0x21, 0x53, 0x7c, 0x85,
    0xe7, 0x6f, 0xbd, 0x93, 0xab, 0x63, 0xba, 0xf0, 0xbd, 0xea,
    0x39, 0x16, 0x47, 0xbf, 0xe6, 0x0c, 0xcb, 0x63, 0xc7, 0xc5,
    0xf1, 0xdc, 0x5a, 0x52, 0xcd, 0x4c, 0x53, 0x8b, 0x7e, 0xb1,
    0xc3, 0x4e, 0xe7, 0x61, 0x25, 0x01, 0xec, 0xae, 0x06, 0x74,
    0x9f, 0xbc, 0xbb, 0x2a, 0x47, 0x46, 0xe8, 0xae, 0xf2, 0xab,
    0x15, 0xed, 0xa6, 0x86, 0x8f, 0x2f, 0xe5, 0x67, 0x0f, 0xdd,
    0xbf, 0x70, 0x53, 0xaa, 0x9b, 0x74,
};
static const int sizeof_bench_dilithium_level5_key = sizeof(bench_dilithium_level5_key);

#endif /* !WOLFSSL_DILITHIUM_NO_SIGN */

#ifndef WOLFSSL_DILITHIUM_NO_VERIFY

/* raw public key without ASN1 syntax from
 * ./certs/dilithium/bench_dilithium_level5_key.der */
static const unsigned char bench_dilithium_level5_pubkey[] = {
    0xef, 0x49, 0x79, 0x47, 0x15, 0xc4, 0x8a, 0xa9, 0x74, 0x2a,
    0xf0, 0x36, 0x94, 0x5c, 0x91, 0x1c, 0x5d, 0xff, 0x2c, 0x83,
    0xf2, 0x8b, 0x04, 0xfc, 0x5d, 0x64, 0xbd, 0x49, 0x73, 0xcd,
    0xcc, 0x99, 0x50, 0x5f, 0x2b, 0x16, 0x3a, 0xbb, 0x98, 0xc0,
    0xa7, 0x69, 0x0e, 0x95, 0x99, 0x0b, 0xa2, 0x6c, 0xfe, 0x6c,
    0xdb, 0xc8, 0xa7, 0x09, 0x46, 0x6c, 0x90, 0x50, 0xa4, 0x75,
    0x30, 0xf7, 0x90, 0xac, 0x31, 0xb6, 0xdd, 0x21, 0xaf, 0xc6,
    0xf9, 0xfe, 0xee, 0xc6, 0x5b, 0xa8, 0x8f, 0x0a, 0x2e, 0xd0,
    0x42, 0xab, 0xa8, 0x3c, 0x8d, 0xbf, 0xf7, 0x44, 0xbd, 0x0d,
    0xcf, 0xf4, 0x68, 0xfc, 0x16, 0x67, 0xf7, 0x39, 0x48, 0x5f,
    0x56, 0xd1, 0xe7, 0x1f, 0x49, 0x80, 0x50, 0xbe, 0x54, 0xd1,
    0xb7, 0xc9, 0xd2, 0x32, 0xc7, 0x08, 0x8c, 0xde, 0x2c, 0x31,
    0xf6, 0x1d, 0xc7, 0xac, 0xb3, 0x79, 0xd7, 0x4b, 0x1b, 0x23,
    0x89, 0x0a, 0xdc, 0x8e, 0x44, 0x41, 0x14, 0x28, 0x99, 0x13,
    0xb3, 0x26, 0xa6, 0x0e, 0x83, 0x60, 0xaa, 0x8d, 0x7c, 0x23,
    0x13, 0xba, 0x6c, 0x28, 0x90, 0x56, 0x84, 0xa1, 0x23, 0x8b,
    0x81, 0x20, 0x97, 0x7c, 0x66, 0x3f, 0xed, 0x5d, 0xd0, 0xe4,
    0x5d, 0xee, 0x46, 0xbc, 0x4b, 0x3c, 0x03, 0xb5, 0xbc, 0x4d,
    0x8d, 0x37, 0xa3, 0x56, 0x4b, 0x33, 0xad, 0xef, 0xd4, 0xb6,
    0xec, 0xdb, 0x04, 0x9a, 0x19, 0x58, 0x57, 0xd8, 0x00, 0x3a,
    0x92, 0x61, 0x0c, 0x0b, 0xc8, 0x52, 0xe5, 0x04, 0x02, 0x9a,
    0x00, 0x7e, 0xec, 0x7e, 0x94, 0xaa, 0xef, 0x2d, 0x7f, 0xb6,
    0x2e, 0x7c, 0xb0, 0x73, 0xa2, 0x20, 0xc0, 0x07, 0x30, 0x41,
    0x50, 0x20, 0x14, 0x18, 0x21, 0x5e, 0x2a, 0x6f, 0x70, 0x21,
    0xd6, 0x97, 0x13, 0xb9, 0xc1, 0x9e, 0x90, 0x67, 0xcc, 0x55,
    0x8a, 0xec, 0xec, 0x0a, 0x1e, 0x90, 0xdc, 0x3f, 0xb0, 0x4d,
    0xd1, 0x18, 0xea, 0x4f, 0xcb, 0x5d, 0x15, 0x4c, 0xb8, 0x35,
    0x9b, 0x34, 0x24, 0x30, 0x06, 0x53, 0x17, 0xf0, 0xbe, 0x27,
    0x36, 0xb3, 0x04, 0x6a, 0xbd, 0xbf, 0xa7, 0x39, 0xee, 0xa9,
    0x8f, 0x0e, 0x98, 0xc5, 0xf5, 0x9f, 0x46, 0x25, 0x93, 0xc9,
    0xf2, 0xf6, 0x2b, 0x8e, 0x92, 0x06, 0x01, 0x3d, 0x81, 0x18,
    0xf2, 0xec, 0xf1, 0x05, 0x4c, 0xad, 0x4b, 0xcb, 0x98, 0xa4,
    0xb5, 0x61, 0x20, 0xda, 0x81, 0xa1, 0xfb, 0x92, 0x4c, 0xaf,
    0x87, 0x6f, 0x6e, 0xd2, 0x57, 0xec, 0xcd, 0x94, 0xb3, 0x79,
    0xbf, 0x59, 0x88, 0x17, 0x81, 0xce, 0x8a, 0x57, 0xce, 0x57,
    0xae, 0x3e, 0x82, 0x81, 0x2f, 0x83, 0x61, 0xd8, 0xf9, 0x68,
    0x21, 0xe7, 0x72, 0x5b, 0xd6, 0x80, 0x55, 0x68, 0x5d, 0x67,
    0x15, 0x0c, 0x8b, 0xdc, 0x4f, 0xc3, 0x89, 0x36, 0x3c, 0xac,
    0xaf, 0x16, 0x5e, 0x1c, 0xfa, 0x68, 0x74, 0x6a, 0xab, 0x68,
    0xd8, 0x59, 0x96, 0x2d, 0x33, 0x62, 0xe4, 0xbd, 0xb3, 0xb7,
    0x4d, 0x88, 0x35, 0xb8, 0xed, 0xb2, 0x16, 0x85, 0x97, 0x08,
    0x71, 0x71, 0x39, 0x7e, 0x0c, 0x53, 0x16, 0xda, 0x38, 0xe5,
    0x28, 0x09, 0x9c, 0xd9, 0x46, 0xec, 0x68, 0xda, 0x8d, 0xd0,
    0xad, 0xb2, 0x79, 0x28, 0x3b, 0x1e, 0x12, 0xc9, 0xdf, 0xa9,
    0x6d, 0x3d, 0x29, 0x99, 0x2f, 0x53, 0xc2, 0xd0, 0xf9, 0x88,
    0x26, 0x94, 0x47, 0xaf, 0xf6, 0x96, 0xf3, 0xe1, 0x11, 0xa6,
    0x82, 0x3d, 0x43, 0x3f, 0x1f, 0xbc, 0xf6, 0x98, 0xbe, 0xff,
    0x06, 0x86, 0x61, 0x27, 0xdc, 0x91, 0x54, 0xd4, 0xfc, 0x68,
    0x83, 0xe8, 0x35, 0x3e, 0xee, 0x94, 0x59, 0x28, 0x2f, 0xde,
    0xdd, 0x03, 0x60, 0x66, 0xc1, 0x49, 0x57, 0xdd, 0xbc, 0xd5,
    0x0a, 0x67, 0x34, 0xf1, 0xa6, 0x0a, 0x57, 0x94, 0x65, 0x02,
    0x2c, 0x52, 0x43, 0x70, 0x3b, 0xc1, 0x9a, 0xff, 0xda, 0x6f,
    0xb9, 0x54, 0x47, 0x01, 0xda, 0x27, 0xe4, 0x48, 0x4a, 0x90,
    0x9f, 0xb5, 0xc3, 0xee, 0x0e, 0x09, 0x57, 0xfe, 0x48, 0x51,
    0x08, 0x34, 0x5e, 0x8f, 0x16, 0xc9, 0x0b, 0x74, 0xd9, 0x7d,
    0x22, 0x3f, 0xd6, 0xb7, 0x5d, 0xd6, 0x76, 0x00, 0x8d, 0x4e,
    0x78, 0x73, 0x86, 0xd6, 0xdb, 0x2a, 0x65, 0xab, 0xdf, 0xb0,
    0xea, 0x11, 0xad, 0xdf, 0xba, 0x43, 0xdb, 0xa8, 0x0a, 0xfb,
    0x04, 0x38, 0x81, 0x2b, 0xa3, 0x29, 0xfc, 0x95, 0x73, 0x9a,
    0x0c, 0x6c, 0x9e, 0xcd, 0xdc, 0xcf, 0x0a, 0x0c, 0x18, 0x41,
    0x6f, 0x1d, 0xa3, 0xf6, 0x12, 0x4c, 0x13, 0xf2, 0x02, 0xc6,
    0x50, 0x99, 0x86, 0x73, 0xa7, 0xf9, 0x7e, 0x84, 0x7f, 0x4c,
    0x00, 0xce, 0x2e, 0x21, 0x76, 0x8e, 0x17, 0x7a, 0x87, 0x6f,
    0x81, 0xe6, 0xc0, 0x52, 0xa5, 0xa0, 0x3c, 0x54, 0x3c, 0xec,
    0xb0, 0x9d, 0x1c, 0x3b, 0xec, 0xe5, 0x4e, 0x4a, 0x37, 0xe7,
    0xd5, 0xa9, 0x07, 0x87, 0x23, 0x28, 0x5d, 0x3d, 0x22, 0x02,
    0x79, 0x40, 0x3f, 0x2d, 0x40, 0xc9, 0xe5, 0xa6, 0x9b, 0xa8,
    0xb8, 0x76, 0xf6, 0x77, 0x5b, 0x8d, 0x72, 0x96, 0x3e, 0x13,
    0xbf, 0x76, 0xfa, 0x7b, 0xb7, 0x82, 0x5f, 0xe7, 0x9d, 0x54,
    0x0e, 0x05, 0x1a, 0x9f, 0xa4, 0x42, 0xa5, 0xb4, 0x93, 0x23,
    0x06, 0x59, 0x43, 0xa8, 0xe8, 0x5c, 0xfc, 0x18, 0x97, 0xdb,
    0xad, 0x9a, 0x80, 0x0a, 0xf2, 0x20, 0x50, 0xac, 0xc1, 0x13,
    0x3e, 0x98, 0x09, 0xde, 0xf2, 0x70, 0x9e, 0x14, 0xc2, 0x5c,
    0xec, 0x65, 0x07, 0x0b, 0xfa, 0x02, 0x5c, 0xf8, 0x71, 0xaa,
    0x9b, 0x45, 0x62, 0xe2, 0x27, 0xaf, 0x77, 0xf8, 0xe3, 0xeb,
    0x7b, 0x24, 0x7b, 0x3c, 0x67, 0xc2, 0x6d, 0x6e, 0x17, 0xae,
    0x6e, 0x86, 0x6f, 0x98, 0xc9, 0xac, 0x13, 0x9f, 0x87, 0x64,
    0x3d, 0x4d, 0x6f, 0xa0, 0xb3, 0x39, 0xc6, 0x68, 0x1b, 0xa7,
    0xeb, 0x3e, 0x0f, 0x6b, 0xc7, 0xa4, 0xe2, 0x20, 0x27, 0x75,
    0x3f, 0x09, 0x16, 0xff, 0x1a, 0xcc, 0xa7, 0xc4, 0x6d, 0xc2,
    0xfc, 0xc3, 0x0b, 0x37, 0x63, 0xff, 0x9b, 0x10, 0xe6, 0x00,
    0xf7, 0x18, 0x43, 0x9f, 0x07, 0x50, 0x31, 0x51, 0xd4, 0xfd,
    0xad, 0xa2, 0x0f, 0x77, 0xda, 0x41, 0xc1, 0x0a, 0x6f, 0x86,
    0xd7, 0xdc, 0x8a, 0x52, 0xd6, 0xa1, 0x27, 0xdb, 0x14, 0x67,
    0x26, 0x91, 0xb3, 0xcd, 0x01, 0x5f, 0x60, 0xa1, 0x7f, 0x43,
    0x15, 0x1a, 0x82, 0x0f, 0xd3, 0x66, 0x5f, 0x60, 0x57, 0x2f,
    0xb2, 0x8c, 0x27, 0x2a, 0x9d, 0x1b, 0xf9, 0xf2, 0x59, 0x20,
    0x39, 0xd9, 0xc5, 0xaf, 0xf2, 0x36, 0x8c, 0x58, 0x00, 0x1b,
    0xd0, 0xc5, 0x8e, 0x1a, 0x49, 0xa8, 0x60, 0xbe, 0xd1, 0xd7,
    0x2a, 0xb0, 0xc2, 0xab, 0x58, 0x8a, 0x7a, 0xa9, 0x41, 0x68,
    0x70, 0xbd, 0xea, 0x73, 0xa5, 0x03, 0x11, 0xb2, 0x27, 0xd9,
    0xcd, 0xf5, 0x09, 0xe8, 0x1c, 0xe2, 0x4f, 0x50, 0x6a, 0x84,
    0x34, 0x62, 0x2e, 0x36, 0xaa, 0x4c, 0xc1, 0x83, 0x78, 0x98,
    0x35, 0x7a, 0x27, 0x7e, 0xfe, 0xf1, 0x6f, 0x59, 0x27, 0x35,
    0x73, 0xce, 0x74, 0xaa, 0xb4, 0x72, 0x82, 0xa8, 0xe2, 0x81,
    0x7a, 0x6b, 0xca, 0x33, 0xa5, 0xda, 0xa2, 0x63, 0xca, 0x2e,
    0x90, 0x03, 0x32, 0xec, 0x63, 0xdb, 0x52, 0x7b, 0x16, 0xfc,
    0x01, 0x2d, 0x30, 0x12, 0x1e, 0xf9, 0xa3, 0x72, 0x21, 0x3c,
    0x75, 0x0c, 0x61, 0x9c, 0x7e, 0x73, 0x04, 0x71, 0x41, 0x45,
    0x5d, 0x7f, 0x49, 0x1c, 0x09, 0x08, 0xa4, 0xec, 0x2f, 0xfd,
    0xc4, 0xfb, 0x59, 0x6a, 0x27, 0x7a, 0xd4, 0xfc, 0x5f, 0x20,
    0x04, 0x34, 0x7d, 0x08, 0xed, 0x82, 0x5a, 0x90, 0xe1, 0xab,
    0xfd, 0x35, 0x3a, 0x8d, 0xbb, 0x0a, 0x9d, 0x73, 0xff, 0x69,
    0xe5, 0xe9, 0x09, 0x55, 0x14, 0xd9, 0x7b, 0x6f, 0x0d, 0x99,
    0xd2, 0x7e, 0x71, 0xf8, 0x4f, 0x72, 0x2f, 0xbb, 0xc6, 0xc4,
    0x36, 0xc9, 0x01, 0xd3, 0x9b, 0x94, 0xab, 0x41, 0x0f, 0x4a,
    0x61, 0x5c, 0x68, 0xe5, 0xd7, 0x0d, 0x94, 0xaa, 0xee, 0xba,
    0x95, 0xcb, 0x8c, 0x0e, 0x85, 0x3a, 0x02, 0x6b, 0x95, 0x50,
    0xfd, 0x02, 0xfd, 0xa4, 0x58, 0x29, 0x78, 0x4f, 0xd0, 0xae,
    0x66, 0xd6, 0x5c, 0xe7, 0x45, 0xfe, 0x98, 0xb0, 0xa3, 0xe2,
    0x87, 0xc0, 0xd2, 0x81, 0x08, 0xf1, 0xf1, 0xe7, 0xda, 0x62,
    0x9e, 0xa0, 0x34, 0x86, 0xeb, 0xa1, 0x6e, 0x4a, 0x26, 0x8e,
    0x39, 0x0c, 0x51, 0x10, 0x33, 0x11, 0x87, 0xf8, 0x79, 0x3c,
    0x49, 0x7a, 0x8b, 0xce, 0xc1, 0x0a, 0x0e, 0xe1, 0xd5, 0x2a,
    0xac, 0xf0, 0x3a, 0x1d, 0x6a, 0x6a, 0xe5, 0xe1, 0x81, 0x70,
    0xad, 0xaf, 0x15, 0x4c, 0x2a, 0x70, 0x2a, 0x6b, 0x22, 0x0d,
    0x30, 0xe7, 0x56, 0xed, 0x2d, 0x4b, 0x85, 0x17, 0x49, 0x72,
    0x3a, 0x1b, 0x6f, 0x57, 0x1c, 0xf7, 0x72, 0x9e, 0x20, 0xdb,
    0x57, 0x1c, 0xfb, 0x36, 0x50, 0x52, 0xec, 0x5b, 0xd6, 0x6a,
    0x1b, 0xf8, 0x74, 0xad, 0xe6, 0x00, 0x74, 0x04, 0xc5, 0x99,
    0x83, 0xe4, 0x5a, 0x0c, 0xc3, 0xe8, 0x6d, 0x3a, 0xd7, 0x3c,
    0x3c, 0xc0, 0x1a, 0x28, 0xb3, 0x29, 0x7a, 0x10, 0x9e, 0x39,
    0x66, 0x5b, 0xc1, 0x38, 0xac, 0x21, 0x4e, 0xcd, 0x01, 0xf2,
    0xf6, 0x30, 0x2c, 0x2b, 0xb6, 0xbf, 0xf5, 0xea, 0x61, 0xaf,
    0x0c, 0xa6, 0x01, 0x11, 0x15, 0x19, 0x09, 0x8c, 0x7e, 0x69,
    0xdf, 0x3b, 0xea, 0xd3, 0x0a, 0x3a, 0xd7, 0xbd, 0xe1, 0x17,
    0xaf, 0x92, 0x3c, 0xf5, 0xfe, 0x35, 0xd6, 0xcf, 0x07, 0xa6,
    0xf7, 0xe9, 0xc1, 0x99, 0xed, 0x80, 0xe3, 0x12, 0xd5, 0x4b,
    0xb9, 0xdf, 0xaf, 0x4e, 0x52, 0xad, 0x8e, 0x66, 0x87, 0xe5,
    0x2c, 0xd0, 0x45, 0x70, 0xd9, 0x78, 0x8f, 0x4b, 0xf4, 0xe1,
    0xf1, 0x22, 0xf2, 0xe3, 0xed, 0x1f, 0xeb, 0xe9, 0x70, 0x31,
    0x4c, 0x65, 0x5f, 0x55, 0xee, 0x5d, 0xaa, 0x83, 0x87, 0x76,
    0xbe, 0x11, 0xae, 0xd7, 0xf2, 0xfb, 0x43, 0xe7, 0x17, 0x81,
    0x33, 0x15, 0x47, 0xa0, 0xf3, 0x8e, 0x84, 0x57, 0xff, 0x35,
    0x9e, 0x4a, 0x8a, 0xab, 0x50, 0x3a, 0x45, 0xe0, 0xc3, 0x73,
    0xca, 0x77, 0x61, 0x68, 0x38, 0xd0, 0xa3, 0x5f, 0x03, 0x8d,
    0x41, 0xc2, 0xd3, 0x4a, 0x17, 0xe0, 0xa8, 0xaa, 0x00, 0xf3,
    0xf2, 0x5b, 0xa8, 0xe1, 0x06, 0xa6, 0x2b, 0xdb, 0xe1, 0x74,
    0xbd, 0xc4, 0xd2, 0x2b, 0x55, 0x9a, 0xb0, 0xf8, 0x35, 0xd8,
    0x6b, 0xec, 0xdb, 0xc5, 0xf4, 0x6c, 0x40, 0x90, 0x6a, 0x68,
    0xc9, 0xb5, 0xcb, 0xbb, 0xd0, 0xb0, 0xbc, 0x9f, 0xb9, 0xaa,
    0x50, 0x14, 0x93, 0x3b, 0x9f, 0x25, 0xcb, 0x40, 0xb8, 0x08,
    0xcc, 0x13, 0xe5, 0xdc, 0x3f, 0x84, 0x96, 0xe0, 0x73, 0x7b,
    0x7d, 0x9e, 0x41, 0x92, 0x5d, 0xcc, 0xa4, 0xea, 0x4f, 0x93,
    0x0c, 0x40, 0x2e, 0x42, 0x8a, 0xe9, 0xb9, 0x12, 0x74, 0xbb,
    0x79, 0x7c, 0xb0, 0x37, 0x20, 0xb6, 0xaf, 0x43, 0x3a, 0x88,
    0x59, 0x7c, 0x68, 0x28, 0x5f, 0x98, 0xc2, 0xf0, 0x2a, 0xbc,
    0xa1, 0x61, 0x88, 0x1f, 0x43, 0xbc, 0x42, 0x8f, 0x43, 0xf3,
    0x7e, 0x16, 0x96, 0xfa, 0x92, 0x70, 0xaf, 0x3c, 0x9f, 0x4b,
    0xd9, 0x60, 0xe9, 0xf6, 0x2e, 0x84, 0xda, 0x88, 0x31, 0x34,
    0xa6, 0x85, 0x10, 0x05, 0xef, 0x40, 0xa8, 0xa5, 0x4f, 0x92,
    0x59, 0xf7, 0xe0, 0xc4, 0x2b, 0x12, 0x17, 0x71, 0xbe, 0x8c,
    0x4a, 0x02, 0xfe, 0x12, 0xb6, 0x3b, 0x85, 0x75, 0x37, 0xf3,
    0x73, 0x2d, 0x9c, 0x00, 0x5d, 0x80, 0xad, 0x20, 0x2f, 0x5a,
    0x0b, 0x17, 0x7e, 0x67, 0x72, 0x24, 0x5a, 0xb9, 0xf3, 0xb1,
    0x33, 0xa4, 0x57, 0x1d, 0x49, 0x72, 0x2c, 0x7f, 0x47, 0x15,
    0x07, 0xe0, 0x45, 0x14, 0xdd, 0x77, 0x86, 0x6d, 0x03, 0xbe,
    0x57, 0xd0, 0xaa, 0x18, 0xa6, 0xdd, 0x94, 0x18, 0x3f, 0x8a,
    0xf3, 0xb5, 0xd7, 0x5a, 0xec, 0xc8, 0x79, 0x7f, 0x51, 0x61,
    0x3c, 0x9b, 0xb2, 0x9b, 0xf3, 0xb4, 0x35, 0xd1, 0x38, 0xbf,
    0x37, 0xce, 0x54, 0xd1, 0xf8, 0xb6, 0x45, 0xeb, 0x52, 0x0d,
    0x9a, 0x09, 0x58, 0x0d, 0x2c, 0x0b, 0xb1, 0xf2, 0x30, 0x3a,
    0x95, 0xc1, 0x13, 0x91, 0xd2, 0x9f, 0x8d, 0x8d, 0xd0, 0x38,
    0x3e, 0x4c, 0xae, 0x4a, 0x55, 0xa7, 0x42, 0x11, 0x83, 0xc4,
    0x70, 0xf0, 0x2b, 0x68, 0x9e, 0x07, 0xad, 0xb7, 0x83, 0xc6,
    0x53, 0x3c, 0xfb, 0x0a, 0x5d, 0x24, 0xdc, 0xe1, 0x55, 0x72,
    0xcf, 0xce, 0x3e, 0xc8, 0xd0, 0x57, 0x8a, 0x82, 0x5e, 0x78,
    0x2b, 0x80, 0xc5, 0xb9, 0x09, 0x46, 0xf8, 0x90, 0x39, 0x52,
    0xa9, 0xce, 0x3f, 0x3d, 0x41, 0x3b, 0x28, 0x45, 0xa3, 0xb3,
    0x21, 0xc2, 0xcd, 0x14, 0x49, 0x41, 0x6c, 0x38, 0xda, 0x1b,
    0x5f, 0x16, 0x49, 0xf9, 0x65, 0x00, 0x4e, 0xb4, 0x20, 0x55,
    0x70, 0xe8, 0x58, 0x1a, 0x18, 0xbf, 0x41, 0xef, 0x31, 0xb1,
    0xe7, 0x8d, 0x89, 0xc1, 0x48, 0xe8, 0xf5, 0x57, 0x35, 0xfa,
    0xc1, 0x79, 0xee, 0x2c, 0xe8, 0x7d, 0xb6, 0x03, 0xcc, 0x66,
    0x09, 0x6f, 0x52, 0x84, 0x0a, 0x34, 0x18, 0x2c, 0x01, 0x45,
    0x81, 0x00, 0xe5, 0x5e, 0x8d, 0xae, 0x1c, 0x96, 0x8b, 0x45,
    0x73, 0x00, 0x0a, 0xb5, 0xcf, 0x8d, 0x0e, 0x35, 0x5d, 0x1a,
    0x0e, 0xbf, 0x64, 0x9a, 0x52, 0x20, 0x48, 0xc6, 0xb9, 0x40,
    0xd3, 0x2c, 0x52, 0xca, 0x93, 0xcf, 0xbb, 0x94, 0x06, 0xf3,
    0x97, 0xee, 0xcc, 0x5d, 0xa3, 0xea, 0xf8, 0x5a, 0x39, 0x77,
    0x34, 0xd7, 0xf6, 0x4e, 0xbe, 0x8a, 0x07, 0x5f, 0x51, 0x53,
    0xc5, 0x1b, 0x8c, 0x47, 0x8f, 0x34, 0x0e, 0x60, 0x0a, 0x90,
    0xe2, 0xda, 0x7b, 0xef, 0xd6, 0xf5, 0x5d, 0xe5, 0x32, 0x37,
    0x75, 0x99, 0x81, 0x4a, 0x2a, 0x78, 0x71, 0xdc, 0xf4, 0xe5,
    0xca, 0xd8, 0x6b, 0x3b, 0x90, 0x68, 0x2e, 0x93, 0xc5, 0x10,
    0x42, 0x5d, 0x38, 0x90, 0x32, 0x46, 0xea, 0x87, 0xe0, 0xbc,
    0xb8, 0x9a, 0x18, 0x20, 0x68, 0x85, 0x6d, 0x9b, 0xc9, 0x8f,
    0x9b, 0xd2, 0xbe, 0x15, 0x12, 0x68, 0xd0, 0xb0, 0x16, 0x5f,
    0xe2, 0x69, 0x1d, 0x04, 0x00, 0xfc, 0x63, 0x33, 0xcd, 0x1f,
    0x89, 0xcd, 0x52, 0xff, 0xec, 0x19, 0x69, 0x74, 0xa3, 0xce,
    0x4d, 0xab, 0x93, 0xe4, 0xc6, 0x13, 0x56, 0x27, 0xc9, 0x25,
    0x5a, 0x01, 0xb2, 0x36, 0x8b, 0x61, 0xe5, 0x8b, 0x98, 0xac,
    0xe4, 0x2a, 0xb6, 0x40, 0x9f, 0x42, 0xe4, 0x1b, 0x52, 0xf7,
    0xfd, 0xd8, 0x30, 0x07, 0x33, 0xf9, 0x47, 0xcb, 0x3c, 0xad,
    0x12, 0xc1, 0xcc, 0x29, 0x62, 0x49, 0x04, 0x0c, 0x23, 0x97,
    0x5a, 0xa4, 0x84, 0x67, 0xde, 0x5a, 0xe5, 0x36, 0xd2, 0x88,
    0xf1, 0xd4, 0xeb, 0x13, 0x81, 0x54, 0x51, 0x11, 0xe3, 0xba,
    0xbc, 0xee, 0xdd, 0x6c, 0xcd, 0xe6, 0xb4, 0xa1, 0x8b, 0x0b,
    0x66, 0xfb, 0x8e, 0x50, 0xa0, 0xda, 0x69, 0x8d, 0xcc, 0x2d,
    0xe4, 0x2c, 0xc4, 0x37, 0xdf, 0x61, 0xc0, 0x03, 0xbd, 0x8b,
    0x28, 0xca, 0xd2, 0x8c, 0x1c, 0xf1, 0xa4, 0x26, 0x69, 0xe5,
    0xcf, 0x45, 0xdb, 0x5a, 0x47, 0x79, 0xed, 0x9f, 0xf7, 0xd2,
    0xdb, 0xba, 0x46, 0x53, 0x4f, 0xce, 0xa8, 0xbe, 0x8f, 0x4a,
    0xd6, 0xdf, 0x2e, 0x06, 0xe6, 0x4c, 0x9a, 0xc1, 0xb6, 0x49,
    0xed, 0xc4, 0xeb, 0xaa, 0xa4, 0x29, 0x6d, 0xd4, 0xcc, 0x8c,
    0xb6, 0x40, 0x11, 0x39, 0x69, 0xf7, 0x75, 0xcd, 0xb1, 0x99,
    0x46, 0x4e, 0xde, 0xcb, 0xf6, 0x9d, 0x32, 0xf3, 0xc9, 0x47,
    0x47, 0x7a, 0xcb, 0xfb, 0xa3, 0x0c, 0x3b, 0xdf, 0xb7, 0xde,
    0xec, 0x99, 0xde, 0xb0, 0x26, 0x04, 0x34, 0xae, 0x6b, 0xfc,
    0x99, 0xbc, 0xde, 0xd5, 0xbe, 0xe7, 0xeb, 0xf9, 0xe7, 0xa6,
    0x01, 0x9a, 0x0c, 0x5e, 0x66, 0xe6, 0x53, 0xe4, 0xd1, 0x58,
    0xac, 0xda, 0x69, 0x77, 0x7b, 0x68, 0xd6, 0x30, 0x2a, 0x9c,
    0x6b, 0xbe, 0x9f, 0x3d, 0x71, 0xd6, 0x54, 0xcd, 0x59, 0x4e,
    0x1f, 0xe3, 0x83, 0x4e, 0xd1, 0x8e, 0xaf, 0x97, 0xa8, 0xe5,
    0xb6, 0x59, 0x77, 0xa8, 0x02, 0x20, 0xe4, 0xeb, 0x44, 0x71,
    0xbc, 0x07, 0x14, 0x79, 0x4f, 0x0c, 0x27, 0x06, 0x39, 0xcf,
    0x7c, 0xef, 0x2b, 0x9b, 0x5e, 0xc4, 0x6d, 0x79, 0x13, 0x00,
    0x43, 0x6f, 0x51, 0x77, 0xb5, 0xc3, 0x72, 0xad, 0x13, 0xa9,
    0xe5, 0x9a, 0x5b, 0x1a, 0x99, 0x74, 0xc0, 0x7a, 0xf9, 0xc5,
    0xb0, 0x58, 0x35, 0x1c, 0xa5, 0x51, 0xdb, 0xa1, 0x14, 0xcd,
    0x26, 0x71, 0xb1, 0xe7, 0xaa, 0x14, 0xa7, 0x46, 0x93, 0xd3,
    0x5c, 0x8c, 0x1a, 0x91, 0x77, 0x46, 0x2e, 0x15, 0xaa, 0x9e,
    0xf7, 0x2b, 0x79, 0x41, 0x76, 0xf7, 0x22, 0x53, 0x7d, 0x51,
    0xdb, 0x98, 0x3d, 0x5b, 0x78, 0x5f, 0xc3, 0xc9, 0x29, 0xa3,
    0xff, 0x75, 0x82, 0x06, 0x9a, 0x16, 0x5e, 0xa4, 0x79, 0x0d,
    0xd1, 0x6d, 0x08, 0xff, 0x43, 0xef, 0x9c, 0xf3, 0x1b, 0x7a,
    0x3f, 0x34, 0xbe, 0x19, 0x15, 0x06, 0x33, 0xdb, 0xa5, 0x71,
    0xcb, 0x5f, 0x6b, 0x8d, 0xbd, 0x5b, 0x32, 0x91, 0xb2, 0x37,
    0x3d, 0xb4, 0x40, 0x9e, 0x02, 0x9b, 0xb7, 0x68, 0x20, 0x58,
    0x5c, 0xab, 0xcb, 0xc8, 0x23, 0x2d, 0x77, 0xcc, 0x0b, 0xf6,
    0x78, 0x6b, 0x80, 0x06, 0x91, 0xa9, 0xfd, 0x7e, 0xfa, 0x25,
    0x98, 0x9f, 0xcc, 0x79, 0x0a, 0x1a, 0x54, 0x83, 0xac, 0x64,
    0x16, 0x90, 0xe5, 0xd9, 0xa7, 0xd7, 0x1b, 0x86, 0x0d, 0xe6,
    0xe6, 0x22, 0x2b, 0x1f, 0x44, 0x49, 0x98, 0x9c, 0x51, 0x6f,
    0xcf, 0x58, 0x4a, 0xfa, 0xfa, 0x84, 0x12, 0xa5, 0x10, 0xf4,
    0xca, 0xf0, 0x98, 0x2b, 0xc9, 0x03, 0x71, 0x37, 0xe7, 0xdc,
    0xc2, 0xb1, 0x4e, 0x64, 0xde, 0x4f, 0x46, 0x0d, 0x6b, 0x25,
    0x88, 0x5d, 0xd6, 0xff, 0x23, 0x46, 0x57, 0x36, 0x14, 0x18,
    0xa7, 0xcb, 0xb8, 0xbd, 0xf0, 0xc5, 0x37, 0x36, 0xee, 0xe1,
    0xed, 0x9f, 0x4d, 0xd4, 0x39, 0xe5, 0x92, 0xcf, 0x95, 0x4d,
    0x66, 0x36, 0x5d, 0xd0, 0xcc, 0x07, 0xcf, 0x15, 0x5a, 0xce,
    0x14, 0xb8, 0xda, 0x0d, 0x3d, 0x1b, 0x45, 0xc5, 0x2e, 0x34,
    0x43, 0x25, 0x02, 0x3a, 0xcd, 0x14, 0x45, 0xfb, 0x3e, 0xf9,
    0x88, 0x5d, 0x0d, 0x29, 0x31, 0xb9, 0xa1, 0xe6, 0x31, 0x18,
    0x52, 0x46, 0x3f, 0x22, 0x4f, 0x9f, 0x7a, 0x65, 0x36, 0x88,
    0xa3, 0x1c, 0x3e, 0x6f, 0x50, 0x7a, 0x36, 0xbe, 0x56, 0x7e,
    0x50, 0xcb, 0x7a, 0x10, 0xa0, 0xec, 0xf6, 0x82, 0xd6, 0x30,
    0x1c, 0xe8, 0x4c, 0x50, 0xf9, 0x3e, 0xdb, 0xac, 0xbe, 0x4f,
    0x90, 0xb1, 0xd5, 0x1b, 0x12, 0x95, 0xfb, 0xe8, 0x08, 0x64,
    0x56, 0x7c, 0x96, 0xcc, 0x90, 0xb1, 0xbc, 0xa0, 0xf5, 0x32,
    0x69, 0xb3, 0x5f, 0x27, 0x0f, 0xbe, 0xc9, 0xbd, 0xeb, 0xfa,
    0x4b, 0x5c, 0xc5, 0x99, 0x9e, 0x5a, 0x04, 0xcc, 0xd0, 0x4d,
    0x29, 0xe8, 0x84, 0x55, 0x8c, 0xd7, 0xc4, 0x06, 0x13, 0x4d,
    0x92, 0xe5, 0x98, 0x9c, 0x4c, 0xc1, 0xf7, 0xaf, 0x7b, 0xd5,
    0x2b, 0x92, 0x68, 0x68, 0x19, 0x70, 0x4c, 0x9e, 0x46, 0xb8,
    0x34, 0xeb, 0x01, 0x47, 0xbe, 0x59, 0xab, 0x0b, 0x22, 0x25,
    0xe7, 0x56, 0xa8, 0xb4, 0x93, 0x3c, 0xd5, 0x98, 0x9f, 0x61,
    0x2e, 0xfa, 0xcb, 0x5f, 0x5b, 0xd8, 0x09, 0x83, 0xe9, 0x40,
    0xe9, 0x0e, 0x42, 0xdd, 0x17, 0xd7, 0x6e, 0x19, 0x8d, 0x95,
    0x0a, 0x93,
};
static const int sizeof_bench_dilithium_level5_pubkey =
    sizeof(bench_dilithium_level5_pubkey);

#endif /* !WOLFSSL_DILITHIUM_NO_VERIFY */

#endif /* HAVE_DILITHIUM */

";

# convert and print sphincs keys
print OUT_FILE "#if defined(HAVE_SPHINCS)\n\n";
for (my $i = 0; $i < $num_sphincs; $i++) {

    my $fname = $fileList_sphincs[$i][0];
    my $sname = $fileList_sphincs[$i][1];

    print OUT_FILE "/* $fname */\n";
    print OUT_FILE "static const unsigned char $sname\[] =\n";
    print OUT_FILE "{\n";
    file_to_hex($fname);
    print OUT_FILE "};\n";
    print OUT_FILE "static const int sizeof_$sname = sizeof($sname);\n\n";
}

print OUT_FILE "#endif /* HAVE_SPHINCS */\n\n";

# convert and print 256-bit cert/keys
print OUT_FILE "#if defined(HAVE_ECC) && defined(USE_CERT_BUFFERS_256)\n\n";
for (my $i = 0; $i < $num_ecc; $i++) {

    my $fname = $fileList_ecc[$i][0];
    my $sname = $fileList_ecc[$i][1];

    print OUT_FILE "/* $fname, ECC */\n";
    print OUT_FILE "static const unsigned char $sname\[] =\n";
    print OUT_FILE "{\n";
    file_to_hex($fname);
    print OUT_FILE "};\n";
    print OUT_FILE "static const int sizeof_$sname = sizeof($sname);\n\n";
}
print OUT_FILE "#endif /* HAVE_ECC && USE_CERT_BUFFERS_256 */\n\n";


print OUT_FILE "/* dh1024 p */
static const unsigned char dh_p[] =
{
    0xE6, 0x96, 0x9D, 0x3D, 0x49, 0x5B, 0xE3, 0x2C, 0x7C, 0xF1, 0x80, 0xC3,
    0xBD, 0xD4, 0x79, 0x8E, 0x91, 0xB7, 0x81, 0x82, 0x51, 0xBB, 0x05, 0x5E,
    0x2A, 0x20, 0x64, 0x90, 0x4A, 0x79, 0xA7, 0x70, 0xFA, 0x15, 0xA2, 0x59,
    0xCB, 0xD5, 0x23, 0xA6, 0xA6, 0xEF, 0x09, 0xC4, 0x30, 0x48, 0xD5, 0xA2,
    0x2F, 0x97, 0x1F, 0x3C, 0x20, 0x12, 0x9B, 0x48, 0x00, 0x0E, 0x6E, 0xDD,
    0x06, 0x1C, 0xBC, 0x05, 0x3E, 0x37, 0x1D, 0x79, 0x4E, 0x53, 0x27, 0xDF,
    0x61, 0x1E, 0xBB, 0xBE, 0x1B, 0xAC, 0x9B, 0x5C, 0x60, 0x44, 0xCF, 0x02,
    0x3D, 0x76, 0xE0, 0x5E, 0xEA, 0x9B, 0xAD, 0x99, 0x1B, 0x13, 0xA6, 0x3C,
    0x97, 0x4E, 0x9E, 0xF1, 0x83, 0x9E, 0xB5, 0xDB, 0x12, 0x51, 0x36, 0xF7,
    0x26, 0x2E, 0x56, 0xA8, 0x87, 0x15, 0x38, 0xDF, 0xD8, 0x23, 0xC6, 0x50,
    0x50, 0x85, 0xE2, 0x1F, 0x0D, 0xD5, 0xC8, 0x6B,
};

/* dh1024 g */
static const unsigned char dh_g[] =
{
  0x02,
};\n\n";

print OUT_FILE "#if defined(NO_ASN) && defined(WOLFSSL_SP_MATH)\n";
print OUT_FILE "/* dh2048 p */
static const unsigned char dh2048_p[] =
{
    0xb0, 0xa1, 0x08, 0x06, 0x9c, 0x08, 0x13, 0xba, 0x59, 0x06, 0x3c, 0xbc,
    0x30, 0xd5, 0xf5, 0x00, 0xc1, 0x4f, 0x44, 0xa7, 0xd6, 0xef, 0x4a, 0xc6,
    0x25, 0x27, 0x1c, 0xe8, 0xd2, 0x96, 0x53, 0x0a, 0x5c, 0x91, 0xdd, 0xa2,
    0xc2, 0x94, 0x84, 0xbf, 0x7d, 0xb2, 0x44, 0x9f, 0x9b, 0xd2, 0xc1, 0x8a,
    0xc5, 0xbe, 0x72, 0x5c, 0xa7, 0xe7, 0x91, 0xe6, 0xd4, 0x9f, 0x73, 0x07,
    0x85, 0x5b, 0x66, 0x48, 0xc7, 0x70, 0xfa, 0xb4, 0xee, 0x02, 0xc9, 0x3d,
    0x9a, 0x4a, 0xda, 0x3d, 0xc1, 0x46, 0x3e, 0x19, 0x69, 0xd1, 0x17, 0x46,
    0x07, 0xa3, 0x4d, 0x9f, 0x2b, 0x96, 0x17, 0x39, 0x6d, 0x30, 0x8d, 0x2a,
    0xf3, 0x94, 0xd3, 0x75, 0xcf, 0xa0, 0x75, 0xe6, 0xf2, 0x92, 0x1f, 0x1a,
    0x70, 0x05, 0xaa, 0x04, 0x83, 0x57, 0x30, 0xfb, 0xda, 0x76, 0x93, 0x38,
    0x50, 0xe8, 0x27, 0xfd, 0x63, 0xee, 0x3c, 0xe5, 0xb7, 0xc8, 0x09, 0xae,
    0x6f, 0x50, 0x35, 0x8e, 0x84, 0xce, 0x4a, 0x00, 0xe9, 0x12, 0x7e, 0x5a,
    0x31, 0xd7, 0x33, 0xfc, 0x21, 0x13, 0x76, 0xcc, 0x16, 0x30, 0xdb, 0x0c,
    0xfc, 0xc5, 0x62, 0xa7, 0x35, 0xb8, 0xef, 0xb7, 0xb0, 0xac, 0xc0, 0x36,
    0xf6, 0xd9, 0xc9, 0x46, 0x48, 0xf9, 0x40, 0x90, 0x00, 0x2b, 0x1b, 0xaa,
    0x6c, 0xe3, 0x1a, 0xc3, 0x0b, 0x03, 0x9e, 0x1b, 0xc2, 0x46, 0xe4, 0x48,
    0x4e, 0x22, 0x73, 0x6f, 0xc3, 0x5f, 0xd4, 0x9a, 0xd6, 0x30, 0x07, 0x48,
    0xd6, 0x8c, 0x90, 0xab, 0xd4, 0xf6, 0xf1, 0xe3, 0x48, 0xd3, 0x58, 0x4b,
    0xa6, 0xb9, 0xcd, 0x29, 0xbf, 0x68, 0x1f, 0x08, 0x4b, 0x63, 0x86, 0x2f,
    0x5c, 0x6b, 0xd6, 0xb6, 0x06, 0x65, 0xf7, 0xa6, 0xdc, 0x00, 0x67, 0x6b,
    0xbb, 0xc3, 0xa9, 0x41, 0x83, 0xfb, 0xc7, 0xfa, 0xc8, 0xe2, 0x1e, 0x7e,
    0xaf, 0x00, 0x3f, 0x93
};

/* dh2048 g */
static const unsigned char dh2048_g[] =
{
  0x02,
};\n";
print OUT_FILE "#endif\n\n";


# convert and print ed25519 cert/keys
print OUT_FILE "#if defined(HAVE_ED25519)\n\n";
for (my $i = 0; $i < $num_ed; $i++) {

    my $fname = $fileList_ed[$i][0];
    my $sname = $fileList_ed[$i][1];

    print OUT_FILE "/* $fname, ED25519 */\n";
    print OUT_FILE "static const unsigned char $sname\[] =\n";
    print OUT_FILE "{\n";
    file_to_hex($fname);
    print OUT_FILE "};\n";
    print OUT_FILE "static const int sizeof_$sname = sizeof($sname);\n\n";
}
print OUT_FILE "#endif /* HAVE_ED25519 */\n\n";


# convert and print CURVE25519 cert/keys
print OUT_FILE "#if defined(USE_CERT_BUFFERS_25519)\n\n";
for (my $i = 0; $i < $num_x; $i++) {

    my $fname = $fileList_x[$i][0];
    my $sname = $fileList_x[$i][1];

    print OUT_FILE "/* $fname, CURVE25519 */\n";
    print OUT_FILE "static const unsigned char $sname\[] =\n";
    print OUT_FILE "{\n";
    file_to_hex($fname);
    print OUT_FILE "};\n";
    print OUT_FILE "static const int sizeof_$sname = sizeof($sname);\n\n";
}
print OUT_FILE "#endif /* USE_CERT_BUFFERS_25519 */\n\n";


print OUT_FILE "#endif /* WOLFSSL_CERTS_TEST_H */\n\n";

# close certs_test.h file
close OUT_FILE or die $!;

# print file as hex, comma-separated, as needed by C buffer
sub file_to_hex {
    my $fileName = $_[0];

    open my $fp, "<", $fileName or die $!;
    binmode($fp);

    my $fileLen = -s $fileName;
    my $byte;

    for (my $i = 0, my $j = 1; $i < $fileLen; $i++, $j++)
    {
        if ($j == 1) {
            print OUT_FILE "        ";
        }
        if ($j != 1) {
            print OUT_FILE " ";
        }
        read($fp, $byte, 1) or die "Error reading $fileName";
        my $output = sprintf("0x%02X", ord($byte));
        print OUT_FILE $output;

        if ($i != ($fileLen - 1)) {
            print OUT_FILE ",";
        }

        if ($j == 10) {
            $j = 0;
            print OUT_FILE "\n";
        }
    }

    print OUT_FILE "\n";

    close($fp);
}
