// Copyright (c) Facebook, Inc. and its affiliates. All Rights Reserved.
#ifndef DBWRAPPER_H
#define DBWRAPPER_H

#include <map>
#include <node.h>

#include "rocksdb/db.h"
#include "rocksdb/slice.h"
#include "rocksdb/options.h"

// Used to encapsulate a particular instance of an opened database.
//
// This object should not be used directly in C++; it exists solely to provide
// a mapping from a JavaScript object to a C++ code that can use the RocksDB
// API.
class DBWrapper : public node::ObjectWrap {
  public:
    static void Init(Handle<Object> exports);

  private:
    explicit DBWrapper();
    ~DBWrapper();

    // Helper methods
    static bool HasFamilyNamed(std::string& name, DBWrapper* db);
    static bool AddToBatch(ROCKSDB_NAMESPACE::WriteBatch& batch, bool del,
                           Handle<Array> array);
    static bool AddToBatch(ROCKSDB_NAMESPACE::WriteBatch& batch, bool del,
                           Handle<Array> array, DBWrapper* db_wrapper,
                           std::string cf);
    static Handle<Value> CompactRangeDefault(const v8::Arguments& args);
    static Handle<Value> CompactColumnFamily(const Arguments& args);
    static Handle<Value> CompactOptions(const Arguments& args);
    static Handle<Value> CompactAll(const Arguments& args);

    // C++ mappings of API methods
    static Persistent<v8::Function> constructor;
    static Handle<Value> Open(const Arguments& args);
    static Handle<Value> New(const Arguments& args);
    static Handle<Value> Get(const Arguments& args);
    static Handle<Value> Put(const Arguments& args);
    static Handle<Value> Delete(const Arguments& args);
    static Handle<Value> Dump(const Arguments& args);
    static Handle<Value> WriteBatch(const Arguments& args);
    static Handle<Value> CreateColumnFamily(const Arguments& args);
    static Handle<Value> CompactRange(const Arguments& args);
    static Handle<Value> Close(const Arguments& args);

    // Internal fields
    ROCKSDB_NAMESPACE::Options options_;
    ROCKSDB_NAMESPACE::Status status_;
    ROCKSDB_NAMESPACE::DB* db_;
    std::unordered_map<std::string, ROCKSDB_NAMESPACE::ColumnFamilyHandle*>
        columnFamilies_;
};

#endif
